package de.uka.ipd.sdq.edp2.example;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Random;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.measure.Measure;
import javax.measure.quantity.Dimensionless;
import javax.measure.unit.BaseUnit;
import javax.measure.unit.SI;
import javax.measure.unit.Unit;

import org.eclipse.emf.common.util.EList;

import de.uka.ipd.sdq.edp2.OrdinalMeasurementsDao;
import de.uka.ipd.sdq.edp2.impl.Measurement;
import de.uka.ipd.sdq.edp2.impl.MeasurementsUtility;
import de.uka.ipd.sdq.edp2.models.ExperimentData.BaseMetricDescription;
import de.uka.ipd.sdq.edp2.models.ExperimentData.CaptureType;
import de.uka.ipd.sdq.edp2.models.ExperimentData.DataSeries;
import de.uka.ipd.sdq.edp2.models.ExperimentData.Description;
import de.uka.ipd.sdq.edp2.models.ExperimentData.ExperimentDataFactory;
import de.uka.ipd.sdq.edp2.models.ExperimentData.ExperimentDataPackage;
import de.uka.ipd.sdq.edp2.models.ExperimentData.ExperimentGroup;
import de.uka.ipd.sdq.edp2.models.ExperimentData.ExperimentRun;
import de.uka.ipd.sdq.edp2.models.ExperimentData.ExperimentSetting;
import de.uka.ipd.sdq.edp2.models.ExperimentData.Measurements;
import de.uka.ipd.sdq.edp2.models.ExperimentData.MeasurementsRange;
import de.uka.ipd.sdq.edp2.models.ExperimentData.MetricSetDescription;
import de.uka.ipd.sdq.edp2.models.ExperimentData.Monotonic;
import de.uka.ipd.sdq.edp2.models.ExperimentData.OrdinalMeasure;
import de.uka.ipd.sdq.edp2.models.ExperimentData.PersistenceKindOptions;
import de.uka.ipd.sdq.edp2.models.ExperimentData.RawMeasurements;
import de.uka.ipd.sdq.edp2.models.ExperimentData.Scale;
import de.uka.ipd.sdq.edp2.models.Repository.Repository;
/**This class demonstrates the use of EDP2 for storing measurements and creating descriptions.
 * The implemented example follows the EDP2 creation steps: 
 * <li>
 * <ul/>Step 1: Build ExperimentSetting
 * <ul/>Step 2: Prepare Experiment Run (Add Raw Measurements, AggregationFunctions)
 * <ul/>Step 3: Run the experiment and generate measurements
 * </li>
 * All Measurements are stored in a Measurements directory.
 * Additionally, all standard descriptions can be retrieved.
 * @author groenda
 */
public class ExampleData {
	/** Logger for this class. */
	private static Logger logger = Logger.getLogger(StoreExample.class.getCanonicalName());
	/** EMF initialization. */
	static ExperimentDataPackage emfp = ExperimentDataPackage.eINSTANCE;
	/** Shortcut to factory. */
	ExperimentDataFactory f = ExperimentDataFactory.eINSTANCE;
	
	// Create default units
	@SuppressWarnings("unchecked")
	Unit timeUnit = SI.SECOND;
	@SuppressWarnings("unchecked")
	Unit numberUnit = new BaseUnit<Dimensionless>("Threads");

	// Metric descriptions
	private BaseMetricDescription SimTime;
	private BaseMetricDescription ResponseTime;
	private BaseMetricDescription DemandedTime;
	private BaseMetricDescription WaitingTime;
	private BaseMetricDescription AssignedThreads;
	private BaseMetricDescription AbsoluteFrequency;
	private MetricSetDescription ResponseTimeSimulated;
	private MetricSetDescription DemandedTimeSimulated;
	private MetricSetDescription WaitingTimeSimulated;
	private MetricSetDescription AssignedThreadsSimulated;
	
	/**Creates all descriptions available as default in EDP2.
	 * @return A collection containing all descriptions.
	 */
	public Collection<Description> createDescriptions() {
		createBaseMetricDescriptions();
		createMetricSetDescriptions();

		Vector<Description> list = new Vector<Description>();
		list.add(SimTime);
		list.add(ResponseTime);
		list.add(DemandedTime);
		list.add(WaitingTime);
		list.add(AssignedThreads);
		list.add(AbsoluteFrequency);
		list.add(ResponseTimeSimulated);
		list.add(DemandedTimeSimulated);
		list.add(WaitingTimeSimulated);
		list.add(AssignedThreadsSimulated);
		return list;
	}

	/**Creates all metric set descriptions.
	 */
	private void createMetricSetDescriptions() {
		MetricSetDescription msd = ExperimentDataFactory.eINSTANCE.createMetricSetDescription(); 
		msd.setUuid("_MenvIiUREd6gmLudJva2Dw");
		msd.setName("Response Time (Simulated)");
		msd.setTextualDescription("Response time determined in a simulation. Consists of the simulation time when the response time began and the response time itself.");
		msd.getSubsumedMetrics().add(SimTime);
		msd.getSubsumedMetrics().add(ResponseTime);
		ResponseTimeSimulated = msd;
		msd = ExperimentDataFactory.eINSTANCE.createMetricSetDescription(); 
		msd.setUuid("_tUkRciUREd6gmLudJva2Dw");
		msd.setName("Demanded Time (Simulated)");
		msd.setTextualDescription("Demanded time determined in a simulation. Consists of the simulation time when the time was demanded and the demanded time itself.");
		msd.getSubsumedMetrics().add(SimTime);
		msd.getSubsumedMetrics().add(DemandedTime);
		DemandedTimeSimulated = msd;
		msd = ExperimentDataFactory.eINSTANCE.createMetricSetDescription(); 
		msd.setUuid("_3cXn4iUREd6gmLudJva2Dw");
		msd.setName("Waiting Time (Simulated)");
		msd.setTextualDescription("Waiting time determined in a simulation. Consists of the simulation time when the waiting time is stored and the waiting time itself.");
		msd.getSubsumedMetrics().add(SimTime);
		msd.getSubsumedMetrics().add(WaitingTime);
		WaitingTimeSimulated = msd;
		msd = ExperimentDataFactory.eINSTANCE.createMetricSetDescription(); 
		msd.setUuid("_M3ynoiUSEd6gmLudJva2Dw");
		msd.setName("Assigned Threads (Simulated)");
		msd.setTextualDescription("Number of threads assigned to a processor within a simulation. Consists of the time when the number has changed and the new number of assigned threads itself.");
		msd.getSubsumedMetrics().add(SimTime);
		msd.getSubsumedMetrics().add(AssignedThreads);
		AssignedThreadsSimulated = msd;
		logger.info("Metric descriptions created.");
	}

	/**Creates all base metric descriptions.
	 */
	private void createBaseMetricDescriptions() {
		BaseMetricDescription bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_38mSASUPEd6gmLudJva2Dw");
		bmd.setName("Simulation Time");
		bmd.setTextualDescription("Time passed within a simulation. Starting with 0.0.");
		bmd.setCaptureType(CaptureType.REAL_NUMBER);
		bmd.setScale(Scale.INTERVAL);
		bmd.setDefaultUnit(timeUnit);
		bmd.setMonotonic(Monotonic.YES);
		SimTime = bmd;
		bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_QC3ucCUQEd6gmLudJva2Dw");
		bmd.setName("Response Time");
		bmd.setTextualDescription("Response Time, for example of a service call.");
		bmd.setCaptureType(CaptureType.REAL_NUMBER);
		bmd.setScale(Scale.INTERVAL);
		bmd.setDefaultUnit(timeUnit);
		bmd.setMonotonic(Monotonic.NO);
		ResponseTime = bmd;
		bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_AiroIZMbEd6Vw8NDgVSYcg");
		bmd.setName("Frequency");
		bmd.setTextualDescription("Absolute frequency of measurements or events. For example, of measurements lying within an interval of a histogram.");
		bmd.setCaptureType(CaptureType.NATURAL_NUMBER);
		bmd.setScale(Scale.RATIO);
		bmd.setDefaultUnit(numberUnit);
		bmd.setMonotonic(Monotonic.NO);
		AbsoluteFrequency = bmd;
		bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_0xrYsCUQEd6gmLudJva2Dw");
		bmd.setName("Assigned Threads");
		bmd.setTextualDescription("Number of threads assigned to a processor. This includes active as well as waiting threads.");
		bmd.setCaptureType(CaptureType.NATURAL_NUMBER);
		bmd.setScale(Scale.INTERVAL);
		bmd.setDefaultUnit(numberUnit);
		bmd.setMonotonic(Monotonic.NO);
		AssignedThreads = bmd;
		bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_nU2AICUQEd6gmLudJva2Dw");
		bmd.setName("Waiting Time");
		bmd.setTextualDescription("Overall time spend waiting during a single resource demand.");
		bmd.setCaptureType(CaptureType.REAL_NUMBER);
		bmd.setScale(Scale.INTERVAL);
		bmd.setDefaultUnit(timeUnit);
		bmd.setMonotonic(Monotonic.NO);
		WaitingTime = bmd;
		bmd = ExperimentDataFactory.eINSTANCE.createBaseMetricDescription();
		bmd.setUuid("_fvNrgCUQEd6gmLudJva2Dw");
		bmd.setName("Demanded Time");
		bmd.setTextualDescription("Time demanded at a resource for processing.");
		bmd.setCaptureType(CaptureType.REAL_NUMBER);
		bmd.setScale(Scale.INTERVAL);
		bmd.setDefaultUnit(timeUnit);
		bmd.setMonotonic(Monotonic.NO);
		DemandedTime = bmd;
	}

	// Store for the experiment setup
	private ExperimentGroup groupA;
	private OrdinalMeasure serviceCallA;
	private OrdinalMeasure cpuA;
	private ExperimentSetting settingA;
	private ExperimentRun runA0;
	private Measurements measurementServiceCallA;
	
	/**Creates all necessary data for an (exemplary) experimental setting.
	 */
	public ExperimentGroup createExperimentalGroupAndSetting() {
		// create experiment group
		groupA = f.createExperimentGroup();
		groupA.setPurpose("Exemplary use of EDP2");
		// create metrics
		serviceCallA = f.createOrdinalMeasure();
		serviceCallA.setPersistencyKind(PersistenceKindOptions.BINARY_PREFERRED);
		serviceCallA.setMeasuredObject("Service Call A");
		serviceCallA.setMetric(ResponseTimeSimulated);
		groupA.getMeasure().add(serviceCallA);
		cpuA = f.createOrdinalMeasure();
		cpuA.setPersistencyKind(PersistenceKindOptions.BINARY_PREFERRED);
		cpuA.setMeasuredObject("CPU A");
		cpuA.setMetric(AssignedThreadsSimulated);
		groupA.getMeasure().add(cpuA);
		// create experiment settings
		settingA = f.createExperimentSetting();
		settingA.setDescription("Experiment Setting #1");
		settingA.getMeasure().add(serviceCallA);
		settingA.getMeasure().add(cpuA);
		groupA.getExperimentSettings().add(settingA);
		logger.info("Experimental setting created.");
		return groupA;
	}
	
	/**Prepares everything for a run of experiment setting A.
	 */
	private void prepareExperimentRun() {
		measurementServiceCallA = f.createMeasurements();
		measurementServiceCallA.setMeasure(serviceCallA);
		runA0 = f.createExperimentRun();
		runA0.getMeasurements().add(measurementServiceCallA);
		settingA.getExperimentRuns().add(runA0);

		MeasurementsRange range = MeasurementsUtility.addMeasurementRange(measurementServiceCallA);
		RawMeasurements rm = f.createRawMeasurements();
		range.setRawMeasurements(rm);
		MeasurementsUtility.addDataSeries(rm);
		logger.info("Experiment run prepared.");
	}
	
	/**Simulates the run of an experiment and generates (dummy) measurements.
	 */
	@SuppressWarnings("unchecked")
	private void generateMeasurements() {
		runA0.setStartTime(new Date());
		Random random = new Random();
		Measurement measurement;
		// create 1000 dummy measurements
		for (int i = 0; i < 1000; i++) {
			measurement = new Measurement(measurementServiceCallA.getMeasure().getMetric());
			measurement.setMeasuredValue(0, Measure.valueOf(i, timeUnit));
			measurement.setMeasuredValue(1, Measure.valueOf(random.nextDouble() * 10.0, timeUnit));
			MeasurementsUtility.storeMeasurement(measurementServiceCallA, measurement);
		}
		logger.info("Measurements generated.");
	}
		
	/**Simulates running an experiment.
	 */
	public void simulateExperimentRun() {
		prepareExperimentRun();
		generateMeasurements();
		logger.info("Example data created.");
	}
	
	/**Reads and prints the data stored for the experiment group in the repository.
	 * @param repo The repository.
	 * @param positionOfExperimentGroup Position in the list of experiment groups in the repository.
	 * @return Formatted data.
	 */
	@SuppressWarnings("unchecked")
	public String printStoredMeasurements(Repository repo, int positionOfExperimentGroup) {
		try {
			String result = "";
			// get DAOs
			EList<DataSeries> series = repo.getExperimentGroups().get(positionOfExperimentGroup).getExperimentSettings().get(0).getExperimentRuns().get(0).getMeasurements().get(0).getMeasurementsRange().get(0).getRawMeasurements().getDataSeries();
			OrdinalMeasurementsDao<Measure> omdSeries1 = MeasurementsUtility.getOrdinalMeasurementsDao(series.get(0));
			OrdinalMeasurementsDao<Measure> omdSeries2 = MeasurementsUtility.getOrdinalMeasurementsDao(series.get(1));
			// print stored data
			result += "Stored example data\n";
			result += "-------------------\n\n";
			// label data series according to metric definitions
			MetricSetDescription md = (MetricSetDescription) series.get(0).getRawMeasurements().getMeasurementsRange().getMeasurements().getMeasure().getMetric();
			result += md.getSubsumedMetrics().get(0).getName() + "\t"
					+ md.getSubsumedMetrics().get(1).getName() + "\n";
			// list data
			List<Measure> list1 = omdSeries1.getMeasurements();
			List<Measure> list2 = omdSeries2.getMeasurements();
			for (int pos = 0; pos < list1.size(); pos++) {
				result += list1.get(pos) + "\t" + list2.get(pos) + "\n";
			}
			return result;
		} catch (NullPointerException npe) {
			logger.log(Level.SEVERE, "Access to created example measurements failed. The data is either not created by the example code or there are store/load errors.");
			return "";
		}
	}

	/**Reads and prints the data stored for the last experiment group in the repository.
	 * Last refers to the position of the experiment group in the list of the repository.
	 * @param repo The repository.
	 * @return Formatted data or <code>null</code> if there is no data.
	 */
	public String printStoredMeasurements(Repository repo) {
		if (repo.getExperimentGroups().size() == 0) {
			logger.warning("No experiment group present for which data can be printed.");
			return null;
		} else {
			return printStoredMeasurements(repo, repo.getExperimentGroups().size() - 1);
		}
	}
}
			