package eu.qimpress.resultmodel.helpers;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;

import org.eclipse.emf.ecore.EObject;

import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.AnalysisResult;
import eu.qimpress.resultmodel.ResultObject;
import eu.qimpress.resultmodel.ResultRepository;

/**
 * This helper class allows for traversing a ResultRepository and searching for
 * ResultObjects.
 * 
 * @author Viliam Simko
 */
public abstract class ResultRepositoryHelper {

	/**
	 * Filters on date/time interval (using Date objects as parameters).
	 * 
	 * @param repo
	 * @param dateFrom
	 * @param dateTo
	 * @return
	 */
	static public Collection<ResultObject> getObjectsByTimestampPeriod(
			ResultRepository repo, Date dateFrom, Date dateTo) {

		// results will be aggregated to this list
		LinkedList<ResultObject> listFound = new LinkedList<ResultObject>();

		// alternatives of repository
		for (AlternativeEvaluation alternativeEvaluation : repo
				.getAnalysisRuns()) {

			// analyzes of an alternative
			// the run date must fit into the time interval
			if (alternativeEvaluation.getRunDate().compareTo(dateFrom) >= 0
					&& alternativeEvaluation.getRunDate().compareTo(dateTo) <= 0) {

				// System.out.println("found:"
				// + alternativeEvaluation.getRunDate());
				// System.out.println("comparing to interval: " + dateFrom
				// + " .. " + dateTo);

				// results of an analysis
				for (AnalysisResult analysisResult : alternativeEvaluation
						.getAnalysisResults()) {

					// result objects of a result
					for (EObject resultObject : analysisResult.eContents()) {
						listFound.add((ResultObject) resultObject);
					}
				}
			}
		}

		return listFound;
	}

	/**
	 * Filters on date/time interval (using arrays as parameters).
	 * 
	 * @param repo
	 * @param dateFrom
	 *            {@link ResultRepositoryHelper#createCalendarFromArray(int[], boolean)}
	 * @param dateTo
	 *            {@link ResultRepositoryHelper#createCalendarFromArray(int[], boolean)}
	 * @return
	 */
	static public Collection<ResultObject> getObjectsByTimestampPeriod(
			ResultRepository repo, int[] dateFrom, int[] dateTo) {

		Calendar calFrom = createCalendarFromArray(dateFrom, true);
		Calendar calTo = createCalendarFromArray(dateTo, false);

		return getObjectsByTimestampPeriod(repo, calFrom.getTime(), calTo
				.getTime());
	}

	/**
	 * Traverse ResultRepository and look for instances of ResultObject.
	 * 
	 * @param repo
	 *            repository to traverse
	 * @return objects found aggregated in a list
	 */
	static public Collection<ResultObject> getAllResultObjects(
			ResultRepository repo) {

		// results will be aggregated to this list
		LinkedList<ResultObject> out = new LinkedList<ResultObject>();

		recursiveGetAllResultObjects(repo, out);
		return out;
	}

	/**
	 * Prepares a calendar from an array.
	 * 
	 * @param arr
	 *            {Year, [Month, [Day, [Hour, [Minute, [Second]]]]] }
	 * @param isBegin
	 *            1.January if TRUE, 31.December if FALSE
	 * @return
	 */
	static private Calendar createCalendarFromArray(int[] arr, boolean isBegin) {
		Calendar cal = Calendar.getInstance();
		if (isBegin)
			cal.set(0, Calendar.JANUARY, 1, 0, 0, 0);
		else
			cal.set(0, Calendar.DECEMBER, 31, 23, 59, 59);

		switch (arr.length) {
		case 6:
			cal.set(Calendar.SECOND, arr[5]);
		case 5:
			cal.set(Calendar.MINUTE, arr[4]);
		case 4:
			cal.set(Calendar.HOUR_OF_DAY, arr[3]);
		case 3:
			cal.set(Calendar.DAY_OF_MONTH, arr[2]);
		case 2:
			cal.set(Calendar.MONTH, arr[1] - 1); // Calendar uses 0-based month
		case 1:
			cal.set(Calendar.YEAR, arr[0]);
			break;
		}
		return cal;
	}

	/**
	 * We are using a recursive method to traverse the object tree. See also:
	 * {@link ResultRepositoryHelper#getAllResultObjects(ResultRepository)}
	 * 
	 * @param root
	 *            where to start
	 * @param out
	 *            results will be aggregated here
	 */
	static private void recursiveGetAllResultObjects(EObject root,
			Collection<ResultObject> out) {

		for (EObject obj : root.eContents()) {
			if (obj instanceof ResultObject) {
				out.add((ResultObject) obj);
			}
			recursiveGetAllResultObjects(obj, out);
		}
	}
}
