package de.fzi.kamp.ui.workplanediting.adapter;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

import de.fzi.kamp.service.maineditor.ICommandHandler;
import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.kamp.ui.workplanediting.listeners.EditableTimeEffortModifyListener;
import de.fzi.kamp.ui.workplanediting.listeners.EffortEstimationEnterListener;
import de.fzi.kamp.ui.workplanediting.listeners.SetFocusAndEnterButtonListener;
import de.fzi.kamp.ui.workplanediting.provider.LowestAbstractionContentProvider;
import de.fzi.kamp.ui.workplanediting.provider.LowestAbstractionLabelProvider;
import de.fzi.kamp.ui.workplanediting.provider.WorkPlanContainmentContentProvider;
import de.fzi.kamp.ui.workplanediting.provider.WorkPlanFollowUpContentProvider;
import de.fzi.kamp.ui.workplanediting.provider.WorkPlanTreeLabelProvider;
import de.fzi.kamp.ui.workplanediting.provider.WorkplanTableColumnConstants;
import de.fzi.maintainabilitymodel.workplan.Activity;
import de.fzi.maintainabilitymodel.workplan.CompositeTask;
import de.fzi.maintainabilitymodel.workplan.Workplan;

public class WorkPlanContentAdapter extends EContentAdapter {
    private final static Logger logger = Logger.getLogger(WorkPlanContentAdapter.class);

    private Tree tree;
    private List<Control> controls;
    private TreeViewer treeViewer;
    private ICommandHandler commandHandler;

    public WorkPlanContentAdapter(Tree tree, ICommandHandler commandHandler) {
        super();
        this.tree = tree;
        this.treeViewer = new TreeViewer(tree);
        this.controls = new LinkedList<Control>();
        this.commandHandler = commandHandler;
    }

    @Override
    public void notifyChanged(Notification notification) {

        // if (notification.getFeatureID(MaintainabilityAnalysisModel.class)) {
        //
        // update(workplan);
        // }
        super.notifyChanged(notification);
    }

    /**
     * Updates the whole workplan if changes in the model have occurred. This method is adressed
     * directly by some other methods (use "References in Workspace to find out) and not activated
     * by the adapter of the EMF model.
     * 
     * @param workplan
     * @param followUp
     * @param isEditableEffort
     * @param lowestAbstraction
     */
    public void update(Workplan workplan, boolean followUp, final boolean isEditableEffort, boolean lowestAbstraction) {

//        if (!controls.isEmpty()) {
//            for (Control control : controls) {
//                control.dispose();
//            }
//            controls = new LinkedList<Control>();
//        }

        if (lowestAbstraction) {
            treeViewer.setContentProvider(new LowestAbstractionContentProvider());
            treeViewer.setLabelProvider(new LowestAbstractionLabelProvider());
        } else {
            if (followUp) {
                treeViewer.setContentProvider(new WorkPlanFollowUpContentProvider());
            } else {
                treeViewer.setContentProvider(new WorkPlanContainmentContentProvider());
            }
            treeViewer.setLabelProvider(new WorkPlanTreeLabelProvider());
        }
        treeViewer.setInput(workplan);
        treeViewer.expandAll();
        treeViewer.addTreeListener(new ITreeViewerListener() {

            @Override
            public void treeCollapsed(TreeExpansionEvent event) {
                resizeAllCurrentColumns((Tree) event.getTreeViewer().getControl());
            }

            @Override
            public void treeExpanded(TreeExpansionEvent event) {
                resizeAllCurrentColumns((Tree) event.getTreeViewer().getControl());
            }
        });

        updateOrPackColumns(isEditableEffort);

        if (isEditableEffort) {
            this.updateItemEditors(true);
        }

        blankOutColumns();// temporary method for Q-Impress EU presentation
    }

    private void updateOrPackColumns(boolean isEditableEffort) {
        for (int i = 0; i < tree.getColumnCount(); i++) {
            if (!isEditableEffort
                    && ((i == WorkplanTableColumnConstants.COLUMN_ESTIMATEDTIMEEFFORT)
                            || (i == WorkplanTableColumnConstants.COLUMN_ACCUMULATEDTIMEEFFORT)
                            || (i == WorkplanTableColumnConstants.COLUMN_COMPONENT)
                            || (i == WorkplanTableColumnConstants.COLUMN_OPERATION) || (i == WorkplanTableColumnConstants.COLUMN_INTERFACEPORT))) {
                tree.getColumns()[i].setWidth(0);
                tree.getColumns()[i].setResizable(false);
            } else {
                tree.getColumns()[i].pack();
            }
        }
        tree.getColumn(WorkplanTableColumnConstants.COLUMN_DEVELOPERSTRUCTURE).setWidth(0);
        tree.getColumn(WorkplanTableColumnConstants.COLUMN_DEVELOPERSTRUCTURE).setResizable(false);
    }

    private void updateItemEditors(boolean doFirstLevel) {

        TreeEditor editor = new TreeEditor(tree);
        if (editor.getEditor() != null)
            editor.getEditor().dispose();

        for (TreeItem item : tree.getItems()) {
            if (doFirstLevel) {
                createEditableTextfield(item);
            }
            updateItemEditorsOfSubItems(item);
        }

        for (TreeColumn column : tree.getColumns()) {
            if (column.getText() != WorkplanTableColumnConstants.CAPTION_COLUMN_DEVELOPERSTRUCTURE)
                column.pack();
        }

        tree.getColumn(WorkplanTableColumnConstants.COLUMN_BASICACTIVITY).setAlignment(SWT.CENTER);
        tree.getColumn(WorkplanTableColumnConstants.COLUMN_ESTIMATEDTIMEEFFORT).setAlignment(SWT.CENTER);
        tree.getColumn(WorkplanTableColumnConstants.COLUMN_ACCUMULATEDTIMEEFFORT).setAlignment(SWT.CENTER);
    }

    private void updateItemEditorsOfSubItems(TreeItem item) {
        if (item.getItems().length > 0) {
            for (TreeItem child : item.getItems()) {
                if (child.getData() != null)
                    createEditableTextfield(child);
                if (child.getItems().length > 0)
                    updateItemEditorsOfSubItems(child);
            }
        }
    }

    /**
     * Creates and places the textfields to insert the effort which is potentially necessary to
     * execute a planned activity in the workplan.
     * 
     * @param item
     */
    private void createEditableTextfield(TreeItem item) {

        if (!(item.getData() instanceof CompositeTask)) {
            item.setFont(SurfaceFactory.getFontStyle().get(SurfaceFactory.STYLE_ANSWER));
            Text text = new Text(tree, SWT.CENTER | SWT.SINGLE);
            
            
            this.controls.add(text);
            Activity activity = (Activity) item.getData();

            text.addModifyListener(new EditableTimeEffortModifyListener(activity, text, tree.getDisplay(),
                    this.treeViewer, item, this.commandHandler));
            text.addKeyListener(new EffortEstimationEnterListener(controls, text));
            text.addMouseListener(new SetFocusAndEnterButtonListener(text));
            createEditor(text, item, WorkplanTableColumnConstants.COLUMN_ESTIMATEDTIMEEFFORT);
        }
    }

    /**
     * Creates the editor to insert the textfield in the workplan tree.
     * 
     * @param control
     * @param item
     * @param column
     */
    private void createEditor(Control control, TreeItem item, int column) {
        TreeEditor editor = new TreeEditor(tree);

        if (control instanceof Text)
            if (item.getData() != null) {
                ((Text) control).setText(Double.toString(((Activity) item.getData()).getWorkTimeEstimate()));
            } else {
                ((Text) control).setText("edit");
            }

        editor.grabHorizontal = true;
        editor.grabVertical = true;
        editor.setEditor(control, item, column);
    }

    public TreeViewer getTreeViewer(Tree tree) {
        if (this.treeViewer.getTree() == tree)
            return this.treeViewer;
        else
            return null;
    }

    /**
     * Blanks out some special columns (see the indices). Has been written for the EU-presentation
     * so hide some content. TODO: Has to be removed!
     */
    private void blankOutColumns() {
        for (TreeColumn column : this.tree.getColumns()) {
            if (column.getText().equals(WorkplanTableColumnConstants.CAPTION_COLUMN_COMPONENT)
                    || column.getText().equals(WorkplanTableColumnConstants.CAPTION_COLUMN_INTERFACEPORT)
                    || column.getText().equals(WorkplanTableColumnConstants.CAPTION_COLUMN_OPERATION)) {
                column.setWidth(0);
                column.setResizable(false);
            }
        }
    }

    /**
     * This method is called when the workplan tree is expanded or collapsed. It makes the columns
     * fitting to its (new) content.
     * 
     * @param tree
     */
    private void resizeAllCurrentColumns(Tree tree) {
        for (TreeColumn column : tree.getColumns()) {
            // all columns which are blanked out have a width of zero.
            // They should not be resized.
            if (column.getWidth() != 0) {
                column.pack();
            }
        }
    }
}
