package de.fzi.kamp.ui.workplanderivation.wizard.initialderivation;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

import de.fzi.kamp.service.workplanmanagement.WorkplanDerivationManager;
import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.kamp.ui.workplanderivation.data.InterfacePortSelectionContainerDecorator;
import de.fzi.kamp.ui.workplanderivation.data.InterfaceSelectionContainerDecorator;
import de.fzi.kamp.ui.workplanderivation.listeners.CheckBoxListenerInterfacePortIfInterfaceBefore;
import de.fzi.kamp.ui.workplanderivation.listeners.InterfacePortCheckBoxListener;
import de.fzi.kamp.ui.workplanderivation.listeners.InterfacePortComboListener;
import de.fzi.kamp.ui.workplanderivation.listeners.SignatureChangeInterfacePortCheckboxListener;
import de.fzi.kamp.ui.workplanderivation.wizards.InitialWiazardElementsFactory;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractInterfacePort;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.AbstractContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.BasicActivity;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.ComponentSelectionContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.CompositeTaskDerivationContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.InterfacePortSelectionContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.InterfaceSelectionContainer;

public class SelectInterfacePortPage extends WizardPage {
    private final static Logger logger = Logger.getLogger(SelectInterfacePortPage.class);

    private static final int COLUMN_SELECTIONCHECKBOX = 0;
    private static final int COLUMN_PREVIOUSELEMENT = 1;
    private static final int COLUMN_INTERFACEPORTNAME = 2;
    private static final int COLUMN_IPTYPE = 3;
    private static final int COLUMN_BASICACTIVITYCOMBO = 4;
    private static final int COLUMN_SIGNATURECHANGE = 5;

    private CompositeTaskDerivationContainer workplanContainer;
    private WorkplanDerivationManager workplanDerivationManager;

    private Composite container;
    private Map<InterfacePortSelectionContainer, InterfacePortSelectionContainerDecorator> container2decoratorMap;
    private Table table;
    private boolean startingPointIsComponent = false;

    private SurfaceFactory factory = new SurfaceFactory();
    
    public SelectInterfacePortPage(CompositeTaskDerivationContainer workplanContainer) {
        super("Derive Work Plan");
        setTitle("Select Interface Port Activities");

        this.workplanContainer = workplanContainer;
        this.workplanDerivationManager = new WorkplanDerivationManager(workplanContainer);
        this.container2decoratorMap = new HashMap<InterfacePortSelectionContainer, InterfacePortSelectionContainerDecorator>();
    }

    public void dispose() {
        this.factory.cleanup();
        super.dispose();
    }
    
    /**
     * The general method which has to be implemented to create the controls of the wizard page.
     * Depending on the kind of architecture element (component, datatype, interface) with which the
     * derivation process has started, the table on the interface port page is designed.
     */
    @Override
    public void createControl(Composite parent) {
        container = this.factory.createCompositeOrGroup(parent, 1, SWT.FILL, 1, SWT.NONE, false, 0, 0);
        setControl(container);

        if (getPreviousPage() instanceof SelectComponentActivitiesWizardPage) {

            startingPointIsComponent = true;

            setDescription("Please check the interface ports which are affected\n"
                    + "by the changes in the previously selected components.");

            if (this.workplanDerivationManager.getSelectedComponents().size() > 0) {
                configurePageForInitialComponentsSelection(this.workplanDerivationManager.getSelectedComponents());
            }
        } else {

            setDescription("Please check the interface ports which are affected\n"
                    + "by the changes in the previously selected interfaces.");

            if (getPreviousPage() instanceof SelectInterfacesPage) {
                if (this.workplanDerivationManager.getSelectedInterfaces().size() > 0) {
                    configurePageForInitialInterfacesSelection(this.workplanDerivationManager.getSelectedInterfaces());
                }
            }
        }
        getShell().pack();
    }

    /**
     * In case that the derivation process has started with marking the changing components, the
     * table is built up by listing the component's refinements.
     * 
     * @param selectedContainer
     */
    private void configurePageForInitialComponentsSelection(List<ComponentSelectionContainer> selectedContainer) {

        this.table = InitialWiazardElementsFactory.createTableForInterfacePortSelection(container, true);
        this.table.addListener(SWT.MeasureItem, new Listener() {
            public void handleEvent(Event event) {
                event.height = 25;
            }
        });

        for (ComponentSelectionContainer componentContainer : selectedContainer) {
            if (componentContainer.isSelected()) {

                List<AbstractInterfacePort> alreadyDisplayed = new LinkedList<AbstractInterfacePort>();

                for (InterfacePortSelectionContainer interfacePortContainer : componentContainer.getRefinements()) {
                    if (!alreadyDisplayed.contains(interfacePortContainer.getInterfaceport())) {
                        alreadyDisplayed.add(interfacePortContainer.getInterfaceport());
                        createItemForInterfacePort(interfacePortContainer, table,
                                componentContainer.getBasicActivity(), false);
                    }
                }
            }
        }

        for (TableColumn column : table.getColumns()) {
            column.pack();
        }
        table.pack();

        setPageComplete(checkedAndSelected());
    }

    /**
     * In case that the previous page has been the interface choosing page, the interface ports
     * which implement the selected interfaces are listed.
     * 
     * @param interfaceContainers
     */
    private void configurePageForInitialInterfacesSelection(List<InterfaceSelectionContainer> interfaceContainers) {

        this.table = InitialWiazardElementsFactory.createTableForInterfacePortSelection(container, false);

        List<InterfaceSelectionContainer> lastShownInterfaceContainers = this.workplanContainer
                .getLastShownInterfaceContainers();
        if (lastShownInterfaceContainers.size() > 0) {

            for (InterfaceSelectionContainer ifaceContainer : interfaceContainers) {
                if (ifaceContainer.isSelected()) {
                    for (InterfacePortSelectionContainer interfacePort : ifaceContainer.getFollowups()) {
                        createItemForInterfacePort(interfacePort, table, ifaceContainer.getBasicActivity(), true);
                    }
                }
            }

            for (TableColumn column : table.getColumns()) {
                column.pack();
            }
            table.pack();
        }
        setPageComplete(checkedAndSelected());
    }

    /**
     * In this method for each interface port an table item is created and filled with information,
     * the checkboxes an the combo for the basic activities.
     * 
     * @param ifacePortContainer
     * @param table
     * @param basicActivity
     * @param changeInterface
     */
    private void createItemForInterfacePort(InterfacePortSelectionContainer ifacePortContainer, Table table,
            BasicActivity basicActivity, boolean changeInterface) {

        // to set the height of the table rows
        this.table.addListener(SWT.MeasureItem, new Listener() {
            public void handleEvent(Event event) {
                event.height = 25;
            }
        });

        TableItem itemForAssociatedInterface = new TableItem(table, SWT.NONE);
        itemForAssociatedInterface.setFont(SurfaceFactory.getFontStyle().get(SurfaceFactory.STYLE_ANSWER));

        setStartingPointDependentInformation(itemForAssociatedInterface, ifacePortContainer);

        if (basicActivity != BasicActivity.REMOVE) {
            setGeneralFieldsForTableItem(ifacePortContainer, table, basicActivity, changeInterface,
                    itemForAssociatedInterface);
        } else {
            Button placeboCheckbox = InitialWiazardElementsFactory.createPlaceboSelectionCheckbox(table,
                    itemForAssociatedInterface);

            Combo placeboCombo = InitialWiazardElementsFactory.createPlaceboBasicActivityCombo(table,
                    itemForAssociatedInterface);

            Button signatureChangeCheckbox = new Button(table, SWT.CHECK);
            signatureChangeCheckbox.setEnabled(false);
            this.factory.setControlWithTableEditor(table, itemForAssociatedInterface, signatureChangeCheckbox,
                    COLUMN_SIGNATURECHANGE);

            // Here the assumption is made, that an interface port is removed, too,
            // if it's parent component is removed.
            ifacePortContainer.setBasicActivity(BasicActivity.REMOVE);
            ifacePortContainer.setSelected(true);

            InterfacePortSelectionContainerDecorator decorator = createDecorator(ifacePortContainer,
                    itemForAssociatedInterface, placeboCheckbox, placeboCombo, signatureChangeCheckbox);

            checkWhetherAlreadyChosen(decorator);

            checkedAndSelected();
        }
    }

    /**
     * Because the table has got different column headings depending on the previously changed
     * architecture elements, the columns for the predecessor (interface or component) and the
     * column for the interface port name have to be handled seperately.
     * 
     * @param itemForAssociatedInterface
     * @param ifacePortContainer
     */
    private void setStartingPointDependentInformation(TableItem itemForAssociatedInterface,
            InterfacePortSelectionContainer ifacePortContainer) {

        if (startingPointIsComponent) {
            itemForAssociatedInterface.setText(COLUMN_PREVIOUSELEMENT, ifacePortContainer.getParent()
                    .getComponenttype().getName());
            itemForAssociatedInterface.setText(COLUMN_INTERFACEPORTNAME, ifacePortContainer.getInterfaceport()
                    .getName());
        } else {
            itemForAssociatedInterface.setText(COLUMN_PREVIOUSELEMENT, ifacePortContainer
                    .getInterfaceSelectionContainer().getReferencedInterface().getName());
            itemForAssociatedInterface.setText(COLUMN_INTERFACEPORTNAME, ifacePortContainer.getParent()
                    .getComponenttype().getName()
                    + "." + ifacePortContainer.getInterfaceport().getName());
        }
    }

    /**
     * The columns for the general selection checkbox, the interface port type (provided or
     * required), the combo for the basic activity and the checkbox for marking whether a signature
     * change is performed occur independently of the previous selected architecture element. Hence,
     * they can be created without considering alternatives.
     * 
     * @param ifacePortContainer
     * @param table
     * @param basicActivity
     * @param changeInterface
     * @param itemForAssociatedInterface
     */
    private void setGeneralFieldsForTableItem(InterfacePortSelectionContainer ifacePortContainer, Table table,
            BasicActivity basicActivity, boolean changeInterface, TableItem itemForAssociatedInterface) {

        Button selectionCheckbox = new Button(table, SWT.CHECK);
        this.factory.setControlWithTableEditor(table, itemForAssociatedInterface, selectionCheckbox,
                COLUMN_SELECTIONCHECKBOX);

        Combo combo = InitialWiazardElementsFactory.createComboForBasicActivity(basicActivity, table,
                itemForAssociatedInterface, COLUMN_BASICACTIVITYCOMBO);
        combo.setEnabled(false);

        Button signatureChangeCheckbox = new Button(table, SWT.CHECK);
        signatureChangeCheckbox.setEnabled(false);
        this.factory.setControlWithTableEditor(table, itemForAssociatedInterface, signatureChangeCheckbox,
                COLUMN_SIGNATURECHANGE);

        // set the type of interface port, whether provided or required
        if (ifacePortContainer.getInterfaceport().isRequired()) {
            itemForAssociatedInterface.setText(COLUMN_IPTYPE, "required");
        } else {
            itemForAssociatedInterface.setText(COLUMN_IPTYPE, "provided");
        }

        InterfacePortSelectionContainerDecorator decorator = createDecorator(ifacePortContainer,
                itemForAssociatedInterface, selectionCheckbox, combo, signatureChangeCheckbox);

        if (getPreviousPage() instanceof SelectComponentActivitiesWizardPage) {
            selectionCheckbox.addSelectionListener(new InterfacePortCheckBoxListener(this, decorator));
        } else {
            selectionCheckbox.addSelectionListener(new CheckBoxListenerInterfacePortIfInterfaceBefore(this, decorator));
        }

        combo.addSelectionListener(new InterfacePortComboListener(this, decorator, changeInterface));
        signatureChangeCheckbox.addSelectionListener(new SignatureChangeInterfacePortCheckboxListener(decorator));

        checkWhetherAlreadyChosen(decorator);
    }

    /**
     * Here the decorator wrapping the actual architecture element and the SWT widgets like the
     * combo and the checkboxes is created.
     * 
     * @param ifaceContainer
     * @param itemForAssociatedInterface
     * @param checkbox
     * @param combo
     * @return
     */
    private InterfacePortSelectionContainerDecorator createDecorator(InterfacePortSelectionContainer ifaceContainer,
            TableItem itemForAssociatedInterface, Button checkbox, Combo combo, Button signatureChangeCheckbox) {
        InterfacePortSelectionContainerDecorator decorator = new InterfacePortSelectionContainerDecorator(
                ifaceContainer);

        container2decoratorMap.put(ifaceContainer, decorator);

        itemForAssociatedInterface.setData(ifaceContainer);
        decorator.setCheckbox(checkbox);
        decorator.setCombo(combo);
        decorator.setTableItem(itemForAssociatedInterface);
        decorator.setSignatureChangeCheckbox(signatureChangeCheckbox);

        this.workplanDerivationManager.addShowedInterfacePort(ifaceContainer);
        return decorator;
    }

    public void disposeOfAll(Table table) {
        for (TableItem item : table.getItems()) {
            if (item.getData() != null) {
                ((InterfaceSelectionContainerDecorator) item.getData()).getCheckbox().dispose();
                ((InterfaceSelectionContainerDecorator) item.getData()).getCombo().dispose();
            } else
                table.removeAll();
        }
    }

    /**
     * After every action performed on the table items, it has to be checked, whether the user is
     * allowed to switch to the next page by clicking the "Next" button or not. The rules for it are
     * implemented here.
     * 
     * @return
     */
    public boolean checkedAndSelected() {

        setErrorMessage(null);

        if (checkWhetherOnlyRemoveActions())
            return true;
        else {

            List<InterfacePortSelectionContainer> selectedItems = this.workplanDerivationManager
                    .getSelectedInterfacePorts();

            if (selectedItems.size() < 1) {
                return false;
            } else {
                boolean checkedAndSelected = true;

                for (InterfacePortSelectionContainer abstractContainer : selectedItems) {

                    if (abstractContainer.getBasicActivity() == BasicActivity.REMOVE)
                        return true;

                    InterfacePortSelectionContainerDecorator decorator = container2decoratorMap.get(abstractContainer);

                    if (abstractContainer.getBasicActivity() == null) {
                        decorator.getTableItem().setBackground(4,
                                this.factory.lookupColor(getContainer().getShell().getDisplay(), 255, 0, 0, "red"));
                        setErrorMessage("If you check a component, you have to chose a basic activity");
                        checkedAndSelected = false;
                    } else {
                        decorator.getTableItem().setBackground(4,
                                this.factory.lookupColor(getContainer().getShell().getDisplay(), 255, 255, 255, "white"));
                    }
                }

                return checkedAndSelected;
            }
        }
    }

    /**
     * Here is checked whether on the previously page has only been chosen "Remove" activities. If
     * this is the case, then the user does not have any more chosing possibilities and he is
     * allowed to press the "Finish" or the "Next" button.
     * 
     * @return
     */
    private boolean checkWhetherOnlyRemoveActions() {

        for (ComponentSelectionContainer container : this.workplanContainer.getLastShownComponentContainers()) {
            if (container.getBasicActivity() != BasicActivity.REMOVE && container.getBasicActivity() != null)
                return false;
        }

        for (InterfaceSelectionContainer container : this.workplanContainer.getLastShownInterfaceContainers()) {
            if (container.getBasicActivity() != BasicActivity.REMOVE && container.getBasicActivity() != null)
                return false;
        }
        return true;
    }

    /**
     * Checks whether an interface port has already been chosen. TODO: CHECK: Can this case occur
     * any more????
     * 
     * @param portDecorator
     */
    private void checkWhetherAlreadyChosen(InterfacePortSelectionContainerDecorator portDecorator) {
        if (portDecorator.getContainer().isSelected()) {
            portDecorator.getCheckbox().setSelection(true);
            portDecorator.getCheckbox().setEnabled(false);

            portDecorator.getCombo().setText(
                    ((AbstractContainer) portDecorator.getTableItem().getData()).getBasicActivity().toString());
        }
    }

    public static int getBasicActivityColumnNumber() {
        return COLUMN_BASICACTIVITYCOMBO;
    }

    public static int getSelectionCheckboxColumnNumber() {
        return COLUMN_SELECTIONCHECKBOX;
    }
}
