package de.fzi.kamp.ui.workorganisation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import de.fzi.kamp.service.commands.ApplyWorkorganisationElementsMappingCommand;
import de.fzi.kamp.service.maineditor.ICommandHandler;
import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.kamp.ui.preparation.providers.AssignedDevelopersTableContentProvider;
import de.fzi.kamp.ui.preparation.providers.AssignedDevelopersTableLabelProvider;
import de.fzi.kamp.ui.workorganisation.listeners.AddDeveloperButtonListener;
import de.fzi.kamp.ui.workorganisation.listeners.AllOrSingleCompaniesButtonListener;
import de.fzi.kamp.ui.workorganisation.listeners.ClearAssignedDevelopersButtonListener;
import de.fzi.kamp.ui.workorganisation.listeners.ComponentToMapListener;
import de.fzi.kamp.ui.workorganisation.listeners.ContextMenuForDeletingAssignedWOElementListener;
import de.fzi.kamp.ui.workorganisation.listeners.RoleComboListener;
import de.fzi.kamp.ui.workorganisation.listeners.SelectSingleCompaniesButtonListener;
import de.fzi.kamp.ui.workorganisation.listeners.TeamComboListener;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractArchitectureModel;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractModelElement;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMComponentProxy;
import de.fzi.maintainabilitymodel.architecturemodel.util.ArchitecturemodelSwitch;
import de.fzi.maintainabilitymodel.workorganisation.Company;
import de.fzi.maintainabilitymodel.workorganisation.Department;
import de.fzi.maintainabilitymodel.workorganisation.Team;
import de.fzi.maintainabilitymodel.workorganisation.WorkOrganisationElement;
import de.fzi.maintainabilitymodel.workorganisation.WorkOrganisationModel;

/**
 * @author tknapp
 */

public class MapElementsToDevelopersDialog extends Dialog {

    private final static Logger logger = Logger.getLogger(MapElementsToDevelopersDialog.class);

    private static final String SHELL_TITLE = "Map developers or teams to compoents";
    private static final String COLUMNTEXT_CENTRALTABLE_COMPONENT = "Component";
    private static final String COLUMNTEXT_CENTRALTABLE_TEAM = "Team";
    private static final String COLUMNTEXT_CENTRALTABLE_ROLE = "Role";

    private SurfaceFactory factory;
    private Composite companySelectionButtonsComposite;
    private WorkOrganisationModel workorganisation;
    private AbstractArchitectureModel architectureModel;
    private ArrayList<Company> selectedCompaniesList;
    private List<Team> currentTeams;
    private Table mappingTable;
    private Label componentName;
    private Table assignedDevelopers;
    private TableViewer assignedDevelopersViewer;
    private Map<SAMMComponentProxy, WorkOrganisationElement> currentMappingMap;
    private ICommandHandler commandHandler;

    private List<TableEditor> usedTableEditors;

    public MapElementsToDevelopersDialog(Shell parentShell, WorkOrganisationModel workorganisation,
            AbstractArchitectureModel architectureModel, ICommandHandler commandHandler) {
        super(parentShell);
        this.workorganisation = workorganisation;
        this.architectureModel = architectureModel;
        this.factory = new SurfaceFactory();
        this.selectedCompaniesList = new ArrayList<Company>();
        this.currentTeams = new ArrayList<Team>();
        this.usedTableEditors = new ArrayList<TableEditor>();
        this.currentMappingMap = new HashMap<SAMMComponentProxy, WorkOrganisationElement>();
        this.commandHandler = commandHandler;
    }

    @Override
    protected Control createContents(Composite parent) {

        getShell().setText(SHELL_TITLE);

        createFilterGroup(parent);
        createMappingTable(parent);

        setAllCompaniesSelected();
        updateTeamsList();
        createComponentPropertiesGroup(parent);
        createBottomButtons(parent);
        fillTable();

        return super.createContents(parent);
    }

    /**
     * Creates the central table, where the components are listed to be mapped to teams, developers
     * or architects.
     * 
     * @param parent
     *            The parent composite (given by Dialog).
     */
    private void createMappingTable(Composite parent) {
        this.mappingTable = this.factory.createTable(parent, 3, new String[] { COLUMNTEXT_CENTRALTABLE_COMPONENT,
                COLUMNTEXT_CENTRALTABLE_TEAM, COLUMNTEXT_CENTRALTABLE_ROLE }, SWT.FILL, true);
        this.factory.setControlHeight(this.mappingTable, 150);
    }

    /**
     * Creates the group on top of the window where the companies can chosen from which the teams
     * are displayed.
     * 
     * @param parent
     *            The parent composite (given by Dialog).
     */
    private void createFilterGroup(Composite parent) {
        Group filterGroup = (Group) this.factory.createCompositeOrGroup(parent, 3, SWT.FILL, 1, SWT.NONE, true);
        filterGroup.setFont(factory.getFontStyle().get(SurfaceFactory.STYLE_TEXT));
        filterGroup.setText("Filter");

        createChooseViewComposite(filterGroup);

        this.factory.setGrabVerticalForGridData(new Label(filterGroup, SWT.SEPARATOR | SWT.VERTICAL), true, SWT.FILL,
                true);

        // buttons to select single companies
        this.companySelectionButtonsComposite = this.factory.createVerticalButtonComposite(filterGroup, SWT.FILL,
                SWT.FILL, getCompanyNames(), SWT.CHECK);
        fillListenerArrayForCheckButtons(companySelectionButtonsComposite.getChildren());
        setCompanySelectionButtonsEnabled(false);
    }

    /**
     * @return Returns all company names given in the corresponding workorganisation.
     */
    private String[] getCompanyNames() {
        String[] companyNames = new String[this.workorganisation.getCompanies().size()];

        int companyIndex = 0;
        for (Company company : this.workorganisation.getCompanies()) {
            if (company.getName() == null)
                companyNames[companyIndex] = "Company No. " + companyIndex;
            else
                companyNames[companyIndex] = company.getName();

            companyIndex++;
        }
        return companyNames;
    }

    /**
     * Adds a listener to every checkbox button representing a company in the filter group for
     * companies.
     */
    private void fillListenerArrayForCheckButtons(Control[] companyButtons) {
        List<Company> companyList = this.workorganisation.getCompanies();
        int companyIndex = 0;
        for (Control button : companyButtons) {
            ((Button) button).addSelectionListener(new SelectSingleCompaniesButtonListener((Button) button, this));
            ((Button) button).setData(companyList.get(companyIndex));
            companyIndex++;
        }
    }

    /**
     * Composite to place the buttons in the filter group for the companies.
     * 
     * @param parent
     *            The parent composite for the created composite.
     */
    private void createChooseViewComposite(Composite parent) {
        Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());

        new Label(comp, SWT.NONE).setText("Show teams of");

        // buttons to choose 'all companies', 'selected companies'
        setAllCompaniesOrSingleCompaniesButtonListener(this.factory.createVerticalButtonComposite(comp, SWT.FILL,
                SWT.FILL, new String[] { "all Companies", "selected Companies" }, SWT.RADIO));

    }

    /**
     * Sets the listeners for the buttons to choose, whether the teams of all companies are
     * displayed or just of the selected companies.
     * 
     * @param buttonComposite
     *            The composite which contains the the buttons.
     */
    private void setAllCompaniesOrSingleCompaniesButtonListener(Composite buttonComposite) {
        Assert.isNotNull(buttonComposite, "Button for selecting all or single companies have not been created");
        ((Button) buttonComposite.getChildren()[0]).addSelectionListener(new AllOrSingleCompaniesButtonListener(false,
                this));
        ((Button) buttonComposite.getChildren()[1]).addSelectionListener(new AllOrSingleCompaniesButtonListener(true,
                this));
    }

    /**
     * Sets the company buttons enabled or not.
     * 
     * @param isEnabled
     *            Boolean variable to indicate whether the buttons for the single companies are
     *            enabled or not.
     * 
     */
    public void setCompanySelectionButtonsEnabled(boolean isEnabled) {
        for (Control button : this.companySelectionButtonsComposite.getChildren()) {
            button.setEnabled(isEnabled);
        }
    }

    /**
     * Sets all companies as selected, so all possible teams are displayed.
     */
    public void setAllCompaniesSelected() {
        this.selectedCompaniesList.clear();
        this.selectedCompaniesList.addAll(this.workorganisation.getCompanies());
        setAllCompanyButtonsSelected();
    }

    /**
     * Updates the list of teams, if the set of selected companies was changed.
     */
    public void updateTeamsList() {
        this.currentTeams.clear();
        for (Company company : this.selectedCompaniesList) {
            for (Department departement : company.getDepartments()) {
                this.currentTeams.addAll(departement.getTeams());
            }
        }
    }

    public List<Company> getSelectedCompaniesList() {
        return this.selectedCompaniesList;
    }

    /**
     * This method fills the table where the architecture elements and the teams are mapped.
     */
    public void fillTable() {
        this.mappingTable.clearAll();
        this.mappingTable.removeAll();
        disposeAllTableEditors();
        this.mappingTable.redraw();

        ArchitecturemodelSwitch<AbstractModelElement> selectComponentSwitch = new ArchitecturemodelSwitch<AbstractModelElement>() {
            public AbstractModelElement caseSAMMComponentProxy(SAMMComponentProxy component) {
                TableItem item = new TableItem(getMappingTable(), SWT.NONE);
                item.setText(component.getComponenttype().getName());
                item.setData(component);
                item.setFont(factory.getFontStyle().get(SurfaceFactory.STYLE_TEXT));
                setTableEditor(item);

                return component;
            }
        };

        for (AbstractModelElement modelElement : this.architectureModel.getModelelements()) {
            selectComponentSwitch.doSwitch(modelElement);
        }

        this.mappingTable.getColumn(0).pack();
        this.mappingTable.getColumn(1).setWidth(150);
        this.mappingTable.getColumn(2).setWidth(150);
        this.mappingTable.addSelectionListener(new ComponentToMapListener(this.mappingTable, componentName,
                this.assignedDevelopersViewer));

    }

    /**
     * Creates the group which displays the properties of the currently selected component in the
     * central table.
     * 
     * @param parent
     *            The parent composite for the group.
     */
    private void createComponentPropertiesGroup(Composite parent) {
        Group teamPropertiesGroup = (Group) this.factory.createCompositeOrGroup(parent, 1, SWT.FILL, 1, SWT.NONE, true);
        teamPropertiesGroup.setFont(factory.getFontStyle().get(SurfaceFactory.STYLE_TEXT));
        teamPropertiesGroup.setText("Component Properties");

        createComponentComposite(teamPropertiesGroup);
        createPropertiesLabel(teamPropertiesGroup);
    }

    /**
     * Creates the composite displaying the name of the selected component.
     * 
     * @param parent
     *            The parent composite of the label.
     */
    private void createComponentComposite(Composite parent) {
        Composite comp = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        layout.marginWidth = 0;
        comp.setLayout(layout);
        createComponentLabelRow(comp);
    }

    /**
     * Creates the labels to display the component's name. The labels are displayed in the composite
     * created in <code>createComponentComposite(Composite parent)</code>.
     * 
     * @param parent
     *            The composite created in <code>createComponentComposite(Composite parent)</code>.
     */
    private void createComponentLabelRow(Composite parent) {
        Label componentLabel = new Label(parent, SWT.NONE);
        componentLabel.setFont(this.factory.getFontStyle().getBold(SurfaceFactory.STYLE_TEXT));
        componentLabel.setText("Component: ");
        this.componentName = new Label(parent, SWT.NONE);
        this.componentName.setText("[No component selected]");
    }

    /**
     * Creates the label appearing over table displaying the already assigned developers/teams.
     * 
     * @param parent
     */
    private void createPropertiesLabel(Composite parent) {
        Label description = new Label(parent, SWT.NONE);
        description.setFont(this.factory.getFontStyle().getBold(SurfaceFactory.STYLE_TEXT));
        description.setText("Assigned Teams/Developers: ");
        configureAssignedDevelopersTable(parent);
    }

    /**
     * Some optical configurations for the table displaying the already assigned developers/teams.
     * 
     * @param parent
     */
    private void configureAssignedDevelopersTable(Composite parent) {
        this.assignedDevelopers = new Table(parent, SWT.BORDER | SWT.READ_ONLY);
        GridData data = new GridData();
        data.horizontalAlignment = SWT.FILL;
        data.grabExcessHorizontalSpace = true;
        data.minimumHeight = 80;
        this.assignedDevelopers.setLayoutData(data);
        this.assignedDevelopers.setFont(this.factory.getFontStyle().get(SurfaceFactory.STYLE_TEXT));

        setProvidersToAssignedDevelopersTable();
    }

    /**
     * This method creates and sets a <code>TableViewer</code> to the table displaying the already
     * assigned developers/teams. The input initially is null, because there has not been chosen a
     * component, yet.
     */
    private void setProvidersToAssignedDevelopersTable() {
        this.assignedDevelopersViewer = new TableViewer(this.assignedDevelopers);
        this.assignedDevelopersViewer.setContentProvider(new AssignedDevelopersTableContentProvider());
        this.assignedDevelopersViewer.setLabelProvider(new AssignedDevelopersTableLabelProvider());
        this.assignedDevelopersViewer.setInput(null);

        this.assignedDevelopers.addMouseListener(new ContextMenuForDeletingAssignedWOElementListener(
                this.commandHandler, this.assignedDevelopersViewer, this.mappingTable));
    }

    /**
     * Creates for every item a combo for the teams and another combo for their corresponding team
     * members (Roles). Also the combo listeners are set. selection listeners
     * 
     * @param item
     *            The TableItem representing an ArchitectureElement
     */
    private void setTableEditor(TableItem item) {
        Combo teamCombo = createAndfillTeamCombo();
        createNewTableEditor().setEditor(teamCombo, item, 1);
        Combo rolesCombo = createRolesCombo();
        createNewTableEditor().setEditor(rolesCombo, item, 2);

        teamCombo.addSelectionListener(new TeamComboListener(teamCombo, rolesCombo, this.componentName,
                (SAMMComponentProxy) item.getData(), this.assignedDevelopersViewer));
        rolesCombo.addSelectionListener(new RoleComboListener(rolesCombo, item, teamCombo,
                this.assignedDevelopersViewer, this.currentMappingMap));
    }

    /**
     * Creates TableEditor to set widgets into a Table.
     * 
     * @return
     */
    private TableEditor createNewTableEditor() {
        TableEditor editor = new TableEditor(this.mappingTable);
        this.usedTableEditors.add(editor);
        editor.grabHorizontal = true;
        editor.grabVertical = true;
        return editor;
    }

    /**
     * Creates and fills a combo with the names of the teams to display.
     * 
     * @return Returns the filled combo.
     */
    private Combo createAndfillTeamCombo() {
        Combo combo = new Combo(this.mappingTable, SWT.READ_ONLY);
        List<Team> teamsData = new LinkedList<Team>();
        combo.setData(teamsData);
        for (Team team : this.currentTeams) {
            combo.add(team.getName());
            teamsData.add(team);
        }
        combo.pack();
        return combo;
    }

    private Combo createRolesCombo() {
        return new Combo(this.mappingTable, SWT.READ_ONLY);
    }

    /**
     * This methods disposes all table editors which were used to place all necessary combos for the
     * teams and it's members.
     */
    private void disposeAllTableEditors() {
        for (TableEditor editor : usedTableEditors) {
            if (editor.getEditor() != null)
                editor.getEditor().dispose();
            if (editor.getItem() != null)
                editor.getItem().dispose();

            editor.dispose();
        }
        this.usedTableEditors.clear();
    }

    /**
     * Enables all checkbox buttons standing for a company.
     */
    private void setAllCompanyButtonsSelected() {
        for (Control companyButton : this.companySelectionButtonsComposite.getChildren()) {
            ((Button) companyButton).setSelection(true);
        }
    }

    public Table getMappingTable() {
        return mappingTable;
    }

    /**
     * Creates the buttons for applying the changes made in the table and for clearing the
     * assignments f a selected component.
     * 
     * @param parent
     */
    private void createBottomButtons(Composite parent) {
        String[] buttonNames = { "Apply developer/team assignments",
                "Clear assigned developer/teams of selected component" };
        SelectionAdapter[] selectionListeners = {
                new AddDeveloperButtonListener(this, this.assignedDevelopersViewer),
                new ClearAssignedDevelopersButtonListener(this.mappingTable, this.assignedDevelopersViewer,
                        this.commandHandler) };
        int i = 0;
        for (String name : buttonNames) {
            Button button = new Button(parent, SWT.NONE);
            button.setText(name);
            button.addSelectionListener(selectionListeners[i]);

            GridData data = new GridData();
            data.grabExcessHorizontalSpace = true;
            data.horizontalAlignment = SWT.CENTER;
            button.setLayoutData(data);

            i++;
        }
    }

    @Override
    protected void okPressed() {
        applyDeveloperAssignments();
        super.okPressed();
    }

    /**
     * Applies all changes made in this dialog to the underlying model.
     */
    public void applyDeveloperAssignments() {
        ApplyWorkorganisationElementsMappingCommand command = new ApplyWorkorganisationElementsMappingCommand(
                this.currentMappingMap);
        this.commandHandler.handleCommand(command);
    }
}
