package de.fzi.kamp.ui.preparation.dialogs;

import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractArchitectureModel;
import de.fzi.maintainabilitymodel.workplan.ArchitecturalAlternative;

public class ArchitectureAlternativeLoadDialog extends Dialog {

    private final static Logger logger = Logger.getLogger(ArchitectureAlternativeLoadDialog.class);

    private List<AbstractArchitectureModel> architectureModelList;
    private List<AbstractArchitectureModel> selectedModels;
    private ArchitecturalAlternative alternative;
    private Composite comp;
    private Combo combo;
    private Text nameText;
    private Text descriptionText;
    private boolean edit;
    private SurfaceFactory factory = new SurfaceFactory();

    public ArchitectureAlternativeLoadDialog(Shell parentShell, List<AbstractArchitectureModel> architectureModelList,
            List<AbstractArchitectureModel> selectedModels, boolean edit, ArchitecturalAlternative alternative) {
        super(parentShell);

        this.architectureModelList = architectureModelList;
        this.selectedModels = selectedModels;
        this.edit = edit;
        this.alternative = alternative;
    }

    @Override
    public void configureShell(Shell shell) {

        super.configureShell(shell);
        if (edit)
            shell.setText("Edit Architectural Alternative");
        else
            shell.setText("Choose Architectural Alternatives to load");

        // center the shell and set width and height
        Rectangle rec = shell.getDisplay().getBounds();
        int shellWidth = 500;
        int shellHeight = 220;
        int indentionLeft = (rec.width - shellWidth) / 2;
        int indentionTop = (rec.height - shellHeight) / 2;
        shell.setBounds(indentionLeft, indentionTop, shellWidth, shellHeight);

    }

    public Control createContents(Composite parent) {

        Control control = super.createContents(parent);

        comp = (Composite) super.getDialogArea();

        setLayout(comp);

        if (!edit) {
            createUnderlyingModelCombo();
        } else {
            createSelectedModelLabel();
        }

        if (this.alternative != null) {
            createNamePart(this.alternative.getName());
            createDescriptionPart(this.alternative.getDescription());
        } else {
            createNamePart("");
            createDescriptionPart("");
        }
        return control;
    }

    public void okPressed() {
        if (!edit) {
            setNewAlternative();
        } else {
            setAlternativeAttributes();
        }
        super.okPressed();
    }

    public void setLayout(Composite comp) {

        GridLayout gl = new GridLayout();
        gl.numColumns = 2;
        comp.setLayout(gl);
    }

    public void createUnderlyingModelCombo() {

        Label label = new Label(comp, SWT.NONE);
        label.setText("Select architecture model: ");

        combo = new Combo(comp, SWT.READ_ONLY);
        setLayoutData(combo);

        // the running variable i is just to have different names when choosing an alternative.
        // It is planned to replace this names by the real file names.
        for (AbstractArchitectureModel architectureModel : this.architectureModelList) {
            combo.add(architectureModel.getName());
        }
    }

    public void createNamePart(String oldtext) {

        Label nameLabel = new Label(comp, SWT.None);
        nameLabel.setText("Name: ");
        this.nameText = new Text(comp, SWT.BORDER);

        GridData gd = new GridData();
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalAlignment = SWT.FILL;
        this.nameText.setLayoutData(gd);

        if (oldtext != null)
            this.nameText.setText(oldtext);
    }

    public void createDescriptionPart(String oldtext) {

        Label descriptionLabel = new Label(comp, SWT.None);
        descriptionLabel.setText("Description : ");
        this.descriptionText = new Text(comp, SWT.BORDER);

        GridData gd = new GridData();
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalAlignment = SWT.FILL;
        this.descriptionText.setLayoutData(gd);

        if (oldtext != null) {
            this.descriptionText.setText(oldtext);
        }
    }

    public void createSelectedModelLabel() {

        Label label = new Label(comp, SWT.None);

        if (alternative != null)
            if (alternative.getArchitecturemodel() != null) {
                String archiAltBase = alternative.getArchitecturemodel().getName();
                label.setText("The architecture model underlying this\n" + "architectural alternative is: "
                        + archiAltBase);
            }

        GridData gd = new GridData();
        gd.horizontalSpan = 2;
        label.setLayoutData(gd);
    }

    public void setNewAlternative() {

        // add chosen SAM to "selectedModels"
        // selectedModels.add(samList.get(selectionIndex));
        int selectedItemIndex = combo.getSelectionIndex();

        if (selectedItemIndex >= 0) {
            selectedModels.add(architectureModelList.get(selectedItemIndex));

            // set name and description. The default value for "name"
            // is the name of the chosen SAM
            if (nameText.getText().isEmpty()) {
                this.alternative.setName(architectureModelList.get(selectedItemIndex).getName());
            } else {
                this.alternative.setName(nameText.getText());
            }

            this.alternative.setDescription(descriptionText.getText());
        } else {
            combo.setToolTipText("Please choose a model!");
            combo.setBackground(factory.lookupColor(comp.getDisplay(), 255, 0, 0, "red"));
        }
    }

    public void setAlternativeAttributes() {

        if (!nameText.getText().isEmpty()) {
            alternative.setName(nameText.getText());
        }
        if (!descriptionText.getText().isEmpty())
            alternative.setDescription(descriptionText.getText());

    }

    private void setLayoutData(Combo combo) {
        GridData gd = new GridData();
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalAlignment = SWT.FILL;
        combo.setLayoutData(gd);
    }

    public Composite getComposite() {
        return this.comp;
    }
}
