package de.fzi.kamp.ui.maineditor.stacklayers;

import org.apache.log4j.Logger;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;

import de.fzi.kamp.service.general.AnalysisManager;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.kamp.ui.activator.Activator;
import de.fzi.kamp.ui.general.ActivityImageProvider;
import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.kamp.ui.maineditor.MainEditorUI;
import de.fzi.kamp.ui.maineditor.listeners.GoToAnalysisInstancePageListener;
import de.fzi.kamp.ui.maineditor.listeners.GoToWorkorganisationView;
import de.fzi.kamp.ui.preparation.adapter.ArchitectureAlternativesContentAdapter;
import de.fzi.kamp.ui.preparation.adapter.ChangeRequestContentAdapter;
import de.fzi.kamp.ui.preparation.listeners.AddChangeRequestListener;
import de.fzi.kamp.ui.preparation.listeners.EditAlternativeListener;
import de.fzi.kamp.ui.preparation.listeners.EditChangeRequestListener;
import de.fzi.kamp.ui.preparation.listeners.LoadArchiAltListener;
import de.fzi.kamp.ui.preparation.listeners.RemoveArchitectureListener;
import de.fzi.kamp.ui.preparation.listeners.RemoveChangeRequestListener;
import de.fzi.kamp.ui.workorganisation.listeners.MapElementsToTeamsButtonlistener;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractArchitectureModel;
import de.fzi.maintainabilitymodel.workplan.ArchitecturalAlternative;
import de.fzi.maintainabilitymodel.workplan.WorkplanFactory;

/**
 * Class represents the layer for the preparation page.
 * 
 * @author tknapp
 * 
 */
public class SpecifyArchitectureAlternativesLayer {

    private final static Logger logger = Logger.getLogger(SpecifyArchitectureAlternativesLayer.class);

    private static final String ARCHITECTURE_ALTERNATIVE_SECTION_HEADING = "Architecture Alternatives:";
    private static final String TABLE_HEADING_NAME = "Name";
    private static final String TABLE_HEADING_DESCRIPTION = "Description";
    private static final String[] ARCHI_ALT_BUTTONLABELS = new String[] { "", "", "",
            "Check Workorganisation Annotation" };

    private Display display;
    private MainEditorUI mainEditorUI;

    private SurfaceFactory factory = new SurfaceFactory();
    
    /**
     * Central method to run all necessary methods to build up preparation page.
     * 
     * @param analysisManager
     *            Manager to handle orders concerning the analysis model.
     * @param mainEditor
     *            The editor to display the preparation layer.
     * @param form
     *            The scrolled form containing the preparation layer.
     * @return Returns built up scrolled f
     */
    public void createArchitectureAlternativeLayer(AnalysisManager analysisManager, IMainEditor mainEditor, ScrolledForm form) {

        setupArchitectureAlternativeLayer(mainEditor);

        createArchitectureAlternativePart(analysisManager, display, form);
    }

    /**
     * Assigns all parameters.
     * 
     * @param mainEditor
     *            The editor to display the preparation page.
     */
    public void setupArchitectureAlternativeLayer(IMainEditor mainEditor) {
        mainEditorUI = (MainEditorUI) mainEditor;
        display = mainEditorUI.getEditorSite().getShell().getDisplay();
    }

    /**
     * Creates the section for the architecture alternatives table and the buttons to modify it.
     * 
     * @param analysisManager
     *            Manager to handle orders concerning the analysis model.
     * @param display
     *            Display showing the editor.
     * @param form
     *            The preparation form.
     */
    private void createArchitectureAlternativePart(AnalysisManager analysisManager, Display display, ScrolledForm form) {
        /*
         * Section for architecture alternatives table
         */
        Section architectureAltSection = factory.createSection(form.getBody(),
                ARCHITECTURE_ALTERNATIVE_SECTION_HEADING, Section.TITLE_BAR, TableWrapData.FILL, true, display, 1);

        Composite sectionClient1 = factory.createCompositeForSection(architectureAltSection, 2,
                TableWrapData.CENTER, 1, SWT.NONE, 15, 15, display);

        Table archiAltTable = factory.createTable(sectionClient1, 1, new String[] { TABLE_HEADING_NAME,
                TABLE_HEADING_DESCRIPTION }, SWT.FILL, true);
        factory.setSizeForControl(archiAltTable, 0, 150, false);

        SelectionAdapter[] listenersAtlernativeButtons = {
                new LoadArchiAltListener(analysisManager),
                new EditAlternativeListener(archiAltTable, analysisManager),
                new RemoveArchitectureListener(analysisManager, archiAltTable),
                // new PreparationCheckAnnotationsListener(archiAltTable, analysisManager),
                new MapElementsToTeamsButtonlistener(archiAltTable, analysisManager.getPreparationManager(), display,
                        mainEditorUI) };

        // register adapter for architecture alternative list (note: in emf "adapter" = "observer")
        analysisManager.getAnalysisModel().eAdapters().add(
                new ArchitectureAlternativesContentAdapter(archiAltTable, analysisManager.getAnalysisModel()));
        /*
         * "Check annotations button is blanked out. If you want to insert it again, just add String
         * "Check Annotations" in the array for the button texts.
         */
        factory.createVerticalButtonComposite(sectionClient1, SWT.RIGHT, SWT.FILL, ARCHI_ALT_BUTTONLABELS, SWT.NONE,
                        listenersAtlernativeButtons, null, new Image[] { ActivityImageProvider.getIconForAddButton(),
                                ActivityImageProvider.getIconForEditButton(),
                                ActivityImageProvider.getIconForDeleteButton(), });

        fillInSAMModels(archiAltTable, analysisManager);

        architectureAltSection.setClient(sectionClient1);
    }

    /**
     * Fills all SAM models existing into the workspace into the table. TODO: Change selection, that
     * just the models existing into the same project are shown in the table.
     */
    private void fillInSAMModels(Table archiAltTable, AnalysisManager analysisManager) {
        for (AbstractArchitectureModel architectureModel : analysisManager.getAnalysisModel().getArchitecturemodels()) {
            if (!architectureModelAlreadyContained(archiAltTable, architectureModel)) {
                ArchitecturalAlternative alternative = WorkplanFactory.eINSTANCE.createArchitecturalAlternative();
                alternative.setArchitecturemodel(architectureModel);
                alternative.setName(architectureModel.getName());
                alternative.setDescription("Alternative from Workspace");
                // add to root
                analysisManager.getAnalysisModel().getArchitecturealternatives().add(alternative);
            }
        }
    }

    /**
     * The method tests whether an architecture model found in the workspace has already been
     * displayed in an earlier session or whether it can be added.
     * 
     * @param archiAltTable
     * @param architectureModel
     * @return
     */
    private boolean architectureModelAlreadyContained(Table archiAltTable, AbstractArchitectureModel architectureModel) {

        for (TableItem tableItem : archiAltTable.getItems()) {
            if (((ArchitecturalAlternative) tableItem.getData()).getArchitecturemodel() == architectureModel) {
                return true;
            }
        }
        return false;
    }
    
    protected void finalize() throws Throwable
    {
      this.factory.cleanup();
      super.finalize(); //not necessary if extending Object.
    } 
    
}
