package de.fzi.kamp.ui.maineditor.page;

import org.apache.log4j.Logger;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

import de.fzi.kamp.service.general.AnalysisManager;
import de.fzi.kamp.service.maineditor.ICentralEditorPage;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.kamp.ui.analysisoverview.listeners.ExportToResultmodelListener;
import de.fzi.kamp.ui.general.ActivityImageProvider;
import de.fzi.kamp.ui.general.SurfaceFactory;
import de.fzi.kamp.ui.maineditor.listeners.GoToAnalysisInstancePageListener;
import de.fzi.kamp.ui.maineditor.listeners.GoToArchitectureAlternativePageListener;
import de.fzi.kamp.ui.maineditor.listeners.GoToChangeScenarioPageListener;
import de.fzi.kamp.ui.maineditor.listeners.GoToWorkorganisationView;
import de.fzi.kamp.ui.maineditor.stacklayers.AnalysisInstanceLayer;
import de.fzi.kamp.ui.maineditor.stacklayers.SpecifyArchitectureAlternativesLayer;
import de.fzi.kamp.ui.maineditor.stacklayers.SpecifyChangeScenarioLayer;
import de.fzi.kamp.ui.maineditor.stacklayers.WorkorganisationTreeLayer;
import de.fzi.kamp.ui.maineditor.stacklayers.WorkplanTableLayer;
import de.fzi.maintainabilitymodel.workplan.Workplan;

public class CentralEditorPage implements ICentralEditorPage {

    private final static Logger logger = Logger.getLogger(CentralEditorPage.class);

    private StackLayout stackLayout;

    private ScrolledForm analysisForm;
    private ScrolledForm workplanForm;
    private ScrolledForm architectureAlternativeForm;
    private ScrolledForm workorganisationForm;
    private ScrolledForm changeScenarioForm;

    private SpecifyArchitectureAlternativesLayer architectureAlternativeLayer;
    private SpecifyChangeScenarioLayer changeScenarioLayer;
    private WorkorganisationTreeLayer workorganisationLayer;
    private AnalysisInstanceLayer analysisInstanceLayer;
    private WorkplanTableLayer workplanLayer;

    private SurfaceFactory factory;
    private FormToolkit toolkit;
    private Composite globalContainer;
    private IMainEditor editor;
    private Display display;

    private AnalysisManager analysisManager;
    
    private boolean effortEstimationSelected = false;
    
    public CentralEditorPage() {
        super();
        this.workorganisationLayer = new WorkorganisationTreeLayer();
        this.architectureAlternativeLayer = new SpecifyArchitectureAlternativesLayer();
        this.changeScenarioLayer = new SpecifyChangeScenarioLayer();
    }

    public Control createAnalysisAndWorkplanForm(final AnalysisManager analysisManager, Display display,
            Composite container, IMainEditor editor) {
        this.display = display;
        this.factory = new SurfaceFactory();
        this.toolkit = new FormToolkit(display);
        this.editor = editor;
        this.analysisManager = analysisManager;
        
        //The composite to make the page content scrollable.
        ScrolledForm sourroundingScrolledForm = this.toolkit.createScrolledForm(container);
        sourroundingScrolledForm.getBody().setLayout(createTableWrapLayout(0, 0, 0, 2, false));
        sourroundingScrolledForm.getBody().setBackground(this.factory.lookupColor(display, 210, 230, 255, "sScrolledForm"));
        sourroundingScrolledForm.getBody().setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
        
        final Form navigationForm = createNavigationForm(display, sourroundingScrolledForm.getBody(), factory, toolkit, "Navigation", 1, false);

        createPreparationSection(navigationForm);        
        
        createAnalysisSection(navigationForm);  
        
        this.globalContainer = new Composite(sourroundingScrolledForm.getBody(), SWT.NONE);
        this.globalContainer.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
        this.stackLayout = new StackLayout();
        this.globalContainer.setLayout(stackLayout);

        this.architectureAlternativeForm = createForm(display, globalContainer, this.factory, toolkit, "Specify Architecture Alternative(s)", 1, false);
        this.architectureAlternativeLayer.createArchitectureAlternativeLayer(analysisManager, editor, this.architectureAlternativeForm);

        this.changeScenarioForm = createForm(display, globalContainer, this.factory, toolkit, "Specify Change Scenario(s)", 1, false);
        this.changeScenarioLayer.createChangeScenarioLayer(analysisManager, editor, this.changeScenarioForm);
        
        createAnalysisInstancesLayer();

        createWorkplanLayer();

        this.workorganisationForm = createForm(display, globalContainer, this.factory, toolkit, "Workorganisation", 1, false);
        this.workorganisationLayer.createWorkorganisationTree(analysisManager, display, this.factory, workorganisationForm,
                editor);

        stackLayout.topControl = architectureAlternativeForm;
        globalContainer.layout();

        return sourroundingScrolledForm;
    }

    private void createAnalysisInstancesLayer() {
        disposeAnalysisInstanceLayer();
        this.analysisForm = createForm(this.display, this.globalContainer, this.factory, this.toolkit, "KAMP Analysis Overview",
                1, false);
        this.analysisInstanceLayer = new AnalysisInstanceLayer();
        this.analysisInstanceLayer.createAnalysisInstancePage(this.analysisManager, this.display, analysisForm, this.editor);
    }
    
    private void disposeAnalysisInstanceLayer() {
        if ((this.analysisForm!=null)&&(!this.analysisForm.isDisposed())) {
            this.analysisForm.dispose();
        }
        this.analysisForm = null;
        
        if (this.analysisInstanceLayer!=null) {
            this.analysisInstanceLayer.disposeWidgets();
        }
        this.analysisInstanceLayer = null;
    }

    private void createWorkplanLayer() {
        disposeWorkplanLayer();
        this.workplanForm = createForm(this.display, this.globalContainer, this.factory, this.toolkit, "Workplan Editor", 1, false);
        this.workplanLayer = new WorkplanTableLayer(this);
        this.workplanLayer.createWorkplanTable(this.display, this.editor, workplanForm);
        
        if (this.effortEstimationSelected) {
            this.workplanLayer.setIsEditableEffort(true);
            this.workplanLayer.showEffortEstimationColumns();
            this.workplanLayer.setAddActivityAndCalcFollowUpsButtonsINVISIBLE();
        } else {
            this.workplanLayer.setIsEditableEffort(false);
            this.workplanLayer.setAddActivityAndCalcFollowUpsButtonsVISIBLE();
        }
        this.effortEstimationSelected = false;
    }
    
    private void disposeWorkplanLayer() {
        if ((this.workplanForm!=null)&&(!this.workplanForm.isDisposed())) {
            this.workplanForm.dispose();
        }
        this.workplanForm = null;
        
        if (this.workplanLayer!=null) {
            this.workplanLayer.disposeWidgets();
        }
        this.workplanLayer = null;
    }


    private void createPreparationSection(final Form form) {
        Section section = toolkit.createSection(form.getBody(), Section.TITLE_BAR|Section.TWISTIE|Section.EXPANDED);

        section.setText("Preparation");
        
        Composite sectionClient = toolkit.createComposite(section);
        sectionClient.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
        sectionClient.setLayout(new GridLayout());

        this.factory.createButton(sectionClient, "Specify Architecture Alternative(s)", SWT.NONE, 
                ActivityImageProvider.getArrowRightIcon(), new GoToArchitectureAlternativePageListener(editor));
        this.factory.createButton(sectionClient, "Specify Change Scenario(s)", SWT.NONE, 
                ActivityImageProvider.getArrowRightIcon(), new GoToChangeScenarioPageListener(editor));
        this.factory.createButton(sectionClient, "Specify Workorganisation", SWT.NONE, 
                ActivityImageProvider.getArrowRightIcon(), new GoToWorkorganisationView(editor));
        
        section.setClient(sectionClient);
    }

    private void createAnalysisSection(final Form form) {
        Section section = toolkit.createSection(form.getBody(), Section.TITLE_BAR|Section.TWISTIE|Section.EXPANDED);

        section.setText("Analysis");
        
        Composite sectionClient = toolkit.createComposite(section);
        sectionClient.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
        sectionClient.setLayout(new GridLayout());

        this.factory.createButton(sectionClient, "Analysis Overview", SWT.NONE, 
                ActivityImageProvider.getArrowRightIcon(), new GoToAnalysisInstancePageListener(editor));

        this.factory.createButton(sectionClient, "Export Results", SWT.NONE, 
                ActivityImageProvider.getIconForExportButton(), 
                new ExportToResultmodelListener(this.analysisManager, editor.getEditingDomain().getResourceSet(), editor));

        section.setClient(sectionClient);
    }

    public TreeViewer getWorkorganisationTreeViewer() {
        return this.workorganisationLayer.getWorkorganisationTreeViewer();
    }
    
    private static Form createNavigationForm(Display display, Composite container, final SurfaceFactory factory,
            final FormToolkit toolkit, String text, int numColums, boolean makeColumsEqualwidth) {

        Form form = toolkit.createForm(container);
        form.setForeground(factory.lookupColor(display, 0, 0, 100, "naviforeground"));
        form.setBackground(factory.lookupColor(display, 210, 230, 255, "navibackground"));

        form.setFont(SurfaceFactory.getFontStyle().get(SurfaceFactory.STYLE_HEADING_EDITORPAGE));
        form.setText(text);

        form.getBody().setLayout(createTableWrapLayout(10, 10, 10, numColums, makeColumsEqualwidth));
        return form;
    }
    
    private static ScrolledForm createForm(Display display, Composite container, final SurfaceFactory factory,
            final FormToolkit toolkit, String text, int numColums, boolean makeColumsEqualwidth) {

        ScrolledForm form = toolkit.createScrolledForm(container);
        form.setForeground(factory.lookupColor(display, 0, 0, 100, "naviforeground"));

        form.setFont(SurfaceFactory.getFontStyle().get(SurfaceFactory.STYLE_HEADING_EDITORPAGE));
        form.setText(text);

        form.getBody().setLayout(createTableWrapLayout(10, 10, 40, numColums, makeColumsEqualwidth));
        return form;
    }

    public static TableWrapLayout createTableWrapLayout(int topMargin, int leftMargin, int rightMargin, int numColumns, boolean makeColumnsEqualWidth) {
        TableWrapLayout layout = new TableWrapLayout();
        layout.topMargin = topMargin;
        layout.leftMargin = leftMargin;
        layout.rightMargin = rightMargin;
        layout.numColumns = numColumns;
        layout.makeColumnsEqualWidth = makeColumnsEqualWidth;
        return layout;
    }

    public static TableWrapData configureTableWrapData(int align, boolean grabHorizontal, int colspan,
            boolean grabVertical, int valign) {
        TableWrapData twd = new TableWrapData();

        twd.align = align;
        twd.grabHorizontal = grabHorizontal;
        twd.colspan = colspan;
        twd.grabVertical = grabVertical;
        twd.valign = valign;

        return twd;
    }

    public void setArchitectureAlternativePageAsTopControl() {
        disposeAnalysisInstanceLayer();
        disposeWorkplanLayer();
        
        stackLayout.topControl = architectureAlternativeForm;
        this.globalContainer.layout();
    }

    public void setChangeScenarioPageAsTopControl() {
        disposeAnalysisInstanceLayer();
        disposeWorkplanLayer();

        stackLayout.topControl = changeScenarioForm;
        this.globalContainer.layout();
    }

    public void setWorkplanAsTopControl() {
        disposeAnalysisInstanceLayer();
        this.createWorkplanLayer();

        stackLayout.topControl = workplanForm;
        this.globalContainer.layout();
    }

    public void setAnalysisPageAsTopControl() {
        disposeWorkplanLayer();
        this.createAnalysisInstancesLayer();

        stackLayout.topControl = analysisForm;
        analysisInstanceLayer.getTableViewer().refresh();
        analysisInstanceLayer.getTableViewer().getTable().redraw();
        analysisInstanceLayer.getTableViewer().getTable().update();
        analysisInstanceLayer.getTableViewer().getTable().layout();
        analysisInstanceLayer.getTableViewer().getTable().layout(true, true);
        analysisInstanceLayer.getTableViewer().getTable().deselectAll();
        analysisInstanceLayer.getTableViewer().getTable().select(0);
        this.globalContainer.layout();
        analysisInstanceLayer.layout();
    }

    public void setWorkorganisationPageAsTopControl() {
        disposeAnalysisInstanceLayer();
        disposeWorkplanLayer();

        stackLayout.topControl = workorganisationForm;
        this.globalContainer.layout();
    }

    public void setViewForOpenedWorkplan(Workplan workplan, boolean followUp, boolean isEditableEffort,
            boolean lowestAbstraction) {
        if (workplanLayer != null && workplanForm != null) {
            this.workplanLayer.setIsEditableEffort(isEditableEffort);
            this.workplanForm.reflow(true);
            updateWorkplanTree(workplan, followUp, isEditableEffort, lowestAbstraction);
        }
    }

    public void showDeveloperStructure(boolean showDeveloper) {
        if (this.workplanLayer != null)
            this.workplanLayer.showDeveloperOrLowestAbstactionStructureView(showDeveloper);
    }

    public void updateWorkplanTree(Workplan workplan, boolean followUp, boolean isEditableEffort,
            boolean lowestAbstraction) {
        if (this.workplanLayer != null)
            this.workplanLayer.getAdapter().update(workplan, followUp, isEditableEffort, lowestAbstraction);
    }

    public void setAddCompositeActivityButtonVisible() {
        if (this.workplanLayer != null)
            this.workplanLayer.setAddActivityAndCalcFollowUpsButtonsVISIBLE();
    }

    public void setAddCompositeActivityButtonInvisible() {
        if (this.workplanLayer != null)
            this.workplanLayer.setAddActivityAndCalcFollowUpsButtonsINVISIBLE();
    }

    public WorkplanTableLayer getWorkplanLayer() {
        return this.workplanLayer;
    }

	@Override
	public boolean effortEstimationViewIsActivated() {
		return this.effortEstimationSelected;
	}
	
	public void setEffortEstimationActivation(boolean active) {
	    this.effortEstimationSelected = active;
	}
}
