package de.fzi.kamp.ui.analysisoverview.listeners;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.MessageBox;

import de.fzi.kamp.service.architecturemodel.impl.BackboneModelLoader;
import de.fzi.kamp.service.general.AnalysisManager;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMArchitectureModelProxy;
import de.fzi.maintainabilitymodel.main.EffortAnalysisInstance;
import de.fzi.maintainabilitymodel.workplan.ArchitecturalAlternative;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.operations.SaveQModelUIOperation;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.AnalysisResult;
import eu.qimpress.resultmodel.MaintainabilityPredictionResult;
import eu.qimpress.resultmodel.ResultModelFactory;
import eu.qimpress.resultmodel.ResultRepository;

/**
 * Listener which exports the analysis instances into an XML file (button on maintainability
 * analysis overview page).
 * 
 * @author tknapp
 * 
 */
public class ExportToResultmodelListener extends SelectionAdapter {
    private final static Logger logger = Logger.getLogger(ExportToResultmodelListener.class);

    private AnalysisManager analysisManager;
    private ResourceSet resourceSet;
    private IMainEditor mainEditor;
    private IQProject qproject;

    /**
     * Constructor
     * 
     * @param analysisManager
     * @param resourceSet
     * @param shell
     *            Shell for the dialog.
     */
    public ExportToResultmodelListener(AnalysisManager analysisManager, ResourceSet resourceSet, IMainEditor mainEditor) {
        super();
        this.analysisManager = analysisManager;
        this.resourceSet = resourceSet;
        this.mainEditor = mainEditor;
    }

    @Override
    public void widgetDefaultSelected(SelectionEvent e) {
        widgetSelected(e);
        super.widgetDefaultSelected(e);
    }

    @Override
    public void widgetSelected(SelectionEvent e) {
        Map<ArchitecturalAlternative, List<EffortAnalysisInstance>> architectureToInstancesMap = getUsedArchitecturalAlternatives();

        for (ArchitecturalAlternative alternative : architectureToInstancesMap.keySet()) {

            AlternativeEvaluation alternativeEvaluation = retrieveAlternativeEvaluation(alternative);

            clearOldMaintPredictionResults(alternativeEvaluation);
            
            MaintainabilityPredictionResult maintPredictionResult = ResultModelFactory.eINSTANCE
                    .createMaintainabilityPredictionResult();
            maintPredictionResult.getEffortanalysisinstances().addAll(architectureToInstancesMap.get(alternative));

            alternativeEvaluation.getAnalysisResults().add(maintPredictionResult);
        }

        if (this.qproject != null) {
            try {
                new SaveQModelUIOperation(this.qproject.getRepository().getResultModel()).run(null);
            } catch (RepositoryException ex) {
                // throw new JobFailedException("Failed to save result model.");
            } catch (InvocationTargetException ex) {
                // throw new JobFailedException("Failed to save result model.");
            } catch (InterruptedException ex) {
                // throw new JobFailedException("Failed to save result model.");
            }
        }

        super.widgetSelected(e);
    }

    private void clearOldMaintPredictionResults(AlternativeEvaluation alternativeEvaluation) {
        List<AnalysisResult> oldMaintResults = new ArrayList<AnalysisResult>();
        for (AnalysisResult result : alternativeEvaluation.getAnalysisResults()) {
            if (result instanceof MaintainabilityPredictionResult) {
                oldMaintResults.add(result);
            }
        }
        if (!oldMaintResults.isEmpty())
            alternativeEvaluation.getAnalysisResults().removeAll(oldMaintResults);
    }

    private AlternativeEvaluation retrieveAlternativeEvaluation(ArchitecturalAlternative alternative) {

        if (alternative.getArchitecturemodel() instanceof SAMMArchitectureModelProxy) {
            String name = ((SAMMArchitectureModelProxy) alternative.getArchitecturemodel()).getName();
            IQModel model = BackboneModelLoader.findIQModelForAlternativeName(name, this.mainEditor);

            List<AlternativeEvaluation> alternativeEvaluations = null;

            String alternativeId = model.getAlternative().getInfo().getId();

            if (this.qproject == null) {
                this.qproject = model.getAlternative().getRepository().getQProject();
            }

            try {
                alternativeEvaluations = model.getAlternative().getRepository()
                        .getAlternativeEvaluationsByAlternativeId(alternativeId);
            } catch (RepositoryException e) {
                return null;
            }

            if (!alternativeEvaluations.isEmpty()) {
                return alternativeEvaluations.get(0);
            } else {
                try {
                    AlternativeEvaluation newAlternativeEvaluation = model.getAlternative().getRepository()
                            .createAlternativeEvaluation(alternativeId);
                    newAlternativeEvaluation.setName(name);
                    return newAlternativeEvaluation;
                } catch (RepositoryException e) {
                }
            }

        }

        return null;
    }

    /**
     * Exports the result repository to an XML file.
     * 
     * @param pathName
     *            The path and file name selected by the user.
     * @param resultRoot
     *            The result repository.
     */
    public void exportToXML(String pathName, ResultRepository resultRoot) {

        Resource resource = this.resourceSet.createResource(URI.createFileURI(pathName));
        resource.getContents().add(resultRoot);

        Map<Object, Object> saveOptions = ((XMLResource) resource).getDefaultSaveOptions();
        saveOptions.put(XMLResource.OPTION_CONFIGURATION_CACHE, Boolean.TRUE);
        saveOptions.put(XMLResource.OPTION_USE_CACHED_LOOKUP_TABLE, new ArrayList<Object>());

        try {
            resource.save(saveOptions);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Opens a file dialog to request for the path where to save the result model and the file name.
     * If an already existing file was chosen, the user is asked whether he really wants to override
     * this file.
     * 
     * @return Full saving path including customized file name.
     */
    private String getSAMMUri() {

        FileDialog dialog = new FileDialog(this.mainEditor.getMainEditorShell());
        dialog.setFileName("sample");
        dialog.setFilterExtensions(new String[] { "*.samm_resultmodel" });
        dialog.open();

        String filePath = dialog.getFilterPath();
        String fileName = dialog.getFileName();

        if (filePath.equals(""))
            return "";
        else {
            if (!fileName.contains(".samm_resultmodel")) {
                return filePath + "\\" + fileName + ".samm_resultmodel";
            } else {
                MessageBox box = new MessageBox(this.mainEditor.getMainEditorShell(), SWT.ICON_QUESTION | SWT.YES
                        | SWT.NO);
                box.setMessage("The file you wanted to create already exists. Override?");
                int response = box.open();
                if (response == SWT.YES) {
                    return filePath + "\\" + fileName;
                } else {
                    return "";
                }
            }
        }
    }

    /**
     * This methods goes through the list of existing analysis instances and returns a map having
     * all occurring architecture alternatives as keys and a list of the analysis instances where
     * the key occurs as value.
     * 
     * @return
     */
    private Map<ArchitecturalAlternative, List<EffortAnalysisInstance>> getUsedArchitecturalAlternatives() {
        Map<ArchitecturalAlternative, List<EffortAnalysisInstance>> alternativeToInstancesMap = new HashMap<ArchitecturalAlternative, List<EffortAnalysisInstance>>();

        for (EffortAnalysisInstance occuringInstance : this.analysisManager.getAnalysisInstanceManager()
                .getAnalysisInstances()) {
            if (alternativeToInstancesMap.containsKey(occuringInstance.getTargetArchitecturalAlternative())) {
                alternativeToInstancesMap.get(occuringInstance.getTargetArchitecturalAlternative()).add(occuringInstance);
            } else {
                List<EffortAnalysisInstance> instanceList = new LinkedList<EffortAnalysisInstance>();
                instanceList.add(occuringInstance);
                alternativeToInstancesMap.put(occuringInstance.getTargetArchitecturalAlternative(), instanceList);
            }
        }
        return alternativeToInstancesMap;
    }
}
