package de.fzi.kamp.service.preparation.impl;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.swt.widgets.Display;

import de.fzi.kamp.service.architecturemodel.IArchitectureModelProvider;
import de.fzi.kamp.service.commands.AddChangeRequestCommand;
import de.fzi.kamp.service.commands.CheckComplexityAnnotationsCommand;
import de.fzi.kamp.service.commands.EditChangeRequestCommand;
import de.fzi.kamp.service.commands.RemoveArchiAltCommand;
import de.fzi.kamp.service.commands.RemoveChangeRequestCommand;
import de.fzi.kamp.service.general.IDialogManager;
import de.fzi.kamp.service.maineditor.ICommandHandler;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.kamp.service.preparation.IPreparationManager;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractArchitectureModel;
import de.fzi.maintainabilitymodel.main.MaintainabilityAnalysisModel;
import de.fzi.maintainabilitymodel.workorganisation.WorkOrganisationModel;
import de.fzi.maintainabilitymodel.workplan.ArchitecturalAlternative;
import de.fzi.maintainabilitymodel.workplan.ChangeRequest;
import de.fzi.maintainabilitymodel.workplan.WorkplanFactory;

public class PreparationManager implements IPreparationManager {
    private final static Logger logger = Logger.getLogger(PreparationManager.class);

    private MaintainabilityAnalysisModel analysisModel;
    private IArchitectureModelProvider architectureModelProvider;
    private IDialogManager dialogManager;

    private ICommandHandler commandHandler;

    public PreparationManager(MaintainabilityAnalysisModel analysisModel, IArchitectureModelProvider provider,
            IDialogManager dialogManager, ICommandHandler commandHandler) {
        this.architectureModelProvider = provider;
        this.analysisModel = analysisModel;
        this.dialogManager = dialogManager;
        this.commandHandler = commandHandler;
    }

    @Override
    public List<ArchitecturalAlternative> loadAndSelectArchitectureModels() {

        List<AbstractArchitectureModel> architectureModelList = this.architectureModelProvider.getArchitectureModels();

        ArchitecturalAlternative alternative = WorkplanFactory.eINSTANCE.createArchitecturalAlternative();

        List<AbstractArchitectureModel> selectedModels = new LinkedList<AbstractArchitectureModel>();
        this.dialogManager.showSelectArchitectureModelDialog(this.analysisModel, architectureModelList, selectedModels,
                alternative, this);

        return getArchitectureAlternatives();
    }

    @Override
    public List<ArchitecturalAlternative> getArchitectureAlternatives() {
        return this.analysisModel.getArchitecturealternatives();
    }

    @Override
    public void editArchitectureAlternative(ArchitecturalAlternative architectureAlternative) {

        this.dialogManager.showEditArchitectureAlternativeDialog(architectureAlternative, this);
    }

    @Override
    public void checkComplexityAnnotationsForArchitectureAlternative(ArchitecturalAlternative architectureAlternative) {
        // Not used at the moment
        CheckComplexityAnnotationsCommand command = new CheckComplexityAnnotationsCommand(this, architectureAlternative);
        this.commandHandler.handleCommand(command);
    }

    @Override
    public void removeArchitectureAlternative(ArchitecturalAlternative architectureAlternative) {
        RemoveArchiAltCommand command = new RemoveArchiAltCommand(architectureAlternative, this.analysisModel);
        this.commandHandler.handleCommand(command);
    }

    public void addChangeRequest() {

        ChangeRequest changeRequest = createChangeRequest("", "");
        if (this.dialogManager.showNewChangeRequestDialog(changeRequest)) {

            AddChangeRequestCommand command = new AddChangeRequestCommand(this, changeRequest);
            this.commandHandler.handleCommand(command);
        }
    }

    @Override
    public List<ChangeRequest> getChangeRequests() {
        return this.analysisModel.getChangerequests();
    }

    @Override
    public void editChangeRequest(ChangeRequest changeRequest) {

        if (this.dialogManager.showEditChangeRequestDialog(changeRequest)) {
            EditChangeRequestCommand command = new EditChangeRequestCommand(this, changeRequest);
            this.commandHandler.handleCommand(command);
        }
    }

    @Override
    public void removeChangeRequest(ChangeRequest changeRequest) {
        RemoveChangeRequestCommand command = new RemoveChangeRequestCommand(this, changeRequest);
        this.commandHandler.handleCommand(command);
    }

    public MaintainabilityAnalysisModel getAnalysisModel() {
        return analysisModel;
    }

    public IArchitectureModelProvider getArchitectureModelProvider() {
        return architectureModelProvider;
    }

    public IDialogManager getDialogManager() {
        return dialogManager;
    }

    @Override
    public void openMapElementsToDevelopersDialog(WorkOrganisationModel workorganisation,
            AbstractArchitectureModel architectureModel, Display display, IMainEditor editor) {
        this.dialogManager.openMapElementsToDevelopersDialog(workorganisation, architectureModel, display, editor);

    }

    /**
     * Creates a <code>ChangeRequest</code> and sets it's name and description.
     * 
     * @param name
     *            The name of the change request.
     * @param description
     *            The description of the change request.
     * @return Returns the created change request.
     */
    private ChangeRequest createChangeRequest(String name, String description) {
        ChangeRequest changeRequest = WorkplanFactory.eINSTANCE.createChangeRequest();

        changeRequest.setName(name);
        changeRequest.setDescription(description);

        return changeRequest;
    }
}
