package de.fzi.kamp.service.architecturemodel.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.impl.XMLParserPoolImpl;
import org.eclipse.swt.widgets.Shell;

import de.fzi.kamp.service.architecturemodel.IArchitectureModelProvider;
import de.fzi.kamp.service.architecturemodel.impl.switches.SAMMProxyCreationSwitch;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractArchitectureModel;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractComponent;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractCompositeComponent;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractDatatype;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractInterface;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractInterfacePort;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractModelElement;
import de.fzi.maintainabilitymodel.architecturemodel.AbstractOperationImplementation;
import de.fzi.maintainabilitymodel.architecturemodel.ArchitecturemodelFactory;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMArchitectureModelProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMComponentProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMCompositeComponentProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMDatatypeProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMInterfacePortProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMInterfaceProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMOperationImplementationProxy;
import de.fzi.maintainabilitymodel.architecturemodel.SAMMOperationProxy;
import de.fzi.maintainabilitymodel.main.MaintainabilityAnalysisModel;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.CompositeComponentSelectionContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.CompositeTaskDerivationContainer;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.samm.core.NamedEntity;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.CompositeComponent;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.MessageType;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.staticstructure.Parameter;
import eu.qimpress.samm.staticstructure.PrimitiveComponent;
import eu.qimpress.samm.staticstructure.Repository;
import eu.qimpress.samm.staticstructure.ServiceArchitectureModel;
import eu.qimpress.samm.staticstructure.SubcomponentInstance;

/**
 * Class which works like a middleware between the underlying ServiceArchitectureMetaModel (SAMM)
 * and the elements that will be used while working in the workplan and it's preparation.
 * 
 * @author tknapp
 * 
 */
public class SAMMArchitectureModelProvider implements IArchitectureModelProvider {
    private final static Logger logger = Logger.getLogger(SAMMArchitectureModelProvider.class);

    /**
     * Map provides the mapping between the SAMM model elements and the abstract model elements used
     * in this tool.
     */
    private Map<NamedEntity, AbstractModelElement> proxyToAbstractModelElementMap;

    /**
     * The corresponding <code>MaintainabilityAnalysisModel</code>.
     */
    private MaintainabilityAnalysisModel analysisModel;
    private IMainEditor mainEditor;

    private Map<AbstractArchitectureModel, IQModel> architectureModelToIQModelMap;
    private Map<IQModel, AbstractArchitectureModel> iQModelToArchitectureModelMap;
    
    /**
     * The only constructor.
     * 
     * @param resourceSet
     *            The resource set underlying this main editor.
     * @param analysisModel
     *            The <code>MaintainabiltyAnalysisModel</code> which needs an architecture model
     *            provider.
     * @param shell
     *            The shell of the used editor.
     */
    public SAMMArchitectureModelProvider(ResourceSet resourceSet, MaintainabilityAnalysisModel analysisModel,
            IMainEditor mainEditor) {
        this.analysisModel = analysisModel;
        this.mainEditor = mainEditor;
        loadArchitectureModels(resourceSet, mainEditor.getMainEditorShell());
        
        if (this.proxyToAbstractModelElementMap == null) {
            this.proxyToAbstractModelElementMap = new SAMMProxyMap();           
        }
        buildProxyMapForAlreadyExistingProxies();
    }

    public List<AbstractArchitectureModel> getArchitectureModels() {
        return this.analysisModel.getArchitecturemodels();
    }

    /**
     * Methods which returns an abstract model element for a passed "real" model element (for
     * example an component or interface).
     * 
     * @param sammElement
     *            The model element for which a proxy should be created.
     * @return The proxy (abstract model element) for the corresponding SAMM element.
     */
    public AbstractModelElement getSAMMProxyElement(NamedEntity sammElement) {
        if (sammElement == null) {
            return null;
        }        
        //For the case that there is more then one ArchitectureAlternative (e.g. target & source), it has to
        //be distinguished between the ArchitectureAlternatives. [...].get(0) is BAD!
        AbstractModelElement proxy = null;

        if (this.proxyToAbstractModelElementMap.containsKey(sammElement)) {
            return this.proxyToAbstractModelElementMap.get(sammElement);
        }else{

	        proxy = new SAMMProxyCreationSwitch().doSwitch(sammElement);
	        proxy.setName(sammElement.getName());
	
            if (this.analysisModel != null) {
            	//TODO: Review whether it is clever to set the architecture model 
            	//always to entry zero of the getArchitecturemodels list
                this.analysisModel.getArchitecturemodels().get(0).getModelelements().add(proxy);
            }
            this.proxyToAbstractModelElementMap.put(sammElement, proxy);
            return proxy;
        }      
    }
    
    /**
     * This method returns an already existing SAMMOperationImplementationProxy or creates and returns a new one.
     * It is handled separated from from the other proxy creation, because of the 2:1 mapping in case of an 
     * operation implementation.  
     * @param abstractInterfacePort
     * @param operation
     * @return
     */
    public SAMMOperationImplementationProxy getSAMMOperationImplementationProxy(AbstractInterfacePort abstractInterfacePort, 
    				Operation operation){
    	if (abstractInterfacePort == null) {
            return null;
        }        
    	InterfacePort interfacePort = null;
    	MapKeyForSAMMOperationImplementationProxy operationImplementationProxyKey = null;
    	SAMMOperationImplementationProxy opImplProxy = null;
    	if(interfacePort instanceof SAMMInterfacePortProxy){
    		interfacePort = ((SAMMInterfacePortProxy)interfacePort).getInterfaceport();
    		operationImplementationProxyKey = new MapKeyForSAMMOperationImplementationProxy();	    		
    		operationImplementationProxyKey.setPort(interfacePort);
        	operationImplementationProxyKey.setOperation(operation);
        	
        	MapKeyForSAMMOperationImplementationProxy originalKey = operationImplementationKeyAlreadyExists(operationImplementationProxyKey);
        	if(originalKey != null){
        		return (SAMMOperationImplementationProxy) this.proxyToAbstractModelElementMap.get(originalKey);
        	}else{
        		opImplProxy = 
        			ArchitecturemodelFactory.eINSTANCE.createSAMMOperationImplementationProxy();
        		opImplProxy.setInterfacePort(interfacePort);
        		opImplProxy.setOperation(operation);
        		this.proxyToAbstractModelElementMap.put(operationImplementationProxyKey, opImplProxy);
        	}
    	}        	
    	return opImplProxy;
    }
    
    private MapKeyForSAMMOperationImplementationProxy operationImplementationKeyAlreadyExists(MapKeyForSAMMOperationImplementationProxy operationImplementationProxyKey){
    	Iterator<NamedEntity> it = this.proxyToAbstractModelElementMap.keySet().iterator();
    	
    	while(it.hasNext()){
    		Object key = it.next();
    		if(operationImplementationProxyKey.equals(key)){
    			return (MapKeyForSAMMOperationImplementationProxy)key;
    		}
    	}
    	return null;
    }

    /**
     * The method fills the map containing proxies (abstract model elements) and their accordingly
     * "real" model elements.
     */
    private void buildProxyMapForAlreadyExistingProxies() {
    	//First loop walks through all existing SAMM proxies in the analysis model
        for (AbstractArchitectureModel architectureModel : analysisModel.getArchitecturemodels()) {
        	//Second loop walks through all elements of the architecture model and creates
        	//the corresponding proxy element
            for (Iterator<EObject> iter = EcoreUtil.getAllContents(architectureModel, true); iter.hasNext();) {
                EObject eObject = (EObject) iter.next();
                if ((eObject != null) && (eObject instanceof AbstractModelElement)) {
                    NamedEntity sammModelElement = null;
                    if (eObject instanceof SAMMComponentProxy) {
                        sammModelElement = ((SAMMComponentProxy) eObject).getComponenttype();
                    } else if (eObject instanceof SAMMInterfacePortProxy) {
                        sammModelElement = ((SAMMInterfacePortProxy) eObject).getInterfaceport();
                    } else if (eObject instanceof SAMMInterfaceProxy) {
                        sammModelElement = ((SAMMInterfaceProxy) eObject).getInterface();
                    } else if (eObject instanceof SAMMOperationProxy) {
                        sammModelElement = ((SAMMOperationProxy) eObject).getOperation();
                    } else if (eObject instanceof SAMMDatatypeProxy) {
                        sammModelElement = ((SAMMDatatypeProxy) eObject).getType();
                    }

                    if (sammModelElement != null)
                        this.proxyToAbstractModelElementMap.put(sammModelElement, (AbstractModelElement) eObject);
                }
            }
        }
    }

    /**
     * The method loads and returns a list of <code>AbstractArchitectureModel</code>s.
     * 
     * @param resourceSet
     *            The resource set underlying this main editor.
     * @param shell
     *            The shell of the used editor.
     */
    @Override
    public List<AbstractArchitectureModel> loadArchitectureModels(ResourceSet resourceSet, Shell shell) {
        List<AbstractArchitectureModel> existingArchitectureModels = this.analysisModel.getArchitecturemodels();
        this.architectureModelToIQModelMap = new HashMap<AbstractArchitectureModel, IQModel>();
        this.iQModelToArchitectureModelMap = new HashMap<IQModel, AbstractArchitectureModel>();

        BackboneModelLoader backboneModelLoader = new BackboneModelLoader(this.mainEditor);
        List<IQModel> newModelList = backboneModelLoader.loadArchitectureModelsFromBackbone();

        for (IQModel model : newModelList) {
            String projectName = model.getCorrespondingResource().getProject().getName();
            String samPath = model.getCorrespondingResource().getProjectRelativePath().toString();
            URI samUri = URI.createPlatformResourceURI(projectName + "/" + samPath, true);

            ServiceArchitectureModel sammModel = loadSAMMModelsFromURI(resourceSet, samUri);
            IQModel repositoryModel = BackboneModelLoader.retrieveRepositoryIQModelForIQModel(model);
            
            String repPath = repositoryModel.getCorrespondingResource().getProjectRelativePath().toString();
            URI repURI = URI.createPlatformResourceURI(projectName + "/" + repPath, true);
            Repository repository = loadSAMMRepositoryFromURI(resourceSet, repURI);

            SAMMArchitectureModelProxy alreadyExistingProxy = proxyForArchitectureModelAlreadyExists(sammModel);

            if (sammModel != null && alreadyExistingProxy == null) {
                SAMMArchitectureModelProxy samProxy = ArchitecturemodelFactory.eINSTANCE
                        .createSAMMArchitectureModelProxy();
                samProxy.setServicearchitecturemodel(sammModel);
                samProxy.setRepository(repository);

                samProxy.setName(BackboneModelLoader.calculateNameIdentifierForIQModel(model));

                if (this.analysisModel != null) {
                    this.analysisModel.getArchitecturemodels().add(samProxy);
                }

                this.architectureModelToIQModelMap.put(samProxy, model);
                this.iQModelToArchitectureModelMap.put(model, samProxy);
            } else {
                this.architectureModelToIQModelMap.put(alreadyExistingProxy, model);
                this.iQModelToArchitectureModelMap.put(model, alreadyExistingProxy);
            }
        }
        return existingArchitectureModels;
    }

    /**
     * Tests whether the SAMM loaded from the backbone has already been wrapped into an proxy. It
     * has to be tested, because otherwise the number of possible alternatives to choose increases
     * with every opening of an kamp-file.
     * 
     * @param sammModel
     * @return
     */
    private SAMMArchitectureModelProxy proxyForArchitectureModelAlreadyExists(ServiceArchitectureModel sammModel) {
        for (AbstractArchitectureModel proxy : this.analysisModel.getArchitecturemodels()) {
            if (proxy instanceof SAMMArchitectureModelProxy) {
                if (((SAMMArchitectureModelProxy) proxy).getServicearchitecturemodel().equals(sammModel)) {
                    return (SAMMArchitectureModelProxy) proxy;
                }
            }
        }
        return null;
    }

    /**
     * Returns a list of all components (<code>AbstractComponent</code>) contained in a given
     * <code>AbstractArchitectureModel</code>.
     * 
     * @param architectureModel
     *            The <code>AbstractArchitectureModel</code> containing the requested components.
     */
    public List<AbstractModelElement> getComponents(AbstractArchitectureModel architectureModel) {
        List<CompositeComponent> rawComponentList = new LinkedList<CompositeComponent>();
        List<AbstractModelElement> componentProxyList = new LinkedList<AbstractModelElement>();
        // Note: now we retrieve all components in repository! We don't look for SubcomponentInstances but all componentTypes!
        
        Repository repository = null;
        if (architectureModel instanceof SAMMArchitectureModelProxy) {
            repository = ((SAMMArchitectureModelProxy) architectureModel).getRepository();
        }

        CompositeComponentCollectorSwitch gettingComponentsSwitch = new CompositeComponentCollectorSwitch(rawComponentList);
        if (repository != null) {
            TreeIterator<EObject> itr = repository.eAllContents();
            //Fills the list with <i>raw</i> components. <i>Raw</i> means, that the list
            //contains the original model elements.
            while (itr.hasNext()) {
                EObject next = itr.next();
                gettingComponentsSwitch.doSwitch(next);
            }           
            
            /*In the rawComponentList are just composite components. The following for-loop
            *walks through all composite components and tries to get the proxy. If the proxy
            *does not exist yet, then it is created.
            *For each composite component all subcomponents (which can be composite components
            *and primitive components) are created in the createSubcomponentInstances()-Method
            *recursively.
            **/
            for (CompositeComponent compositeComponent : rawComponentList) {
               componentProxyList.add(this.getSAMMProxyElement(compositeComponent));
               createSubcomponentInstances(compositeComponent, componentProxyList);
            }
        }
        //At this time the componentProxyList contains all components of the given
        //architecture alternative, including composite and primitive components.
        return componentProxyList;
     }
    
    /**
     * Creates for a given composite component the proxies for all it's subcomponents (recursively).  
     * @param compositeComponent
     * @param componentProxyList
     */
     private void createSubcomponentInstances(CompositeComponent compositeComponent, List<AbstractModelElement> componentProxyList){
    	 for(SubcomponentInstance subCompInstance : compositeComponent.getSubcomponents()){
    		 
    		 if(subCompInstance.getRealizedBy() instanceof CompositeComponent){
    			 createSubcomponentInstances((CompositeComponent)subCompInstance.getRealizedBy(), componentProxyList);
    		 }else{
    			 componentProxyList.add(this.getSAMMProxyElement(compositeComponent));
    		 }
    	 }
     }

    /**
     * Returns a list of all interfaces contained in a given <code>AbstractArchitectureModel</code>.
     * The list does not directly contain the interfaces, but the
     * <code>TransportInterfaceInformationContainer</code> wrapping some information regarding the
     * interface.
     * 
     * @param architectureModel
     *            The <code>AbstractArchitectureModel</code> containing the requested components.
     */
    public List<TransportInterfaceInformationContainer> getInterfaces(AbstractArchitectureModel architectureModel) {
        List<TransportInterfaceInformationContainer> interfaceList = new LinkedList<TransportInterfaceInformationContainer>();

        // Note: now we retrieve all interfaces in repository! We don't look for componentInstances but all componentTypes!

        ServiceArchitectureModel sam = null;
        Repository repository = null;
        if (architectureModel instanceof SAMMArchitectureModelProxy) {
            sam = ((SAMMArchitectureModelProxy) architectureModel).getServicearchitecturemodel();
            repository = ((SAMMArchitectureModelProxy) architectureModel).getRepository();
        }

        if (repository != null) {
            TreeIterator<EObject> itr = repository.eAllContents();
            while (itr.hasNext()) {

                EObject obj = itr.next();

                if (obj instanceof PrimitiveComponent) {
                    //ComponentType compType = ((SubcomponentInstance) obj).getRealizedBy();
                    ComponentType compType = (ComponentType)obj;

                    createOrUpdateTransportContainerForInterfacePortList(compType.getProvided(), interfaceList,
                            compType);

                    createOrUpdateTransportContainerForInterfacePortList(compType.getRequired(), interfaceList,
                            compType);

                }
            }
        }

        return interfaceList;
    }

    /**
     * This methods creates, if not already done, or updates a container containing an interface
     * port list. Helping method for
     * <code>getInterfaces(AbstractArchitectureModel architectureModel)</code>.
     */
    private void createOrUpdateTransportContainerForInterfacePortList(List<InterfacePort> interfacePortList,
            List<TransportInterfaceInformationContainer> interfaceList, ComponentType compType) {

        if (compType instanceof CompositeComponent) {
            return;
        }
        
        for (InterfacePort port : interfacePortList) {
            TransportInterfaceInformationContainer containerForCurrentInterfacePort = isAlreadyAdded(interfaceList,
                    (AbstractInterface) this.getSAMMProxyElement(port.getInterfaceType()));

            if (containerForCurrentInterfacePort == null) {
                containerForCurrentInterfacePort = new TransportInterfaceInformationContainer((AbstractInterface) this
                        .getSAMMProxyElement(port.getInterfaceType()), (AbstractComponent) this.getSAMMProxyElement(compType));
                interfaceList.add(containerForCurrentInterfacePort);
            }

            containerForCurrentInterfacePort.getInterfacePorts().add((AbstractInterfacePort) this.getSAMMProxyElement(port));
        }
    }

    /**
     * Returns a list of datatypes contained in the passed <code>AbstractArchitectureModel</code>.
     * 
     * @param architectureModel
     *            The <code>AbstractArchitectureModel</code> from which the datatypes should be
     *            returned.
     */
    public List<AbstractDatatype> getDatatypes(AbstractArchitectureModel architectureModel) {
        List<TransportInterfaceInformationContainer> interfaceList = getInterfaces(architectureModel);

        List<AbstractDatatype> datatypes = new LinkedList<AbstractDatatype>();

        for (TransportInterfaceInformationContainer interfaceInformationContainer : interfaceList) {
            if (interfaceInformationContainer.getIface() != null)
                getDataTypesFromInterface(interfaceInformationContainer.getIface(), datatypes);
        }

        return datatypes;
    }

    /**
     * This method searches for all datatypes (<code>AbstractDatatype</code>), which are used by a
     * special interface passed as a parameter.
     * 
     * @param iface
     *            The interface for which the used datatypes are requested.
     * @param datatypes
     *            The list for the datatyes. It is empty while being passed as a parameter and is
     *            filled by the method or returns empty, if there are no datatypes used by the
     *            interface.
     */
    private void getDataTypesFromInterface(AbstractInterface iface, List<AbstractDatatype> datatypes) {
        List<Operation> operations = ((SAMMInterfaceProxy) iface).getInterface().getSignatures();

        for (Operation operation : operations) {
            if (operation.getInput() != null)
                getDatatypesFromMessageType(operation.getInput(), datatypes);
            if (operation.getOutput() != null)
                getDatatypesFromMessageType(operation.getOutput(), datatypes);
        }
    }

    /**
     * Helping method for
     * <code>getDataTypesFromInterface(AbstractInterface iface, List<AbstractDatatype> datatypes)</code>
     * . Returns a list of all <code>AbstractDatatype</code>a contained in a
     * <code>MessageType</code>.
     * 
     * @param messageType
     *            Parameter containing the used datatypes.
     * @param datatypes
     *            Empty list for datatypes which will be filled if there are datatypes used.
     */
    private void getDatatypesFromMessageType(MessageType messageType, List<AbstractDatatype> datatypes) {

        for (Parameter parameter : messageType.getParameters()) {
            if (parameter.getType() != null) {
                AbstractDatatype proxy = (AbstractDatatype) this.getSAMMProxyElement(parameter.getType());
                if (!datatypes.contains(proxy))
                    datatypes.add(proxy);
            }
        }
    }

    /**
     * Loads an <code>ServiceArchitectureModel</code> from underlying resource set.
     * 
     * @param resourceSet
     *            The underlying set of sources containing the architecture models.
     * @param uriString
     *            A string describing the URI to the service architecture model.
     * @return Returns the service architecture model lying in the directory the URI points to or
     *         null if no model was found.
     */
    private static ServiceArchitectureModel loadSAMMModelsFromURI(ResourceSet resourceSet, URI resourceURI) {

        // TODO Introduce some exceptions for the case that no architecture model was found.

        Resource resource = loadResource(resourceSet, resourceURI);

        if (!resource.getContents().isEmpty()) {
            if (resource.getContents().get(0) instanceof ServiceArchitectureModel) {
                EcoreUtil.resolveAll(resource.getContents().get(0));
                return (ServiceArchitectureModel) resource.getContents().get(0);
            }
        }

        return null;

    }
    
    private static Repository loadSAMMRepositoryFromURI(ResourceSet resourceSet, URI resourceURI) {

        // TODO Introduce some exceptions for the case that no architecture model was found.

        Resource resource = loadResource(resourceSet, resourceURI);

        if (!resource.getContents().isEmpty()) {
            if (resource.getContents().get(0) instanceof Repository) {
                EcoreUtil.resolveAll(resource.getContents().get(0));
                return (Repository) resource.getContents().get(0);
            }
        }

        return null;

    }

    private static Resource loadResource(ResourceSet resourceSet, URI resourceURI) {
        Resource resource = null;

        // Map loadOptions = ((XMLResourceImpl)resource).getDefaultLoadOptions();
        Map loadOptions = resourceSet.getLoadOptions();
        loadOptions.put(XMLResource.OPTION_DEFER_ATTACHMENT, Boolean.TRUE);
        loadOptions.put(XMLResource.OPTION_DEFER_IDREF_RESOLUTION, Boolean.TRUE);
        loadOptions.put(XMLResource.OPTION_USE_DEPRECATED_METHODS, Boolean.TRUE);
        loadOptions.put(XMLResource.OPTION_USE_PARSER_POOL, new XMLParserPoolImpl());
        loadOptions.put(XMLResource.OPTION_USE_XML_NAME_TO_FEATURE_MAP, new HashMap());

        resource = resourceSet.getResource(resourceURI, true);

        ((ResourceImpl) resource).setIntrinsicIDToEObjectMap(new HashMap());
        return resource;
    }
    

    /**
     * Method which tests whether an <code>AbstractInterface</code> is already added to a given
     * list. Because the interfaces are wrapped by a
     * <code>TransportInterfaceInformationContainer</code>, <code>List.contains(...)</code> cannot
     * be used.
     * 
     * @param interfaceList
     *            The list to test whether a given container is already added or not.
     * @param currentInterface
     *            The interface to test.
     * @return
     */
    private static TransportInterfaceInformationContainer isAlreadyAdded(
            List<TransportInterfaceInformationContainer> interfaceList, AbstractInterface currentInterface) {
        for (TransportInterfaceInformationContainer iface : interfaceList) {
            if (iface.getIface().getId().equals(currentInterface.getId()))
                return iface;
        }

        return null;
    }

    /**
     * @param componenttype
     *            The <code>AbstractComponent</code> containing the interface ports to find out.
     * @return Returns a list of interface ports provided by a given component. May be empty if no
     *         interface ports were found.
     */
    @Override
    public List<AbstractInterfacePort> getProvidedInterfacePorts(AbstractComponent componenttype) {

        List<AbstractInterfacePort> result = new ArrayList<AbstractInterfacePort>();

        if (componenttype instanceof SAMMComponentProxy) {
            ComponentType component = ((SAMMComponentProxy) componenttype).getComponenttype();

            for (InterfacePort port : component.getProvided()) {
                result.add((AbstractInterfacePort) this.getSAMMProxyElement(port));
            }
        }
        return result;
    }

    /**
     * @param componenttype
     *            The <code>AbstractComponent</code> containing the interface ports to find out.
     * @return Returns a list of interface ports required by a given component. May be empty if no
     *         interface port was found.
     */
    @Override
    public List<AbstractInterfacePort> getRequiredInterfacePorts(AbstractComponent componenttype) {

        List<AbstractInterfacePort> result = new ArrayList<AbstractInterfacePort>();

        if (componenttype instanceof SAMMComponentProxy) {
            ComponentType component = ((SAMMComponentProxy) componenttype).getComponenttype();

            for (InterfacePort port : component.getRequired()) {
                result.add((AbstractInterfacePort) this.getSAMMProxyElement(port));
            }
        }

        return result;
    }

    /**
     * @param interfacePort
     *            The interface port containing the operations to find out.
     * @return Returns a list of <code>AbstractOperation</code>s contained in an
     *         <code>AbstractInterfacePort</code>. may be empty if no operation was found.
     */
    @Override
    public List<AbstractOperationImplementation> getOperationImplementations(AbstractInterfacePort interfacePort) {

        List<AbstractOperationImplementation> result = new ArrayList<AbstractOperationImplementation>();

        if (interfacePort instanceof SAMMInterfacePortProxy) {
            InterfacePort sammInterfacePort = ((SAMMInterfacePortProxy) interfacePort).getInterfaceport();
           
            if ((sammInterfacePort != null) && (sammInterfacePort.getInterfaceType() != null))
                for (Operation operation : sammInterfacePort.getInterfaceType().getSignatures()) {
                	result.add(this.getSAMMOperationImplementationProxy(interfacePort, operation));
                }
        }
        return result;
    }

    /**
     * @return Returns a list of <code>AbstractDatatype</code>s from a given
     *         <code>AbstractInterface</code> using
     *         <code>getDataTypesFromInterface(AbstractInterface iface, List<AbstractDatatype> datatypes)</code>
     *         .
     */
    public List<AbstractDatatype> getDataTypesOfInterface(AbstractInterface referencedInterface) {
        List<AbstractDatatype> result = new ArrayList<AbstractDatatype>();

        getDataTypesFromInterface(referencedInterface, result);

        return result;
    }
    
    public List<AbstractArchitectureModel> getAllParentModels(AbstractArchitectureModel selectedModel) {
        List<AbstractArchitectureModel> parentArchitectureModelList = new LinkedList<AbstractArchitectureModel>();

        IQModel correspondingIQModel = this.architectureModelToIQModelMap.get(selectedModel);
        if (correspondingIQModel != null) {
            IQElement correspondingIQElement = correspondingIQModel.getAlternative().getParent();

            if (correspondingIQElement instanceof IQRepository) {
                AbstractArchitectureModel zeroAbstractArchitectureModel = createZeroModel();
                // TODO: add the empty model to list, if it could be created properly
            } else {
                if (correspondingIQElement instanceof IQAlternative) {
                    parentArchitectureModelList.add(this.iQModelToArchitectureModelMap
                            .get(((IQAlternative) correspondingIQElement)));
                    getAllParentModels(this.iQModelToArchitectureModelMap.get(correspondingIQElement.getParent()));
                }
            }
        }
        return parentArchitectureModelList;
    }

    /**
     * TODO: Implement
     * @return
     */
    private AbstractArchitectureModel createZeroModel() {
        return null;
    }
    
    public AbstractModelElement getProxy(NamedEntity key){
    	if(this.proxyToAbstractModelElementMap.containsKey(key)){
    		return this.proxyToAbstractModelElementMap.get(key);
    	}else{
    		return null;
    	}
    }

	@Override
	public void assignComponentNestingToContainerStructure(
			CompositeComponentSelectionContainer compositeComponentSelectionContainer, CompositeTaskDerivationContainer workplanContainer) {
		//TODO: Implement Method so that nesting structure is assigned to container structure.		
	}
}
