package de.fzi.kamp.service.analysisinstance.impl;

import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.part.MultiPageEditorPart;

import de.fzi.kamp.derivation.WorkplanDerivationJob;
import de.fzi.kamp.service.analysisinstance.AlternativeAndChangeRequestTuple;
import de.fzi.kamp.service.analysisinstance.IAnalysisInstanceManager;
import de.fzi.kamp.service.architecturemodel.impl.BackboneModelLoader;
import de.fzi.kamp.service.commands.AddAnalysisInstanceCommand;
import de.fzi.kamp.service.commands.RemoveAnalysisInstanceCommand;
import de.fzi.kamp.service.general.AnalysisManager;
import de.fzi.kamp.service.general.IDialogManager;
import de.fzi.kamp.service.general.IWizardManager;
import de.fzi.kamp.service.maineditor.IMainEditor;
import de.fzi.kamp.service.workplanmanagement.ContainerFactory;
import de.fzi.kamp.service.workplanmanagement.IWorkPlanManager;
import de.fzi.kamp.service.workplanmanagement.NewWorkplanBuilder;
import de.fzi.kamp.service.workplanmanagement.WorkPlanManager;
import de.fzi.maintainabilitymodel.main.EffortAnalysisInstance;
import de.fzi.maintainabilitymodel.main.MainFactory;
import de.fzi.maintainabilitymodel.main.MaintainabilityAnalysisModel;
import de.fzi.maintainabilitymodel.workplan.ArchitecturalAlternative;
import de.fzi.maintainabilitymodel.workplan.ChangeRequest;
import de.fzi.maintainabilitymodel.workplan.CompositeTask;
import de.fzi.maintainabilitymodel.workplan.TaskRationale;
import de.fzi.maintainabilitymodel.workplan.Workplan;
import de.fzi.maintainabilitymodel.workplan.WorkplanFactory;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.CompositeTaskDerivationContainer;
import de.fzi.maintainabilitymodel.workplan.selectioncontainer.impl.SelectioncontainerFactoryImpl;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.ide.backbone.core.model.IQModel;

/**
 * A class to centralize all operations regarding the <code>EffortAnalysisInstance</code> objects in
 * the <code>MaintainabilityAnalysisModel</code>.
 * 
 * @author tknapp
 * 
 */
public class AnalysisInstanceManager implements IAnalysisInstanceManager {
    private final static Logger logger = Logger.getLogger(AnalysisInstanceManager.class);

    private MaintainabilityAnalysisModel analysisModel;

    private IDialogManager dialogManager;
    private IWizardManager wizardManager;
    private IMainEditor mainEditor;
    private IWorkPlanManager workplanManager;

    public AnalysisInstanceManager(MaintainabilityAnalysisModel analysisModel, IDialogManager dialogManager,
            IWizardManager wizardManager, IMainEditor mainEditor) {

        this.analysisModel = analysisModel;
        this.dialogManager = dialogManager;
        this.wizardManager = wizardManager;
        this.mainEditor = mainEditor;
    }

    /**
     * Adds a new <code>EffortAnalysisInstance</code> to the
     * <code>MaintainabilityAnalysisModel</code>.
     * 
     * @param analysisManager
     *            An <code>AnalysisManager</code> object to handle the
     *            <code>MaintainabilityAnalysisModel</code>.
     */
    public void addAnalysisInstance(AnalysisManager analysisManager) {

        if (analysisManager.getAnalysisModel().getChangerequests().isEmpty()) {
            wizardManager.showErrorMessage("Please first define at least one change scenario!", mainEditor.getMainEditorShell());
            return;
        }
        
        AlternativeAndChangeRequestTuple result = ((AnalysisInstanceManager) analysisManager
                .getAnalysisInstanceManager()).getDialogManager()
                .showSelectArchitectureAlternativeAndChangeRequestDialog(analysisManager.getAnalysisModel());

        if (result != null) {
            AddAnalysisInstanceCommand command = new AddAnalysisInstanceCommand(result, analysisManager);
            this.mainEditor.handleCommand(command);
        }
    }
    
    /**
     * Creates a new <code>EffortAnalysisInstance</code> object consisting of an
     * <code>ArchitecturalAlternative</code> and a <code>ChangeRequest</code> and inserts it into
     * the <code>MaintainabilityAnalysisModel</code>.
     * 
     * @param sourceArchitectureAlternative
     *            An architecture alternative.
     * @param changeRequest
     *            A change request.
     * @param analysisManager
     *            An <code>AnalysisManager</code> object to handle the
     *            <code>MaintainabilityAnalysisModel</code>.
     * @return Returns an <code>EffortAnalysisInstance</code> object consisting of the passed
     *         <code>ArchitecturalAlternative</code> and the <code>ChangeRequest</code>.
     */
    public EffortAnalysisInstance createAnalysisInstance(ArchitecturalAlternative sourceArchitectureAlternative,
            ChangeRequest changeRequest, AnalysisManager analysisManager, ArchitecturalAlternative targetArchitectureAlternative) {

        EffortAnalysisInstance analysisInstance = MainFactory.eINSTANCE.createEffortAnalysisInstance();

        if (targetArchitectureAlternative != null) {
            analysisInstance.setTargetArchitecturalAlternative(targetArchitectureAlternative);
        } else {
            logger.error("Target Alternative was null!");
        }

        if (sourceArchitectureAlternative != null)
            analysisInstance.setSourceArchitecturalAlternative(sourceArchitectureAlternative);
        
        analysisInstance.setChangerequest(changeRequest);

        this.analysisModel.getAnalysisinstances().add(analysisInstance);

        if (workplanManager == null) {
            this.workplanManager = new WorkPlanManager(this.dialogManager, analysisInstance);
        }
        
        createWorkplanRoot(analysisInstance);

        return analysisInstance;
    }

    

    /**
     * @return Return a list of EffortAnalysisInstances.
     */
    @Override
    public List<EffortAnalysisInstance> getAnalysisInstances() {
        return this.analysisModel.getAnalysisinstances();
    }

    /**
     * Delegates the order to remove the <code>EffortAnalysisInstance</code> passed as parameter to
     * a command.
     * 
     * @param analysisInstance
     *            The <code>EffortAnalysisInstance</code> to remove.
     * @param item
     *            The <code>TableItem</code> representing the <code>EffortAnalysisInstance</code> to
     *            remove. It is necessary to update representation.
     */
    @Override
    public void removeAnalysisInstance(EffortAnalysisInstance analysisInstance, TableItem item) {
        RemoveAnalysisInstanceCommand command = new RemoveAnalysisInstanceCommand(this, analysisInstance, item);
        this.mainEditor.handleCommand(command);
    }

    /**
     * Implements strategy whether the derive workplan button in the analysis instance table is
     * enabled or not.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledDeriveWorkPlanForAnalysisInstance(EffortAnalysisInstance analysisInstance) {

        // TODO: decide strategy!
        if (analysisInstance.getWorkplan() == null
                || analysisInstance.getWorkplan().getCompositetaskderivationcontainer().isEmpty()
                || analysisInstance.getWorkplan().getCompositetaskderivationcontainer().get(0)
                        .getTopLevelActivityContainer().isEmpty()) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Delegates the order to begin the drive process to a command.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     * @param editor
     *            The main editor necessary to execute the command.
     */
    @Override
    public void deriveWorkplanForAnalysisInstance(EffortAnalysisInstance analysisInstance, IMainEditor editor) {
        if (analysisInstance.getChangerequest().isAutomaticDerivation()) {
            logger.info("Run Workplan Derivation Job");
            createAndRunWorkplanDerivationJob(analysisInstance, editor);
        } else {
            logger.info("Run Workplan Derivation Wizard");
            runDerivationWizard(analysisInstance, editor);
        }
    }

    private void runDerivationWizard(EffortAnalysisInstance analysisInstance, IMainEditor editor) {
        this.wizardManager.showAddCompositeActivityWizard(analysisInstance, ((MultiPageEditorPart) editor).getSite()
                .getShell(), editor, editor.getArchitectureModelProvider());
    }

    private void createAndRunWorkplanDerivationJob(EffortAnalysisInstance analysisInstance, IMainEditor editor) {
        
        if (analysisInstance.getSourceArchitecturalAlternative() == null) {
            MessageBox messageBox =   new MessageBox(editor.getMainEditorShell(), SWT.OK|SWT.CANCEL);
            messageBox.setMessage("Source Architecture Alternative was null: Try to open KAMP Editor from Project Explorer and recreate AnalysisInstance!");
            if (messageBox.open() == SWT.OK){  };
            return;
        }
        
        IQModel sourceIQModel = BackboneModelLoader.findIQModelForAlternativeName(analysisInstance.getSourceArchitecturalAlternative().getName(), editor);
        IQModel targetIQModel = BackboneModelLoader.findIQModelForAlternativeName(analysisInstance.getTargetArchitecturalAlternative().getName(), editor);
        
        WorkplanDerivationJob job = new WorkplanDerivationJob(
                sourceIQModel.getAlternative(), targetIQModel.getAlternative(), analysisInstance, 
                editor.getArchitectureModelProvider(), new ContainerFactory(), new NewWorkplanBuilder(this.mainEditor));
        
        try {
            job.execute(null);
        } catch (JobFailedException e) {
            e.printStackTrace();
        } catch (UserCanceledException e) {
            e.printStackTrace();
        }
    }

    /**
     * Implements strategy whether the edit workplan button for selected analysis instance in the
     * analysis instance table is enabled or not.
     * 
     * @param The
     *            relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledEditWorkPlanForAnalysisInstance(EffortAnalysisInstance analysisInstance) {
        if (analysisInstance.getWorkplan() == null) {
//                || analysisInstance.getWorkplan().getCompositetaskderivationcontainer().isEmpty()
//                || analysisInstance.getWorkplan().getCompositetaskderivationcontainer().get(0)
//                        .getTopLevelActivityContainer().isEmpty()) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Delegates the order to edit the workplan for the passed <code>EffortAnalysisInstance</code>.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     * @param editor
     *            The main editor necessary to execute the command.
     */
    @Override
    public void editWorkPlanForAnalysisInstance(EffortAnalysisInstance analysisInstance, IMainEditor editor) {

        if (analysisInstance.getWorkplan() != null) {
            editor.showWorkplanChanges(analysisInstance.getWorkplan(), false);
        }
    }

    /**
     * Strategy to decide whether the calculate workplan complexity button is enabled or not
     * (PROVISIONAL).
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledCalculateWorkComplexityForAnalysisInstance(EffortAnalysisInstance analysisInstance) {
        return analysisInstance.getWorkplan() != null;
    }

    /**
     * The work complexity button is not implemented at the moment.
     */
    @Override
    public void calculateWorkComplexityForAnalysisInstance(EffortAnalysisInstance analysisInstance) {

        // TODO: insert process for work complexity calculation
    }

    /**
     * Strategy to decide whether the button to start effort estimation interview is enabled
     * (PROVISIONAL).
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledStartEffortEstimationInterview(EffortAnalysisInstance analysisInstance) {
        return (analysisInstance.getWorkplan() != null && !analysisInstance.getWorkplan().getTasks().isEmpty());
    }

    /**
     * Interview button is permanently disabled at the moment. Has to be implemented.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public void startEffortEstimationInterview(EffortAnalysisInstance analysisInstance) {
        // TODO Implement strategy.
    }

    /**
     * Strategy to decide whether the button to switch to the page to edit the effort estimates is
     * enabled.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledEditEffortEstimates(EffortAnalysisInstance analysisInstance) {
        return (analysisInstance.getWorkplan() != null && !analysisInstance.getWorkplan().getTasks().isEmpty());
    }

    /**
     * Delegates the order to switch to the page to edit the effort estimates to a command.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     * @param editor
     *            The main editor necessary to execute the command.
     */
    @Override
    public void editEffortEstimates(EffortAnalysisInstance analysisInstance, IMainEditor editor) {

        if (analysisInstance.getWorkplan() != null) {
            editor.showWorkplanChanges(analysisInstance.getWorkplan(), true);
        }
    }

    /**
     * Strategy to decide whether the show result summary button is enabled (PROVISIONAL).
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public boolean isEnabledShowResultSummary(EffortAnalysisInstance analysisInstance) {
        return (analysisInstance.getWorkplan() != null && !analysisInstance.getWorkplan().getTasks().isEmpty());
    }

    /**
     * At the moment the show result summary button is not visible. As a consequence no delegation
     * is necessary.
     * 
     * @param analysisInstance
     *            The relevant <code>EffortAnalysisInstance</code>.
     */
    @Override
    public void showResultSummary(EffortAnalysisInstance instance) {

        // TODO: show view with result summary
    }

    public MaintainabilityAnalysisModel getAnalysisModel() {
        return analysisModel;
    }

    public IDialogManager getDialogManager() {
        return dialogManager;
    }

    public IWizardManager getWizardManager() {
        return wizardManager;
    }

    public IWorkPlanManager getWorkplanManager() {
        return workplanManager;
    }
    
    /**
     * In this method the routine to create all containers wrapping the architecture elements and
     * their corresponding graphical elements is started.
     * 
     * @param architectureModelProvider
     */
    private void createWorkplanRoot(EffortAnalysisInstance analysisInstance) {
    	Workplan workplan = analysisInstance.getWorkplan();
    	if (workplan == null) {
            workplan = WorkplanFactory.eINSTANCE.createWorkplan();
            analysisInstance.setWorkplan(workplan);
        }
    	
    	CompositeTask workplanActivity = WorkplanFactory.eINSTANCE.createCompositeTask(); 
        CompositeTaskDerivationContainer compositeTaskContainer = SelectioncontainerFactoryImpl.eINSTANCE.createCompositeTaskDerivationContainer();
        
        compositeTaskContainer.setActivity(workplanActivity);
        workplanActivity.setSelectioncontainer(compositeTaskContainer);
        
        TaskRationale workplanRootTaskTationale = createTaskRationale();
        workplanActivity.setTaskrationale(workplanRootTaskTationale);
                
        workplan.getCompositetaskderivationcontainer().add(compositeTaskContainer);
        workplan.getTasks().add(workplanActivity);
    }
    
    private TaskRationale createTaskRationale(){
    	TaskRationale rationale = WorkplanFactory.eINSTANCE.createTaskRationale();
        rationale.setDescription("");
        rationale.setKeyword("Complete Workplan");
        
        return rationale;
    }
}
