package de.fzi.kamp.derivation;

import java.io.IOException;
import java.util.Collections;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.compare.diff.metamodel.DiffModel;
import org.eclipse.emf.compare.diff.service.DiffService;
import org.eclipse.emf.compare.match.metamodel.MatchModel;
import org.eclipse.emf.compare.match.service.MatchService;
import org.eclipse.emf.compare.util.ModelUtils;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;

import de.fzi.maintainabilitymodel.main.EffortAnalysisInstance;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;

public class WorkplanDerivationJob implements IJob {

    private final static Logger logger = Logger.getLogger(WorkplanDerivationJob.class);

    public final static String SAMM_SERVICEARCHITECTUREMODEL = "samm_servicearchitecturemodel";
    public final static String SAMM_REPOSITORY = "samm_repository";
    public final static String SAMM_SEFF = "samm_seff";
    public final static String SAMM_HARDWARE = "samm_hardware";
    public final static String SAMM_QOSANNOTATION = "samm_qosannotation";
    public final static String SAMM_USAGEMODEL = "samm_usagemodel";
    public final static String SAMM_TARGETENVIRONMENT = "samm_targetenvironment";

    private IQAlternative sourceAlternative;
    private IQAlternative targetAlternative;
    private EffortAnalysisInstance analysisInstance;
    private IArchitectureModelResolver architectureModelResolver;
    private IContainerFactory containerFactory;
    private IWorkplanBuilderResolver builderSolver;

    public WorkplanDerivationJob(IQAlternative sourceAlternative, IQAlternative targetAlternative,
            EffortAnalysisInstance analysisInstance, IArchitectureModelResolver architectureModelResolver,
            IContainerFactory containerFactory, IWorkplanBuilderResolver builderSolver) {
        super();
        this.sourceAlternative = sourceAlternative;
        this.targetAlternative = targetAlternative;
        this.analysisInstance = analysisInstance;
        this.architectureModelResolver = architectureModelResolver;
        this.containerFactory = containerFactory;
        this.builderSolver = builderSolver;
    }

    @Override
    public String getName() {
        return "KAMP Workplan Derivation";
    }

    @Override
    public void rollback(IProgressMonitor monitor) throws RollbackFailedException {
    }

    @Override
    public void execute(IProgressMonitor monitor) throws JobFailedException, UserCanceledException {

        if (this.sourceAlternative == null) {
            logger.error("SourceAlternative was null");
            return;
        }

        if (this.targetAlternative == null) {
            logger.error("TargetAlternative was null");
            return;
        }

        try {
            logger.info("Run KAMP Workplan Derivation!");
            deriveWorkplanForAlternatives(this.sourceAlternative, this.targetAlternative);
        } catch (Exception e) {
            throw new JobFailedException(e.getLocalizedMessage(), e);
        }
    }

    private void deriveWorkplanForAlternatives(IQAlternative sourceAlternative, IQAlternative targetAlternative) {

        String[] analysedModelTypes = new String[] { SAMM_REPOSITORY }; // ,
                                                                        // SAMM_SERVICEARCHITECTUREMODEL,
                                                                        // SAMM_SEFF};

        for (String modelType : analysedModelTypes) {
            logger.info("Analysing model type: " + modelType);
            IQModel targetModel = getModelOfType(targetAlternative, modelType);
            IQModel sourceModel = getModelOfType(sourceAlternative, modelType);

            if ((targetModel != null) && (sourceModel != null)) {
                matchAndDiffModels(sourceModel, targetModel, modelType);
            }
        }
    }

    private void matchAndDiffModels(IQModel sourceModel, IQModel targetModel, String modelType) {

        IResource sourceResource = sourceModel.getCorrespondingResource();
        IResource targetResource = targetModel.getCorrespondingResource();

        ResourceSet resourceSet = new ResourceSetImpl();

        EObject sourceEObject = null;
        EObject targetEObject = null;
        try {
            sourceEObject = ModelUtils.load(createURI(sourceResource), resourceSet);
            targetEObject = ModelUtils.load(createURI(targetResource), resourceSet);
        } catch (IOException e) {
            throw new RuntimeException("Load of SourceModel or TargetModel failed", e);
        }

        if ((sourceEObject != null) && (targetEObject != null)) {
            MatchModel match = calculateMatch(sourceEObject, targetEObject);

            if (match != null) {
                DiffModel diff = DiffService.doDiff(match);

                String filePath = targetResource.getFullPath().removeLastSegments(1).addTrailingSeparator().toString()
                        + modelType + ".emfdiff";

                saveEmfModelToResource(diff, filePath, resourceSet);

                examineDiffModel(diff);
            }
        }
    }

    private DiffWorkplanBuilder workplanBuilder;

    private void examineDiffModel(DiffModel diff) {

        initializeWorkplanBuilder(diff);
        this.workplanBuilder.buildWorkplanFromDiffResult();
    }

    private void initializeWorkplanBuilder(DiffModel diff) {

        if (this.workplanBuilder == null) {
            this.workplanBuilder = new DiffWorkplanBuilder(diff, this.analysisInstance, this.architectureModelResolver,
                    this.containerFactory, this.builderSolver);
        }
    }

    private URI createURI(IResource sourceResource) {
        String fullpath = sourceResource.getFullPath().toString();
        URI uri = URI.createPlatformResourceURI(fullpath, false);
        return uri;
    }

    private MatchModel calculateMatch(EObject sourceEObject, EObject targetEObject) {
        MatchModel match = null;
        try {
            match = MatchService.doMatch(targetEObject, sourceEObject, null);
        } catch (InterruptedException e) {
            throw new RuntimeException("Matching was interrupted", e);
        }
        return match;
    }

    private IQModel getModelOfType(IQAlternative alternative, String modelType) {
        IQModel[] models = alternative.getModels();

        for (IQModel model : models) {
            if (model.getType().equals(modelType)) {
                return model;
            }
        }

        return null;
    }

    private IQAlternative getAlternativeFromSelection(ISelection selection) {
        if (selection != null) {
            if (selection instanceof TreeSelection) {
                Object firstElement = ((TreeSelection) selection).getFirstElement();

                if ((firstElement != null) && (firstElement instanceof IQAlternative)) {
                    return (IQAlternative) firstElement;
                }
            }
        }

        return null;
    }

    private IQAlternative determineParentAlternative(IQAlternative selectedAlternative) {
        if (selectedAlternative != null) {
            IQElement parentElement = selectedAlternative.getParent();

            if ((parentElement != null) && (parentElement instanceof IQAlternative)) {
                return (IQAlternative) parentElement;
            }
        }

        return null;
    }

    public static void saveEmfModelToResource(EObject model, String filePath, ResourceSet resourceSet) {
        if (filePath != null) {
            URI saveURI = URI.createFileURI(filePath);
            Resource resource = resourceSet.createResource(saveURI);

            resource.getContents().add(model);

            try {
                resource.save(Collections.EMPTY_MAP);
            } catch (IOException e) {
                throw new RuntimeException("Saving of resource failed", e);
            } finally {

            }
        }
    }

}
