package org.cocome.tradingsystem.testdriver;

import java.io.Serializable;
import java.util.concurrent.TimeoutException;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.TopicSubscriber;

import org.cocome.tradingsystem.systests.interfaces.IUpdateReceiver;

/**
 * This is the base for all classes implementing the IUpdateReceiver. The idea
 * is to listen on a specified channel and keep only messages of certain types.
 * 
 * @author Benjamin Hummel
 * @author $Author: hummel $
 * @version $Rev: 63 $
 * @levd.rating GREEN Rev: 63
 */
public class UpdateReceiver implements IUpdateReceiver, MessageListener {

	/** The last message (of those we are interested in) received. */
	private Serializable lastRelevantMessage = null;

	/**
	 * A flag indicating if this is a new message or if this has been read
	 * before.
	 */
	private boolean isNewMessage = false;

	/** The array of message types we are interested in. */
	private final Class<?>[] relevantMessages;

	/**
	 * Creates a new update receiver.
	 * 
	 * @param subscriber
	 *            the subscriber to get the messages from.
	 * @param relevantMessageClasses
	 *            the classes of messages we are interested in.
	 */
	public UpdateReceiver(TopicSubscriber subscriber,
			Class<?>... relevantMessageClasses) throws JMSException {
		subscriber.setMessageListener(this);
		this.relevantMessages = relevantMessageClasses;
	}

	/** {@inheritDoc} */
	public synchronized void waitForUpdate(int maxMilliseconds)
			throws TimeoutException {

		try {
			// multiplying by 10 makes the tests slightly slower, however I have
			// a chance of getting them work on my system (even with so many
			// processes running).
			this.wait(10 * maxMilliseconds);
		} catch (InterruptedException e) {
			// should not happen
		}

		if (!isNewMessage) {
			throw new TimeoutException();
		}
	}

	/**
	 * Signals that the next expected message should be of the given type. This
	 * checks if such a message is available, and waits again for a short time
	 * to get the chance of receiving the missing message. This is mostly used
	 * to circumvent situations where two messages are sent at the (nearly) same
	 * time.
	 * 
	 * @param messageType
	 *            the type of the message expected.
	 */
	protected void expectMessage(Class<? extends Serializable> messageType) {
		if (lastRelevantMessage == null) {
			return;
		}

		// This could lead to an infinite loop, if there are too many messages.
		// However in the cash desk (where this is used) the
		// number of messages is small.
		while (!messageType.isAssignableFrom(lastRelevantMessage.getClass())) {
			// throw away old message
			isNewMessage = false;
			try {
				waitForUpdate(100);
			} catch (TimeoutException e) {
				// there are no more messages, so stop now
				return;
			}
		}
	}

	/** Returns the last relevant message received. */
	protected synchronized Serializable getLastRelevantMessage() {
		isNewMessage = false;
		return lastRelevantMessage;
	}

	/** {@inheritDoc} */
	public synchronized void onMessage(Message message) {

		if (!(message instanceof ObjectMessage)) {
			return;
		}

		ObjectMessage objMessage = (ObjectMessage) message;
		Serializable eventObject;
		try {
			eventObject = objMessage.getObject();
		} catch (JMSException e) {
			// nothing much we can do here
			return;
		}

		for (Class<?> relevantMessage : relevantMessages) {
			if (relevantMessage.isAssignableFrom(eventObject.getClass())) {
				isNewMessage = true;
				lastRelevantMessage = eventObject;
				this.notify();
				return;
			}
		}
	}
}
