package org.cocome.tradingsystem.testdriver;

import javax.jms.JMSException;
import javax.naming.NamingException;

import org.cocome.tradingsystem.systests.interfaces.IBank;
import org.cocome.tradingsystem.systests.interfaces.ICashDesk;
import org.cocome.tradingsystem.systests.interfaces.IEnterprise;
import org.cocome.tradingsystem.systests.interfaces.IStorePC;
import org.cocome.tradingsystem.systests.interfaces.ITestDriver;

/**
 * This is the test driver for the reference implementation of the system.
 * 
 * @author Benjamin Hummel
 * @author $Author: hummel $
 * @version $Rev: 63 $
 * @levd.rating GREEN Rev: 63
 */
public class TestDriver implements ITestDriver {

	/** The bank we are using. */
	private Bank bank;

	/** The next used store id (= the number of stores created so far). */
	private int currentStore = 0;

	/**
	 * The maximal number of stores supported (as prepared in the properties
	 * files).
	 */
	private static final int MAX_STORES = 3;

	/** The preloaded store PCs. */
	private StorePC[] storePCs = new StorePC[MAX_STORES];

	/**
	 * For each store the next used cash desk id (= number of cash desks of the
	 * store).
	 */
	private final int[] currentCashdesk = new int[MAX_STORES];

	/**
	 * The maximal number of cash desks supported (as prepared in the properties
	 * files).
	 */
	private static final int MAX_CASHDESKS = 3;

	/** The enterprise. */
	private Enterprise enterprise = null;

	/** {@inheritDoc} */
	public ICashDesk createCashDesk(IStorePC store) throws NamingException,
			JMSException {
		StorePC storePC = (StorePC) store;
		int storeIndex = storePC.getIndex();
		if (storeIndex > 0) {
			throw new IllegalStateException(
					"Cash desks are only available for the first store (0)!");
		}
		if (currentCashdesk[storeIndex] >= MAX_CASHDESKS) {
			throw new IllegalStateException(
					"Maximal number of cash desks for store " + storeIndex
							+ " already created!");
		}
		return new CashDesk(storePC, currentCashdesk[storeIndex]++);
	}

	/** {@inheritDoc} */
	public IStorePC createStore() {
		if (currentStore >= MAX_STORES) {
			throw new IllegalStateException(
					"Maximal number of stores already created!");
		}
		return storePCs[currentStore++];
	}

	/** {@inheritDoc} */
	public IEnterprise initializeSystem() throws Exception {
		enterprise = new Enterprise();
		bank = new Bank();

		// precreate stores (so they always get the same store id)
		for (int i = 0; i < MAX_STORES; ++i) {
			storePCs[i] = new StorePC(i, enterprise);
		}

		return enterprise;
	}

	/** {@inheritDoc} */
	public void shutdownSystem() {
		enterprise = null;
		try {
			bank.unregister();
		} catch (Exception e) {
			// nothing we could handle here
		}
		bank = null;
	}

	/** {@inheritDoc} */
	public IBank getBank() throws Exception {
		return bank;
	}
}
