package org.cocome.tradingsystem.testdriver;

import java.io.Serializable;

import javax.jms.JMSException;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;

import org.cocome.tradingsystem.cashdeskline.datatypes.KeyStroke;
import org.cocome.tradingsystem.cashdeskline.datatypes.PaymentMode;
import org.cocome.tradingsystem.cashdeskline.events.CashAmountEnteredEvent;
import org.cocome.tradingsystem.cashdeskline.events.CashBoxClosedEvent;
import org.cocome.tradingsystem.cashdeskline.events.ChangeAmountCalculatedEvent;
import org.cocome.tradingsystem.cashdeskline.events.CreditCardPaymentEnabledEvent;
import org.cocome.tradingsystem.cashdeskline.events.PaymentModeEvent;
import org.cocome.tradingsystem.cashdeskline.events.SaleFinishedEvent;
import org.cocome.tradingsystem.cashdeskline.events.SaleStartedEvent;
import org.cocome.tradingsystem.systests.interfaces.ICashBox;

/**
 * Glue code for the cash box.
 * 
 * @author Benjamin Hummel
 * @author $Author: hummel $
 * @version $Rev: 63 $
 * @levd.rating GREEN Rev: 63
 */
public class CashBox extends UpdateReceiver implements ICashBox {

	/** Publisher for sending events. */
	private final TopicPublisher publisher;

	/** Session used for creating object messages. */
	private final TopicSession session;

	/** Constructor. */
	public CashBox(TopicPublisher publisher, TopicSession session,
			TopicSubscriber subscriber) throws JMSException {
		super(subscriber, ChangeAmountCalculatedEvent.class);
		this.publisher = publisher;
		this.session = session;
	}

	/** Publish a serializable message. */
	private void publish(Serializable message) throws JMSException {
		publisher.publish(session.createObjectMessage(message));
	}

	/** {@inheritDoc} */
	public void enterReceivedCash(int centAmount) throws JMSException {
		// we have to generate multiple events for this one
		if (centAmount >= 100) {
			for (char c : Integer.toString(centAmount / 100).toCharArray()) {
				publish(new CashAmountEnteredEvent(charToStroke(c)));
			}
		}

		publish(new CashAmountEnteredEvent(KeyStroke.COMMA));

		String cents = Integer.toString(centAmount % 100);
		if (cents.length() < 2) {
			cents = "0" + cents;
		}
		for (char c : cents.toCharArray()) {
			publish(new CashAmountEnteredEvent(charToStroke(c)));
		}

		publish(new CashAmountEnteredEvent(KeyStroke.ENTER));
	}

	/** Convert characters to keystrokes. */
	private KeyStroke charToStroke(char c) {
		switch (c) {
		case '0':
			return KeyStroke.ZERO;
		case '1':
			return KeyStroke.ONE;
		case '2':
			return KeyStroke.TWO;
		case '3':
			return KeyStroke.THREE;
		case '4':
			return KeyStroke.FOUR;
		case '5':
			return KeyStroke.FIVE;
		case '6':
			return KeyStroke.SIX;
		case '7':
			return KeyStroke.SEVEN;
		case '8':
			return KeyStroke.EIGHT;
		case '9':
			return KeyStroke.NINE;
		default:
			throw new IllegalArgumentException("Only number supported!");
		}
	}

	/** {@inheritDoc} */
	public void finishSale() throws JMSException {
		publish(new SaleFinishedEvent());
	}

	/** {@inheritDoc} */
	public void setCashboxStatus(boolean closed) throws JMSException {
		if (closed) {
			publish(new CashBoxClosedEvent());
		} else {
			// this signal os not used by the implementation
		}
	}

	/** {@inheritDoc} */
	public void startCashPayment() throws JMSException {
		publish(new PaymentModeEvent(PaymentMode.CASH));
	}

	/** {@inheritDoc} */
	public void startCreditCardPayment() throws JMSException {
		publish(new PaymentModeEvent(PaymentMode.CREDITCARD));
	}

	/** {@inheritDoc} */
	public void startNewSale() throws JMSException {
		publish(new SaleStartedEvent());
	}

	/** {@inheritDoc} */
	public boolean wasOpenSignalSent() {
		// this is slightly bogus, but this is how it is implemented in the GUI
		return getLastRelevantMessage() instanceof ChangeAmountCalculatedEvent;
	}

	/** {@inheritDoc} */
	public void manuallyEnableCreditCardReader() throws JMSException {
		publish(new CreditCardPaymentEnabledEvent());
	}
}
