package org.cocome.tradingsystem.testdriver;

import java.rmi.AccessException;
import java.rmi.NotBoundException;
import java.rmi.RMISecurityManager;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.util.HashMap;
import java.util.Map;

import org.cocome.tradingsystem.external.Debit;
import org.cocome.tradingsystem.external.TransactionID;
import org.cocome.tradingsystem.systests.interfaces.IBank;

/**
 * Implementation of a bank that we can control (as opposed to the bank coming
 * with the implementation).
 * 
 * @author Benjamin Hummel
 * @author $Author: hummel $
 * @version $Rev: 63 $
 * @levd.rating GREEN Rev: 63
 */
public class Bank extends UnicastRemoteObject implements IBank,
		org.cocome.tradingsystem.external.Bank {

	/** The offset added to create a transaction ID from a card number. */
	private static final int TID_OFFSET = 42;

	/** Create a new Bank. */
	protected Bank() throws RemoteException {
		super();
		register();
	}

	/** ID for serialization in the RMI context. */
	private static final long serialVersionUID = 1556930771225198542L;

	/** All available credit cards. */
	private final Map<Integer, CreditCard> creditCards = new HashMap<Integer, CreditCard>();

	/** Register at RMI. */
	private void register() throws AccessException, RemoteException {
		if (System.getSecurityManager() == null) {
			System.setSecurityManager(new RMISecurityManager());
		}
		Registry reg = LocateRegistry.getRegistry("localhost", 1098);
		reg.rebind("Bank", this);
	}

	/** Unregister from RMI. */
	public void unregister() throws AccessException, RemoteException,
			NotBoundException {
		Registry reg = LocateRegistry.getRegistry("localhost", 1098);
		reg.unbind("Bank");
	}

	/** {@inheritDoc} */
	public void createCreditCard(int cardNumber, int pinNumber,
			int availableMoney) {
		creditCards.put(cardNumber, new CreditCard(pinNumber, availableMoney));
	}

	/** {@inheritDoc} */
	public void deleteCreditCard(int cardNumber) {
		creditCards.remove(cardNumber);
	}

	/** {@inheritDoc} */
	public int getAvailableMoney(int cardNumber) {
		return creditCards.get(cardNumber).money;
	}

	/** {@inheritDoc} */
	public Debit debitCard(TransactionID id) throws RemoteException {
		int number = findCardFromTID(id);
		if (number < 0) {
			return Debit.TRANSACTION_ID_NOT_VALID;
		}

		CreditCard cc = creditCards.get(number);
		if (cc == null) {
			return Debit.TRANSACTION_ID_NOT_VALID;
		}

		// We have to use a simple approach here, as the reference
		// implementation does not care about the amount of money
		if (cc.money > 0) {
			return Debit.OK;
		}
		return Debit.NOT_ENOUGH_MONEY;
	}

	/**
	 * Extract the card number from a transaction ID. This is not an efficient
	 * method, but the TransactionID class does not provide any access to its
	 * internals and also does not implement hashCode (although it should, when
	 * overriding equals).
	 * 
	 * @return the number of the credit card contained in this TID, or -1 if no
	 *         matching card was found.
	 */
	private int findCardFromTID(TransactionID id) {
		for (int ccNumber : creditCards.keySet()) {
			if (new TransactionID(ccNumber + TID_OFFSET).equals(id)) {
				return ccNumber;
			}
		}
		return -1;
	}

	/** {@inheritDoc} */
	public TransactionID validateCard(String cardInformation, int pinnumber)
			throws RemoteException {
		int cardNumber;
		try {
			cardNumber = Integer.parseInt(cardInformation);
		} catch (NumberFormatException e) {
			return null;
		}

		CreditCard cc = creditCards.get(cardNumber);
		if (cc == null || cc.pin != pinnumber) {
			return null;
		}

		// we use the card number plus an offset for the transaction ID (not
		// perfect, but works in this context)
		return new TransactionID(cardNumber + TID_OFFSET);
	}

	/** Data storage for credit card. */
	private static final class CreditCard {
		/** The pin code of the card. */
		private final int pin;

		/** The available money for the card. */
		private int money;

		/** Create a new card. */
		public CreditCard(int pinNumber, int availableMoney) {
			this.pin = pinNumber;
			this.money = availableMoney;
		}
	}
}
