package org.cocome.tradingsystem.systests.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.cocome.tradingsystem.systests.interfaces.IProduct;
import org.cocome.tradingsystem.systests.interfaces.IStorePC;

/**
 * This class is used to generate (reproducably) random stock items.
 * 
 * @author Benjamin Hummel
 * @author Christian Pfaller
 * @author $Author: pfaller $
 * @version $Rev: 60 $
 * @levd.rating GREEN Rev: 60
 */
public class StockGenerator {

	/** The random number generator used. */
	private final Random rng;

	/** The storePC used for adding stock. */
	private final IStorePC storePC;

	/** The product generator containing all products we may refer to. */
	private final ProductGenerator productGenerator;

	/** A list of all stock items generated so far. */
	private final List<GeneratedStockItem> generatedStock = new ArrayList<GeneratedStockItem>();

	/**
	 * Construct a new stock generator.
	 * 
	 * @param randomSeed
	 *            the random seed used for initializing the random number
	 *            generator.
	 * @param storePC
	 *            the store used for generating stock.
	 * @param productGenerator
	 *            the product generator containing all products we may refer to.
	 */
	public StockGenerator(long randomSeed, IStorePC storePC,
			ProductGenerator productGenerator) {
		rng = new Random(randomSeed);
		this.storePC = storePC;
		this.productGenerator = productGenerator;
	}

	/** Returns the number of stock items generated so far. */
	public int getNumberOfStockItems() {
		return generatedStock.size();
	}

	/**
	 * Returns the stock item generated when <code>i</code> stock items had
	 * been generated already.
	 */
	public GeneratedStockItem getGeneratedStockItem(int i) {
		return generatedStock.get(i);
	}

	/** Generate stock items for all known products. */
	public void generateAll() throws Exception {
		for (int i = generatedStock.size(); i < productGenerator
				.getNumberOfProducts(); ++i) {
			generate(i);
		}
	}
	

	/**
	 * Generate a single random stock item from the product of index
	 * <code>i</code> in the product generator.
	 */
	private void generate(int productIndex) throws Exception {

		int minAmount = 1 + rng.nextInt(50);
		int amount = minAmount + rng.nextInt(1000);
		int maxAmount = amount + rng.nextInt(1000);

		generate(productIndex, minAmount, amount, maxAmount);
	}
	
	/**
	 * Generate a single random stock item from the product of index
	 * <code>i</code> in the product generator with the given amounts in stock.
	 */
	public void generate(int productIndex, int minAmount, int amount, int maxAmount) throws Exception {
		GeneratedProduct gp = productGenerator
				.getGeneratedProduct(productIndex);

		IProduct product = gp.getProduct();

		// prices may at most triple
		int salesPrice = (int) (gp.getPurchasePrice() * (1. + 2. * rng
				.nextDouble()));


		storePC.insertStockItem(product, salesPrice, amount, minAmount);
		generatedStock.add(new GeneratedStockItem(gp, salesPrice, amount,
				minAmount, maxAmount));
	}
}
