package org.cocome.tradingsystem.systests.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import org.cocome.tradingsystem.systests.interfaces.IEnterprise;
import org.cocome.tradingsystem.systests.interfaces.IProduct;

/**
 * This class is used to generate random but reproducable products.
 * 
 * @author Benjamin Hummel
 * @author Christian Pfaller
 * @author $Author: pfaller $
 * @version $Rev: 60 $
 * @levd.rating GREEN Rev: 60
 */
public class ProductGenerator {

	/** Available alphabet for product name generation. */
	private static final String ALPHA = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.-_";

	/** The random number generator used. */
	private final Random rng;

	/** The enterprise used for creating products. */
	private final IEnterprise enterprise;

	/** The set of used barcodes to avoid repetition. */
	private final Set<Integer> usedBarcodes = new HashSet<Integer>();

	/** The set of used names to avoid repetition. */
	private final Set<String> usedNames = new HashSet<String>();

	/** A list of all products generated so far. */
	private final List<GeneratedProduct> generatedProducts = new ArrayList<GeneratedProduct>();

	/**
	 * Construct a new product generator.
	 * 
	 * @param randomSeed
	 *            the random seed used for initializing the random number
	 *            generator.
	 * @param enterprise
	 *            the enterprise used for creating products.
	 */
	public ProductGenerator(long randomSeed, IEnterprise enterprise) {
		rng = new Random(randomSeed);
		this.enterprise = enterprise;
	}

	/** Returns the number of products generated so far. */
	public int getNumberOfProducts() {
		return generatedProducts.size();
	}

	/**
	 * Returns the product generated when <code>i</code> products had been
	 * generated already.
	 */
	public GeneratedProduct getGeneratedProduct(int i) {
		return generatedProducts.get(i);
	}

	/** Generate the given number of random products. */
	public void generate(int n) throws Exception {
		for (int i = 0; i < n; ++i) {
			generate();
		}
	}
	

	/** Generate a single random product. 
	 * @return index of new product
	 * @throws Exception
	 */
	public int generate() throws Exception {

		int barcode = getUnusedBarcode();
		int purchasePrice = rng.nextInt(100000);

		String name;
		do {
			name = "Prod";
			for (int i = 0; i < 4; ++i) {
				name += ALPHA.charAt(rng.nextInt(ALPHA.length()));
			}
		} while (usedNames.contains(name));

		IProduct product = enterprise.createProduct(barcode, purchasePrice,
				name);
		generatedProducts.add(new GeneratedProduct(barcode, purchasePrice,
				name, product));
		
		return generatedProducts.size() - 1;
	}

	/**
	 * Returns a new barcode which is not already used for any product.
	 */
	public int getUnusedBarcode() {
		int barcode = 0;
		do {
			barcode = rng.nextInt(1000000);
		} while (usedBarcodes.contains(barcode));
		return barcode;
	}

}
