package org.cocome.tradingsystem.systests.scenarios;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import junit.framework.TestCase;

import org.cocome.tradingsystem.systests.TestManager;
import org.cocome.tradingsystem.systests.interfaces.IBank;
import org.cocome.tradingsystem.systests.interfaces.ICashDesk;
import org.cocome.tradingsystem.systests.interfaces.IEnterprise;
import org.cocome.tradingsystem.systests.interfaces.IStorePC;
import org.cocome.tradingsystem.systests.interfaces.ITestDriver;
import org.cocome.tradingsystem.systests.util.GeneratedProduct;
import org.cocome.tradingsystem.systests.util.GeneratedStockItem;
import org.cocome.tradingsystem.systests.util.ProductGenerator;
import org.cocome.tradingsystem.systests.util.StockGenerator;

/**
 * This is the base class for all test scenarios. It prepares the system for
 * testing and provides helper methods to simplify test setup and testing.
 * 
 * @author Benjamin Hummel
 * @author Christian Pfaller
 * @author $Author: hummel $
 * @version $Rev: 64 $
 * @levd.rating GREEN: 64
 */
public abstract class TestScenarioBase extends TestCase {

	/** Random number generator used to generate seeds for other RNGs. */
	protected final Random seedRng = new Random(42);

	/** The test driver used for this test. */
	protected ITestDriver testDriver;

	/** The (globally unique) enterprise used. */
	protected IEnterprise enterprise;

	/** Generator used for adding new products. */
	protected ProductGenerator productGenerator;

	/** The list of stores. */
	protected List<StoreWrapper> stores;

	/** The bank */
	protected IBank bank;

	/** {@inheritDoc} */
	@Override
	protected void setUp() throws Exception {
		super.setUp();
		testDriver = TestManager.getInstance().createTestDriver();
		enterprise = testDriver.initializeSystem();

		productGenerator = new ProductGenerator(seedRng.nextLong(), enterprise);
		stores = new ArrayList<StoreWrapper>();
		bank = testDriver.getBank();
	}

	/**
	 * Copy the setup information from another scenario. This is used to
	 * initialize this class without actually creating new objects which is
	 * useful for concurrent testing where this is only the base for some tests.
	 */
	protected void copySetup(TestScenarioBase otherBase) {
		this.testDriver = otherBase.testDriver;
		this.enterprise = otherBase.enterprise;
		this.productGenerator = otherBase.productGenerator;
		this.stores = otherBase.stores;
		this.bank = otherBase.bank;
	}

	/**
	 * Creates a new store with the given number of cash lines and a generated
	 * stock.
	 */
	protected StoreWrapper createStore(int numCashLines) throws Exception {
		StoreWrapper wrapper = new StoreWrapper(testDriver.createStore());
		wrapper.getStockGenerator().generateAll();
		for (int i = 0; i < numCashLines; ++i) {
			wrapper.createCashDesk();
		}

		stores.add(wrapper);
		return wrapper;
	}

	/** {@inheritDoc} */
	@Override
	protected void tearDown() throws Exception {
		testDriver.shutdownSystem();
		super.tearDown();
	}

	/** Class holding a store and some management information. */
	protected class StoreWrapper {

		private final Random rng;

		/** The store PC for the store. */
		private final IStorePC storePC;

		/** The stock generator used for generator stock items. */
		private final StockGenerator stockGenerator;

		/** The list of cash desks for this store. */
		private final List<ICashDesk> cashDesks = new ArrayList<ICashDesk>();

		/**
		 * Creates a new instance.
		 * 
		 * @param storePC
		 *            the store PC for the store.
		 */
		public StoreWrapper(IStorePC storePC) {
			this.rng = new Random();
			this.storePC = storePC;
			this.stockGenerator = new StockGenerator(seedRng.nextLong(),
					storePC, productGenerator);
		}

		/** Creates a new cash line for this store. */
		public void createCashDesk() throws Exception {
			cashDesks.add(testDriver.createCashDesk(storePC));
		}

		/** Returns the number of cash desks for this store. */
		public int getNumberOfCashDesks() {
			return cashDesks.size();
		}

		/** Returns the cash desk of the given index. */
		public ICashDesk getCashDesk(int i) {
			return cashDesks.get(i);
		}

		/** Returns the stock generator used for generator stock items. */
		public StockGenerator getStockGenerator() {
			return stockGenerator;
		}

		/** Returns the store PC for the store. */
		public IStorePC getStorePC() {
			return storePC;
		}

		/** Returns a product which is ready for sale (amount in stock > 0) */
		public GeneratedProduct getProductReadyForSale() {
			List<GeneratedProduct> productsForSale = new ArrayList<GeneratedProduct>();
			for (int i = 0; i < getStockGenerator().getNumberOfStockItems(); i++) {
				GeneratedStockItem stockItem = getStockGenerator()
						.getGeneratedStockItem(i);
				GeneratedProduct product = stockItem.getProduct();
				if (stockItem.getAmount() > 0
						&& !productsForSale.contains(product)) {
					productsForSale.add(product);
				}
			}
			return productsForSale.get(rng.nextInt(productsForSale.size()));
		}

		/** Returns a product which is low on stock (amount <= minAmount) */
		public GeneratedStockItem getItemLowOnStock() {
			List<GeneratedStockItem> itemsForSale = new ArrayList<GeneratedStockItem>();
			for (int i = 0; i < getStockGenerator().getNumberOfStockItems(); i++) {
				GeneratedStockItem stockItem = getStockGenerator()
						.getGeneratedStockItem(i);
				if (stockItem.getAmount() <= stockItem.getMinAmount()) {
					itemsForSale.add(stockItem);
				}
			}
			return itemsForSale.get(rng.nextInt(itemsForSale.size()));
		}

	}

}
