package org.cocome.tradingsystem.systests.scenarios;

import java.util.concurrent.TimeoutException;

import org.cocome.tradingsystem.systests.interfaces.ICashBox;
import org.cocome.tradingsystem.systests.interfaces.ICashDesk;
import org.cocome.tradingsystem.systests.interfaces.IProduct;
import org.cocome.tradingsystem.systests.interfaces.IStorePC;
import org.cocome.tradingsystem.systests.util.GeneratedStockItem;

/**
 * This abstract class holds basic operations needed for a purchase and its
 * variants as described in use case 1.
 * 
 * @author Benjamin Hummel
 * @author Christian Pfaller
 * @author $Author: hummel $
 * @version $Rev: 64 $
 * @levd.rating GREEN Rev: 64
 */
public class ProcessSaleBase extends TestScenarioBase {

	/** The number of items sold for this test case. */
	private static final int NUM_SOLD_ITEMS = 15;

	/** The number of items currently sold. */
	protected int currentlySold = 0;

	/** The store used for this sale */
	protected IStorePC store;

	/** The cash desk used for this sale */
	protected ICashDesk cashDesk;

	/** The cash box used for this sale */
	protected ICashBox cashBox;

	/** Array which holds the expected remaining amount of a product in the store */
	protected int[] expectedAmounts;

	/** The list of products which where bought */
	protected IProduct[] products;

	/** Holds the sum to pay */
	protected int priceSum;

	/** Setup only a single store with two cash desks. */
	@Override
	protected void setUp() throws Exception {
		super.setUp();
		productGenerator.generate(100);
		createStore(3);
	}

	/**
	 * Executes a standard purchase process at a cash desk with no exceptions.
	 * Payment will be by cash.
	 */
	public void purchase() throws Exception {
		initializeCashDesk(0, 0);
		startNewSale();
		enterAllRemainingProducts();

		finishSale();

		handleCashPayment();

		updateInventory();
	}

	/**
	 * Executes actions for initializing cash desk when a customer arrives
	 * there. Corresponds to step 1 of use case 1.
	 * 
	 * @param storeIndex
	 *            the index of the store used.
	 * @param cashDeskIndex
	 *            the index of the cash desk in the store.
	 */
	protected void initializeCashDesk(int storeIndex, int cashDeskIndex)
			throws Exception {
		// 1. The customer arrives at cash desk with goods and/or services to
		// purchase.
		store = stores.get(storeIndex).getStorePC();
		cashDesk = stores.get(storeIndex).getCashDesk(cashDeskIndex);
		cashBox = cashDesk.getCashBox();
	}

	/**
	 * Executes actions for starting a new sale with a default number of items.
	 * Corresponds to step 2 of use case 1.
	 */
	protected void startNewSale() throws Exception {
		startNewSale(NUM_SOLD_ITEMS);
	}

	/**
	 * Executes actions for starting a new sale. Corresponds to step 2 of use
	 * case 1.
	 * 
	 * @param how_many_items
	 *            the number of items the sale should contain in total
	 */
	protected void startNewSale(int how_many_items) throws Exception {
		// 2. The cashier starts new sale by pressing a button at the cash box.
		cashBox.startNewSale();

		priceSum = 0;
		currentlySold = 0;
		expectedAmounts = new int[how_many_items];
		products = new IProduct[how_many_items];
	}

	/**
	 * Executes actions for entering products in the cashbox. This is done by
	 * using the bar code scanner. Correspondes to steps 3 and 4 of use case 1.
	 * This method enters all remaining products at once.
	 */
	protected void enterAllRemainingProducts() throws Exception,
			TimeoutException {
		enterProducts(products.length - currentlySold);
	}

	/**
	 * Executes actions for entering products in the cashbox. This is done by
	 * using the bar code scanner. Correspondes to steps 3 and 4 of use case 1.
	 * 
	 * @param howMany
	 *            the number of products to be sold next.
	 */
	protected void enterProducts(int howMany) throws Exception,
			TimeoutException {
		for (int i = 0; i < howMany && currentlySold < products.length; ++i, ++currentlySold) {
			// 3. The cashier enters item identifier.
			GeneratedStockItem stockItem = stores.get(0).getStockGenerator()
					.getGeneratedStockItem(2 * currentlySold + 3);
			products[currentlySold] = stockItem.getProduct().getProduct();
			expectedAmounts[currentlySold] = stockItem.getAmount() - 1;
			priceSum += stockItem.getSalesPrice();

			int barcode = stockItem.getProduct().getBarcode();
			// This can be done ... by using the barcode scanner.
			cashDesk.getBarcodeScanner().sendBarcode(barcode);

			// 4. The system records sale item and presents product description,
			// price, and running total.
			cashDesk.getUserDisplay().waitForUpdate(1000);
			assertTrue(cashDesk.getUserDisplay().isPriceShown(
					stockItem.getSalesPrice()));
			assertTrue(cashDesk.getUserDisplay().isProductNameShown(
					stockItem.getProduct().getName()));
		}
	}

	/**
	 * Executes actions for finishing sale (after entering all products in
	 * cashbox, before payment). Corresponds to step 5 of use case 1.
	 */
	protected void finishSale() throws Exception {
		// 5. The cashier indicates the end of entering items by pressing the
		// SaleFinished-button at the cash box.
		cashBox.finishSale();
	}

	/**
	 * Executs actions of handling cash payment. Corresponds to step 5 a. of use
	 * case 1.
	 */
	protected void handleCashPayment() throws Exception {
		// Holds the received payment cash
		int payment;

		// a. The cashier presses the button for bar payment.
		cashBox.startCashPayment();

		payment = priceSum + 42;
		// iii. The cashier enters the cash received and confirms by pressing
		// Enter.
		cashBox.enterReceivedCash(payment);

		// i. The cash box opens. (NOTE: in the use case the cash box opens
		// BEFORE the amount is entered. However the reference implementation
		// expects the amount to be entered before opening the cash box, so we
		// use this order here)
		cashBox.waitForUpdate(1000);
		assertTrue("The system should open the cashbox", cashBox
				.wasOpenSignalSent());
		cashBox.setCashboxStatus(false);

		// iv. The received money and the change amount are displayed, and the
		// cashier hands over the change.
		cashDesk.getUserDisplay().waitForUpdate(1000);

		// v. The cashier closes the cash box.
		cashBox.setCashboxStatus(true);
	}

	/**
	 * Executes actions after reciept of payment. Corresponds to steps 6, 7 and
	 * 8 of use case 1.
	 */
	protected void updateInventory() throws Exception {
		// 6. The system logs completed sale and sends sale information to the
		// inventory system to update the stock.
		Thread.sleep(500);
		for (int i = 0; i < currentlySold; ++i) {
			assertEquals("The amount should now be as calculated", store
					.getAmount(products[i]), expectedAmounts[i]);
		}

		// 7. The system prints the receipt and the cashier hands out the
		// receipt.
		cashDesk.getPrinter().waitForUpdate(500);

		// 8. The customer leaves with receipt and goods.
	}

}
