package org.cocome.tradingsystem.systests;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.regex.Pattern;

import org.cocome.tradingsystem.systests.interfaces.ITestDriver;

/**
 * This is a factory for the test driver. The information which driver should be
 * used is taken from a property file. The actual test driver is then created
 * via reflection.
 * 
 * @author Benjamin Hummel
 * @author $Author: hummel $
 * @version $Rev: 62 $
 * @levd.rating GREEN Rev: 62
 */
public class TestManager {

	/** The name of the file containing the properties controlling system tests. */
	private static final String propertiesFile = "systests.properties";

	/** The search path for the property file. */
	private static final String[] pathPrefixes = { "config/", "./" };

	/** The name of the key giving the class name for the test driver. */
	private static final String testDriverClassKey = "test-driver.class";

	/** The only instance of this singleton. */
	private static TestManager instance = null;

	/** The contents of the properties file used to control system tests. */
	private final Properties properties = new Properties();

	/** The class of the test driver used. */
	private Class<ITestDriver> testDriverClass;

	/**
	 * Creates a new test manager class. This is private as this is a singleton.
	 * Use getInstance instead.
	 */
	private TestManager() throws TestManagerException {
		loadProperties();
		loadClass();
	}

	/**
	 * Loads the class for the test driver later used to construct test driver
	 * instances.
	 */
	private void loadClass() throws TestManagerException {
		String className = properties.getProperty(testDriverClassKey, "");
		if (className.length() == 0) {
			throw new TestManagerException("The required key "
					+ testDriverClassKey
					+ " was not found in the test configuration.");
		}
		try {
			Class<?> clazz = Class.forName(className);
			if (!ITestDriver.class.isAssignableFrom(clazz)) {
				throw new TestManagerException("The class '" + className
						+ "' does not seem to implement the "
						+ ITestDriver.class.getSimpleName() + " interface!");
			}

			// this cast is ok, as we checked assignability before!
			@SuppressWarnings("unchecked")
			Class<ITestDriver> tempClass = (Class<ITestDriver>) clazz;

			testDriverClass = tempClass;
		} catch (ClassNotFoundException e) {
			throw new TestManagerException("The provided class '" + className
					+ "' could not be found! Is the class path correct?", e);
		}
	}

	/**
	 * Locate and load the properties file.
	 */
	private void loadProperties() throws TestManagerException {
		for (String prefix : pathPrefixes) {
			prefix.replaceAll("/", Pattern.quote(File.separator));
			if (!prefix.endsWith(File.separator)) {
				prefix += File.separator;
			}
			File file = new File(prefix + propertiesFile);
			if (file.canRead()) {
				try {
					properties.load(new FileInputStream(file));
				} catch (IOException e) {
					throw new TestManagerException(
							"Reading the configuration failed!", e);
				}
				return;
			}
		}
		throw new TestManagerException("Could not find the file "
				+ propertiesFile + " in the expected locations! CWD is "
				+ new File(".").getAbsolutePath());
	}

	/** Returns a newly created test driver. */
	public ITestDriver createTestDriver() throws InstantiationException,
			IllegalAccessException {
		return testDriverClass.newInstance();
	}

	/**
	 * Returns the single instance of this class.
	 */
	public static TestManager getInstance() throws TestManagerException {
		if (instance == null) {
			instance = new TestManager();
		}
		return instance;
	}
}
