/**
 * 
 */
package eu.qimpress.ide.editors.form.qoseditor.properties;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.gef.EditPart;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.PropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.ide.backbone.core.operations.SaveQModelUIOperation;
import eu.qimpress.ide.backbone.core.ui.models.adapters.QModelsComposedAdapterFactoryProvider;
import eu.qimpress.ide.editors.form.qoseditor.QoSEditorUtils;
import eu.qimpress.samm.qosannotation.Annotation;

/**
 * Tab section for showing a list of associated annotations.
 * 
 * @author Michal Malohlava
 *
 */
public class AssociatedQoSAnnotationSection extends AbstractPropertySection {
	
	private static final Logger logger = Logger.getLogger(AssociatedQoSAnnotationSection.class);

	/* Groups */
	protected Composite composite;
	
	protected Group annotationListGroup;
	
	protected Group annotationInfoGroup;
	
	protected PropertySheetPage annotationPage;

	/* Widgets */
	protected List annotationList;

	protected Button createNewAnnotationButton;

	protected Button addAnnotationButton;
	
	protected Button saveAnnotationButton;
	
	/** Selected element in project explorer */	
	private EObject selectedEntity;	
	
	/**
	 * 
	 */
	public AssociatedQoSAnnotationSection() {
		super();		
	}
	
	@Override
	public void setInput(IWorkbenchPart part, ISelection selection) {
		this.annotationPage.selectionChanged(part, new StructuredSelection());
		
		this.selectedEntity = null;
		
		if (selection instanceof IStructuredSelection) {
			Object o = ((IStructuredSelection) selection).getFirstElement();			
			this.selectedEntity = transformSelection(o);
		}
		
		super.setInput(part, new StructuredSelection(this.selectedEntity));
	}
	
	protected EObject transformSelection(Object o) {
		if (o instanceof EObject) {
			return (EObject) o;
		} else if (o instanceof EditPart) {
			Object model = ((EditPart) o).getModel();
			return model instanceof View ? ((View) model).getElement() : null;
		}
		
		return null;
	}
	
	
	@Override
	public void createControls(Composite parent, final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);
		
		initializeControls(parent, aTabbedPropertySheetPage);
	}

	protected void initializeControls(Composite parent, final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		composite = parent;
		
		Composite groups = getWidgetFactory().createComposite(composite);
		groups.setLayout(new GridLayout(1, false));
		createAnnotationListGroup(groups);
		annotationListGroup.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false));
				
		createAnnotationInfoGroup(groups, aTabbedPropertySheetPage);
		annotationInfoGroup.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, false, true));		
	}
	
	private Group createAnnotationListGroup(Composite parent) {
		annotationListGroup = getWidgetFactory().createGroup(parent, "Associated annotations");
		GridLayout layout = new GridLayout(2, false);
		annotationListGroup.setLayout(layout);
		
		annotationList = getWidgetFactory().createList(annotationListGroup, SWT.BORDER | SWT.SINGLE);
		GridDataFactory.fillDefaults().span(1, 2).grab(true,true).applyTo(annotationList);
		annotationList.addSelectionListener(new SelectionAdapter() {
			
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (annotationList.getSelectionIndex() != -1) {					
					Annotation annotation = ((Annotation[]) annotationList.getData())[annotationList.getSelectionIndex()];
										
					AssociatedQoSAnnotationSection.this.annotationPage.selectionChanged(null, new StructuredSelection(new Object[] {annotation}));
					AssociatedQoSAnnotationSection.this.saveAnnotationButton.setEnabled(true);
				}
			}			
		});
				
		// add annotation button
		addAnnotationButton = new Button(annotationListGroup, SWT.PUSH);
		addAnnotationButton.setText("Add ...");
		addAnnotationButton.setToolTipText("Associated annotation with the element ");
		addAnnotationButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		addAnnotationButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				IProject project = QImpressCore.getProject(AssociatedQoSAnnotationSection.this.selectedEntity.eResource().getURI());
				
				if (project != null) {
					handleAddAnnotationButton(project, e.widget.getDisplay().getActiveShell());
				} else {
					handleNullProject(e.display.getActiveShell(), "Error occured");
				}
			}
		});
		
		// create button for adding new annotations
		createNewAnnotationButton = new Button(annotationListGroup, SWT.PUSH);
		createNewAnnotationButton.setText("New ...");
		createNewAnnotationButton.setToolTipText("Create new annotation");
		createNewAnnotationButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));		
		createNewAnnotationButton.addSelectionListener(new SelectionAdapter() {			
			@Override
			public void widgetSelected(SelectionEvent e) {
				IProject project = QImpressCore.getProject(AssociatedQoSAnnotationSection.this.selectedEntity.eResource().getURI());
				
				if (project != null) {
					handleCreateNewAnnotationButton(project, e.display.getActiveShell());
				} else {
					handleNullProject(e.display.getActiveShell(), "Error occured");					
				}
			}
		});		
		
		return annotationListGroup;
	}
	
	private Group createAnnotationInfoGroup(Composite parent, final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		annotationInfoGroup = getWidgetFactory().createGroup(parent, "Annotation info");
		
		GridLayout layout = new GridLayout(2, false);		
		annotationInfoGroup.setLayout(layout);
		
		Composite pSheetComposite = getWidgetFactory().createFlatFormComposite(annotationInfoGroup) ;
		GridDataFactory.fillDefaults().grab(true, true).applyTo(pSheetComposite);
		
		annotationPage = new PropertySheetPage();		
		annotationPage.setPropertySourceProvider(new AdapterFactoryContentProvider(QModelsComposedAdapterFactoryProvider.getInstance().getAdapterFactory()));

		annotationPage.createControl(pSheetComposite);
		FormData data = new FormData();
        data.left = new FormAttachment(0, 0);
        data.right = new FormAttachment(100, 0);
        data.top = new FormAttachment(0, 0);
        data.bottom = new FormAttachment(100, 0); 
        data.height = 150;
        annotationPage.getControl().setLayoutData(data);		
	
		annotationPage.getControl().addControlListener(new ControlAdapter() {

			public void controlResized(ControlEvent e) {
				aTabbedPropertySheetPage.resizeScrolledComposite();
			}
		});
		
		saveAnnotationButton = new Button(annotationInfoGroup, SWT.PUSH);
		saveAnnotationButton.setText("Save");
		saveAnnotationButton.setToolTipText("Save annotation changes");
		saveAnnotationButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING | GridData.VERTICAL_ALIGN_BEGINNING));
		saveAnnotationButton.setEnabled(false);
		saveAnnotationButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {				
				IProject project = QImpressCore.getProject(AssociatedQoSAnnotationSection.this.selectedEntity.eResource().getURI());
				Shell shell = e.display.getActiveShell();
				
				handleAnnotationSaveButton(project, shell);
			}
		});
				
		return annotationInfoGroup;		
	}
	
	protected void handleAnnotationSaveButton(IProject project, Shell shell) {
		IQModel qModel = null;
		try {
			qModel = QImpressCore.getQProject(project).getRepository().getDefaultAlternative().getModel(RepositoryModels.ANNOTATION_MODEL_EXT);
			
			SaveQModelUIOperation saveOperation = new SaveQModelUIOperation(qModel);
			saveOperation.run(new NullProgressMonitor());
		} catch (Exception e) {
			logger.error("Cannot save annotation model: qModel:" + qModel, e);		
			MessageDialog.openError(shell,
					"Annotation model save",
					"Cannot save annotation model because of " + e.getMessage());
					
		}
			
	}

	@Override
	public void refresh() {
		if (this.selectedEntity != null) {
			try {
				Annotation[] annotations = QoSEditorUtils.getAttachedAnnotations(this.selectedEntity);
				
				annotationList.removeAll();
				
				StringBuilder sb = new StringBuilder();
				for (Annotation annotation : annotations) {
					QoSEditorUtils.getAnnotationLabel(annotation, sb);
					String key = sb.toString();
					
					annotationList.add(key);					
				}
				annotationList.setData(annotations);
			} catch (RepositoryException e) {
				logger.debug("Cannot get a list of annotations attached to element " + this.selectedEntity, e);
			}
		}		
	}
	
	protected void handleAddAnnotationButton(IProject project, Shell shell) {
		assert(project != null);
		
		Annotation annotation = null;
		try {
			project = QImpressCore.getProject(this.selectedEntity.eResource().getURI());
			
			annotation = QoSEditorUtils.browseForAnnotationInstance(shell, 
					QoSEditorUtils.getAccessibleAnnotationsForType(
							project, 
							AssociatedQoSAnnotationSection.this.selectedEntity.getClass()),
					null);
			
		} catch (RepositoryException ex) {
			MessageDialog.openError(shell, 
					"Cannot open instance browser", "Error occured during annotation instance browser opening");
			logger.error("Cannot get a list of accesible annotations for entity " + AssociatedQoSAnnotationSection.this.selectedEntity, ex);			
		}
		
		if (annotation == null) {
			return;
		}
		
		
		// update the reference to selected entity
		if (!QoSEditorUtils.setAnnotationReference(annotation, this.selectedEntity)) {
			logger.warn("Cannot setup reference [" + this.selectedEntity + "] for annotation:" + annotation);
		} else {					
			refresh();
		}
	}	
	
	protected void handleCreateNewAnnotationButton(IProject project, Shell shell) {		
		assert(project != null);
		
		// user selects desired annotation type
		EClass eClazz = QoSEditorUtils.browseForAnnotationType(shell,						
				QoSEditorUtils.getAnnotationTypesForType(AssociatedQoSAnnotationSection.this.selectedEntity.getClass()), 
				null);
		
		if (eClazz == null) {
			return;
		}
		
		// create new annotation in the default alternative 
		try {
			Annotation annotation = QoSEditorUtils.addNewAnnotation(project, eClazz);
			
			if (annotation != null) {
				// update the reference to selected entity
				if (!QoSEditorUtils.setAnnotationReference(annotation, this.selectedEntity)) {
					logger.warn("Cannot setup reference [" + this.selectedEntity + "] for annotation:" + annotation);
				} else {					
					refresh();
				}
				
			}
		} catch (RepositoryException e) {
			MessageDialog.openError(shell, 
					"Create new annotation", 
					"Error occured during new annotation creation");

			logger.error("Cannot create new annotation in the default alternative", e);
		}
	}
	
	protected void handleNullProject(Shell shell, String title) {
		MessageDialog.openError(shell, 
				title,
				"Cannot obtain reference to project");
	}
}
