package eu.qimpress.ide.editors.form.qoseditor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.provider.EcoreItemProviderAdapterFactory;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.ide.editors.form.qoseditor.internal.QoSAnnotationCache;
import eu.qimpress.samm.qosannotation.Annotation;
import eu.qimpress.samm.qosannotation.QosAnnotations;
import eu.qimpress.samm.qosannotation.QosannotationFactory;
import eu.qimpress.samm.qosannotation.util.QosannotationAdapterFactory;

/**
 * Utils for QoS reflective editor.
 * 
 * @author Michal Malohlava
 *
 */
public class QoSEditorUtils {
	
	private static final Annotation[] EMPTY_ANNOTATION_LIST = new Annotation[] {};
	
	private static final Logger logger = Logger.getLogger(QoSEditorUtils.class);
	
	/**
	 * Returns a list of SAMM-related annotations types defined in the platform.
	 * 
	 * @return list of EClassifier 
	 */
	public static EClass[] getDefinedAnnotationsTypes() {
		return QoSAnnotationCache.getInstance().getDefinedAnnotationTypes();		
	}
		
	/**
	 * Returns true if a given <code>annotationType</code> can annotate class <code>classToBeAnnotate</code>.
	 * 
	 * @param annotationType annotation type
	 * @param classToBeAnnotated class which is a possible target of annotation type.
	 * 
	 * @return true if annotationType annotates classToBeAnnotated
	 */
	public static boolean containsAttributeForType(EClass annotationType, Class classToBeAnnotated) {		
		
		for(EStructuralFeature esf : QoSAnnotationCache.getInstance().getAnnotationFeatures(annotationType)) {
			if (esf.isChangeable() 
					&& !esf.isMany()
					&& esf.getEType().getInstanceClass().isAssignableFrom(classToBeAnnotated)) {
				
				if (logger.isTraceEnabled())
					logger.trace("EClass [" + annotationType.getName() + "] contains attribute [" + esf.getName() + "] of type :" + classToBeAnnotated);
				
				return true;				
			}
		}
		
		return false;			
	}
	/**
	 * Returns a list of unique entities contained in models in the project, that could be annotated by
	 * given annotation
	 * @param IQProject project
	 * @param annotationType Annotation Type
	 * @return List of Classes that could be annotated by this entity.
	 */
	public static Object[] getAnnotatableEntitiesByAnnotation(IQProject project, EClass annotationType){
		List<Object> AnnotatableEntities = new ArrayList<Object>();
		try {
			IQModel[] models = project.	getRepository().
										getDefaultAlternative().
										getModels();
			
			for(IQModel model : models) {
				Resource emfResource =  model.getTopLevelEObject().eResource();
				 TreeIterator<EObject> modelEntitiesIterator = emfResource.getAllContents();
			    EObject modelEntity = null; 
			    while( modelEntitiesIterator.hasNext()){
			    	modelEntity = (EObject)modelEntitiesIterator.next();
					if (containsAttributeForType(annotationType, modelEntity.getClass())){
						AnnotatableEntities.add(modelEntity);
					}
				}
			    
			}
		} catch (RepositoryException e) {
			System.err.println("Uncastched Repository exception. Can not get project repository");
		}
		return AnnotatableEntities.toArray(new Object[AnnotatableEntities.size()]);
	}
	
	

	
	
	public static EClass[] getAnnotationTypesForType(Class clazz) {
		EClass[] annotations = getDefinedAnnotationsTypes();
		Collection<EClass> result = new ArrayList<EClass>();
		
		for(EClass annotation : annotations) {
			if (containsAttributeForType(annotation, clazz)) {
				result.add(annotation);
			}
		}
		
		return result.toArray(new EClass[result.size()]);
	}
	
	public static Annotation[] getAttachedAnnotations(EObject eo) throws RepositoryException {
		IProject project = QImpressCore.getProject(eo.eResource().getURI());
		if (project == null) {
			return EMPTY_ANNOTATION_LIST;			
		}
		
		Annotation[] definedAnnotations = getAccessibleAnnotations(project);
		List<Annotation> result = new ArrayList<Annotation>();
		
		for(Annotation annotation : definedAnnotations) {
			EStructuralFeature[] eSFeatures = QoSAnnotationCache.getInstance().getAnnotationFeatures(annotation.eClass());
			
			for(EStructuralFeature esf : eSFeatures) {
				Object value = annotation.eGet(esf);
			
				if (value instanceof EObject) {
										
					if (EcoreUtil.getURI((EObject) value).equals(EcoreUtil.getURI(eo))) {					
						result.add(annotation);					
						break;
					}
				}
			}
		}
		
		return result.toArray(new Annotation[result.size()]);
	}
	
	public static QosAnnotations[] getAnnotationRepositories(IProject project) {
		// TODO dat do Q-ICore methodu, ktera vrati vsechny IQModely pro dany TYP modelu - tj. global i default alternative repository
		return null;	
	}
	
	public static Annotation[] getAccessibleAnnotationsForType(IProject project, Class clazz) throws RepositoryException {
		Annotation[] annotations = getAccessibleAnnotations(project);
		EClass[] annotationTypes = getAnnotationTypesForType(clazz);
		
		List<Annotation> result = new ArrayList<Annotation>(4);
		
		for(Annotation annotation : annotations) {
			for(EClass annotationType : annotationTypes) {
				if (annotation.eClass().equals(annotationType)) {
					result.add(annotation);
				}
			}
		}
		
		return result.toArray(new Annotation[result.size()]);		
	}
	
	public static Annotation[] getAccessibleAnnotations(IProject project) throws RepositoryException {
		IQProject qProject = QImpressCore.getQProject(project);
		
		if (qProject == null) {
			return EMPTY_ANNOTATION_LIST;
		}
		
		IQModel model = qProject.getRepository().getDefaultAlternative().getModel(RepositoryModels.ANNOTATION_MODEL_EXT);
		
		QosAnnotations annotations =  model.getTopLevelEObject(QosAnnotations.class, QosannotationFactory.eINSTANCE.getQosannotationPackage().getQosAnnotations());
				
		return annotations.getAnnotation().toArray(new Annotation[annotations.getAnnotation().size()]);
	}
	
	protected static Annotation[] getQModelAnnotations(IQModel qModel) {
		QosAnnotations annotations = (QosAnnotations) qModel.getTopLevelEObject(QosAnnotations.class, QosannotationFactory.eINSTANCE.getQosannotationPackage().getQosAnnotations());
		
		return annotations.getAnnotation().toArray(new Annotation[annotations.getAnnotation().size()]);
	}
	
	/**
	 * TODO 
	 * 
	 * @param eClazz
	 * @param clazz
	 * @return
	 */
	public static EStructuralFeature getStructuralFeatureForType(EClass eClazz, Class clazz) {
		EStructuralFeature[] features = QoSAnnotationCache.getInstance().getAnnotationFeatures(eClazz);
		
		for(EStructuralFeature esf : features) {
			if (esf.isChangeable() 
					&& !esf.isMany()
					&& esf.getEType().getInstanceClass().isAssignableFrom(clazz)) {
				return esf;
			}
		}
		
		return null;
	}
	
	/**
	 * TODO
	 * 
	 * @param annotation
	 * @param referencedObject
	 * @return true if the reference was configured, else returns false
	 */
	public static boolean setAnnotationReference(Annotation annotation, EObject referencedObject) {
		EStructuralFeature esf = getStructuralFeatureForType(annotation.eClass(), referencedObject.getClass());
		if (esf != null) {
			annotation.eSet(esf, referencedObject);
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * Opens dialog for selecting annotation types.
	 * 
	 * @param shell 
	 * @param elements
	 * @param initialPattern
	 * @return selected Annotation type or null
	 */
	public static final EClass browseForAnnotationType(Shell shell, EClass[] eclasses, String initialPattern) {
		
		ElementListSelectionDialog dialog= getSelectionDialog(shell,
				eclasses,
				new AdapterFactoryLabelProvider(new EcoreItemProviderAdapterFactory()),
				initialPattern);
		
		dialog.setTitle("Create new annotation in default alternative");
		dialog.setMessage("Select annotation type");
		dialog.open();
		
		if (dialog.getReturnCode() == Window.OK) {
			EClass type= (EClass) dialog.getResult()[0];
							
			return type;
		}
		
		return null;
	}
	
	/**
	 * Opens dialog for selecting annotation instance.
	 * 
	 * @param shell 
	 * @param elements
	 * @param initialPattern
	 * @return selected Annotation type or null
	 */
	public static final Annotation browseForAnnotationInstance(Shell shell, Annotation[] annotations, String initialPattern) {
		
		ElementListSelectionDialog dialog= getSelectionDialog(shell,
				annotations,
				new AdapterFactoryLabelProvider(new QosannotationAdapterFactory()),
				initialPattern);
		
		dialog.setTitle("Attach existing annotation to the element");
		dialog.setMessage("Select annotation instance");
		dialog.open();
		
		if (dialog.getReturnCode() == Window.OK) {
			Annotation annotation = (Annotation) dialog.getResult()[0];
							
			return annotation;
		}
		
		return null;
	}
	
	protected static ElementListSelectionDialog getSelectionDialog(Shell shell, Object[] objects, ILabelProvider renderer, String filter) {
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(shell, renderer);
		
		dialog.setAllowDuplicates(false);
		dialog.setElements(objects);
		dialog.setFilter(filter);
		dialog.setSize(120, 18);
		
		return dialog;
	}
	
	public static Annotation addNewAnnotation(IProject project, EClass eClazz) throws RepositoryException {
		IQModel model = QImpressCore.getQProject(project).getRepository().getDefaultAlternative().getModel(RepositoryModels.ANNOTATION_MODEL_EXT);
		Annotation annotation = null;
		
		if (model != null) {
			// add new annotation
			QosAnnotations annotations = model.getTopLevelEObject(QosAnnotations.class, QosannotationFactory.eINSTANCE.getQosannotationPackage().getQosAnnotations());
			
			assert(annotations != null);
									
			EPackage ePackage = eClazz.getEPackage();			
			annotation = (Annotation) ePackage.getEFactoryInstance().create(eClazz);
			
			annotations.getAnnotation().add(annotation);
			
			// save model
			// ?			
		}
		
		return annotation;
	}
	
	
	public static Annotation addNewAnnotationInstance(IProject project, Annotation annotationInstance) throws RepositoryException {
		IQModel model = QImpressCore.getQProject(project).getRepository().getDefaultAlternative().getModel(RepositoryModels.ANNOTATION_MODEL_EXT);
		
		if (model != null) {
			// add new annotation
			QosAnnotations annotations = model.getTopLevelEObject(QosAnnotations.class, QosannotationFactory.eINSTANCE.getQosannotationPackage().getQosAnnotations());
			
			assert(annotations != null);
									
		
			
			annotations.getAnnotation().add(annotationInstance);
			
			// save model
			// ?			
		}
		
		return annotationInstance;
	}	
	
	public static Map<EClass, Annotation> createPrototypeInstances(EClass[] annotationTypes) {
		Map<EClass, Annotation> result = new HashMap<EClass, Annotation>();
		
		for(EClass eClass : annotationTypes) {
			EPackage ePackage = eClass.getEPackage();			
			Annotation annotation = (Annotation) ePackage.getEFactoryInstance().create(eClass);
			
			result.put(eClass, annotation);			
		}
		
		return result;		
		
	}
	
	public static void getAnnotationLabel(Annotation annotation, StringBuilder sb) {
		sb.setLength(0);
		sb.append(annotation.getName());
		sb.append(" (");
		sb.append(annotation.eClass().getName());
		sb.append(')');
	}
}
