package org.ow2.dsrg.fm.tbpjava.utils;

import java.util.HashMap;
import java.util.Map;

import org.ow2.dsrg.fm.tbpjava.checker.StatesMapperLayer;
import org.ow2.dsrg.fm.tbpjava.checker.VariablesEvaluation;

/**
 * Standard hash map with one entry added. {@link #hashCode}
 * 
 * Note: It's permitted to only method defined here.
 * <br>
 * Used in the {@link StatesMapperLayer} to hold hashCode for one entry ({@link VariablesEvaluation} and associated automaton states {@link ThreadStateStack}
 * 
 */
public final class HashMapWithHashCode<Key extends Integer, Value extends Object> extends HashMap<Key, Value> implements Comparable<HashMapWithHashCode<? extends Key, ? extends Value>> {
	private static final boolean INTERNAL_CHECKS = false; // Enables/disables internal hash code validity checks

	private static final long serialVersionUID = 20100607001L;

	private static final int INITIAL_HASH_VALUE = 0xAA00AAAA; /// Initial value for property hashCode
	private static final int KEY_OFFSET = 2;
	
	private int hashCode = INITIAL_HASH_VALUE;
	
	private final VariablesEvaluation varEval; /// To this VariableEvaluation are associated {@link ThreadStateStack} mapped. (It's necessary to correctly handle the hashCodes)
	
	/**
	 * Creates empty hash map.
	 */
	public HashMapWithHashCode(VariablesEvaluation varEval) {
		super();
		this.varEval = varEval;
		this.hashCode = calculateHashCode();
	}
	
	/**
	 * Makes deep copy of the {@link HashMapWithHashCode}
	 * 
	 * @param source Original to clone. Cann't be null.
	 */
	public HashMapWithHashCode(HashMapWithHashCode<? extends Key,? extends Value> source) {
		super(source);
		assert source != null;
		this.varEval = source.varEval;
		this.hashCode = source.hashCode;

		if (INTERNAL_CHECKS) { source.DEBUG_checkConsistency(); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}
	/**
	 * Creates a new Map with the same mappings as the specified Map.
	 * @param m Mapping to copy.
	 */
	public HashMapWithHashCode(VariablesEvaluation varEval, HashMapWithHashCode<? extends Key,? extends Value> m) {
		super(m);
		this.varEval = varEval;
		this.hashCode = calculateHashCode();
	}
	
	public VariablesEvaluation getAssociatedVariablesEvaluation() {
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		return varEval;
	}
	
	public int getHashCode() {
		return hashCode;
	}
	
	/**
	 * Nasty trick. This is not immutable hash code as required by specification !!.
	 * Remove from any Set/Collection/Map (which uses the hahCode) before any change !!!!
	 */
	public int hashCode() {
		return hashCode;
	}
	
	public int calculateHashCode() {
		int result = INITIAL_HASH_VALUE;
		
		result ^= varEval.getChecherHashCode();
		for(Map.Entry<Key, Value> entry : this.entrySet()) {
			result ^= ((entry.getKey()+KEY_OFFSET) * entry.getValue().hashCode());
		}
		
		return result;
	}
	
	@Override
	public Value put(Key key, Value value) {
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		Value oldValue = super.put(key, value);
		// Remove oldValue from the hashCode
		if (oldValue != null) {
			hashCode ^= ((key+KEY_OFFSET) * oldValue.hashCode());
		}
		// Add new value into hasCode
		hashCode ^= ((key+KEY_OFFSET) * value.hashCode());
		
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		return oldValue;
	}

	@Override
	public void putAll(Map<? extends Key, ? extends Value> map) {
		throw new RuntimeException("Unsupported operation");
	}

	@Override
	public Value remove(Object key) {
		throw new RuntimeException("Unsupported operation - use typed remove operator instead");
	}
	
	public Value remove(Key key) {
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		Value removed = super.remove(key);
		if (removed != null) {
			hashCode ^= ((key+KEY_OFFSET) * removed.hashCode());
		}
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		return removed;
	}

	//@Override
	public void clear() {
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		super.clear();
		hashCode = calculateHashCode();
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}

	
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		HashMapWithHashCode<? extends Integer, ? extends Object> other = (HashMapWithHashCode<?,?>) obj;

		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		if (INTERNAL_CHECKS) { other.DEBUG_checkConsistency(); }

		if (getHashCode() != other.getHashCode()) {
			return false;
		}
		
		if (size() != other.size()) {
			return false;
		}
		
	if (varEval == null) {
			if (other.varEval != null) {
				return false;
			}
		} else if (!varEval.equalsValues(other.varEval)) {
			return false;
		}
		
	for(Map.Entry<Key, Value> entry : this.entrySet()) {
			Object otherVal = other.get(entry.getKey());
			if (entry.getValue() == null) {
				if (otherVal != null) {
					return false;
				}
			} else if (!entry.getValue().equals(otherVal)) {
				return false;
			}
		}
		
		return true;
	}

	public void DEBUG_checkConsistency() {
		int caclHashCode = calculateHashCode();
		if (caclHashCode != hashCode) {
			// Breakpoint place
			System.err.println(HashMapWithHashCode.class.getName() + " - Internal CHECK failed !!!!");
			throw new RuntimeException(HashMapWithHashCode.class.getName() + " - Internal CHECK failed !!!!");
		}
	}

	@Override
	public int compareTo(HashMapWithHashCode<? extends Key, ? extends Value> o) {
		if (o == null) {
			return 1;
		}
		
		if (hashCode() > o.hashCode()) {
			return 1;
		} else if (hashCode() < o.hashCode()) {
			return -1;
		} else {
			return 0;
		}
	}

}
