package org.ow2.dsrg.fm.tbpjava.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class Configuration {
	private final static boolean DEBUG = false;

	// Configuration choices
	
	/// Name of checked component
	private String ComponentName = null;

	
	/**
	 *  Name of class (classes) of the checked component.
	 *   In case of the "pure" component it's sufficient to provide 
	 *     only the "main" class which implements managing interfaces of the component system.
	 *     (For SOFA - {@link org.objectweb.dsrg.sofa.SOFAClient} )
	 *   If environment is generated for non component based code, all classes of one component has to be specified.
	 */
	private List<String> ComponentImplementationClasses = null;
	
	/// Maps provided protocol interfaces names into equivalent java interfaces
	private Map<String, String> ComponentProvidedInterfaces = new HashMap<String, String>();

	/// Maps required interface protocol names into implementing java classes
	private Map<String, String> ComponentRequiredInterfaces = new HashMap<String, String>();

	/// Limit used for converting unlimited reentrancy into limited one
	private int ComponentReentrancyLimit = 2; // 2 is default value
	
	/// Class where value of parameters are created
	private String EnvValueSets = null;

	/// Directory where store generated environment files 
	private String EnvTargetDir = "." + File.separator; 
	
	/// File with description of TBP of checked component 
	private String EnvProtocol = null;

	/// Flag whether generate environment (if not set, environment will not be generated and reused existing one, if (exists)) 
	private boolean EnvGenerate = true;

	
	/// Enumerate possibilities how generated code can sets required interfaces for tested component. 
	public enum RequiredInterfaceSettingMethod { 
		SETTERS, /// Use setter method in form set<InterfaceName>( interface class);
		SOFA2_CLIENT, /// Use SofaClient interface. Method void setRequired(String name, Object iface);
		FRACTAL, /// Uses fractal {@link org.objectweb.fractal.api.control.BindingController} to set provided and required interfaces

		/**
		 * Finds appropriate (component implementation) class with entry named according to the Required Interface name and set this field directly.
		 *  Prints error if no such field is found or if its private (value cannot be assigned).
		 *  <p>It is intended to be used for non component based code in conjunction with {@link ProvidedInterfaceAccessMethod#SEPARATE_IMPLEMENTATION_CLASSES}/
		 */
		FIELDS 
	};

	/// Specifies how should generated environment sets required interfaces for tested component.
	private RequiredInterfaceSettingMethod EnvRequierdItfcSettingMethod =  RequiredInterfaceSettingMethod.SETTERS;
	
	/// Enumerate possibilities for generating accept call on provided interfaces. 
	public enum ProvidedInterfaceAccessMethod {
		COMPONENT_CLASSES, /// Component classes implements all provided interfaces
		SEPARATE_CLASSES, /// Components implements SOFAServer. Use this interface for obtaining separate instance for each provided interface.
		
		//TODO REMOVE THIS ENTRY
		/**
		 * Create field for class specified in {@link Configuration#ComponentImplementationClasses} implementation class.
		 *   Provided calls are generated according implemented interface of the {@link Configuration#ComponentImplementationClasses}.
		 * <p>This entry is intended to be used for non component based code.
		 */
		SEPARATE_IMPLEMENTATION_CLASSES
	}
	/// Specifies hoe should by called method on provided interfaces of tested component.
	private ProvidedInterfaceAccessMethod EnvProvidedItfsAccesMethod = ProvidedInterfaceAccessMethod.COMPONENT_CLASSES;

	/// Mark if no error where found in configuration file  
	private boolean ConfigurationError = false;	

	// Flag that marks that configuration entry is specified in given configuration (file or command line) 
	/// Configuration contains "component.name" entry.
	private boolean setFlagComponentName = false;
	/// Configuration contains "component.implclass" entry.
	private boolean setFlagComponentImplementationClass = false;
	/// Configuration contains "component.provitfs" entry.
	@SuppressWarnings("unused")
	private boolean setFlagComponentProvidedInterfaces = false;
	/// Configuration contains "component.reqitfs" entry.
	@SuppressWarnings("unused")
	private boolean setFlagComponentRequiredInterfaces = false;
	/// Configuration contains "component.reentrancylimit" entry.
	@SuppressWarnings("unused")
	private boolean setFlagComponentReentrancyLimit = false;
	/// Configuration contains "env.valuesets" entry.
	private boolean setFlagEnvValueSets = false;
	/// Configuration contains "env.targetdir" entry.
	@SuppressWarnings("unused")
	private boolean setFlagEnvTargetDir = false;
	/// Configuration contains "env.protocol" entry.
	private boolean setFlagEnvProtocol = false;
	/// Configuration contains "env.reqitfs_setmethod" entry.
	@SuppressWarnings("unused")
	private boolean setFlagEnvRequierdItfcSettingMethod = false;
	/// Configuration contains "env.provitfs_acces_method" entry.
	@SuppressWarnings("unused")
	private boolean setFlagEnvProvidedItfsAccesMethod = false;
	/// Configuration contains "env.generate" entry.
	@SuppressWarnings("unused")
	private boolean setFlagEnvGenerate = false;

	
	// Separators
	/// String used for separating Identifier and Value part of line
	private static final String SeparatorIDValue = "=";
	/// String used for separating for multiple Values in value part
	private static final String SeparatorMultipleValues = ";";
	/// String used for dividing one value into components 
	private static final String SeparatorPairValue = "-";
	/// String used for line comment
	private static final String SeparatorComment = "#";
	
	// Identifiers
	private static final String IDComponentName = "component.name";
	private static final String IDComponentImplementationClass = "component.implclass";
	private static final String IDComponentProvidedInterfaces = "component.provitfs";
	private static final String IDComponentRequiredInterfaces = "component.reqitfs";
	private static final String IDComponentReentrancyLimit = "component.reentrancylimit";
	private static final String IDEnvValueSets = "env.valuesets";
	private static final String IDEnvTargetDir = "env.targetdir";
	private static final String IDEnvProtocol = "env.protocol";
	private static final String IDEnvRequierdItfcSettingMethod = "env.reqitfs_setmethod";
	private static final String IDEnvProvidedItfsAccesMethod = "env.provitfs_acces_method";
	private static final String IDEnvGenerate = "env.generate";

	// RequiredInterfaceSettingMethod constants .... case insensitive
	private static final String RISM_SETTERS = "setters"; 
	private static final String RISM_SOFA2_CLIENT = "sofa2"; 
	private static final String RISM_FRACTAL = "fractal"; 
	private static final String RISM_FIELDS = "fields";
	
	// ProvidedInterfaceAccessMethod constants .... case insensitive
	private static final String PIAM_COMPONENT_CLASS = "commponnent"; 
	private static final String PIAM_SEPARATE_CLASSES = "separate"; 
	private static final String PIAM_SEPARATE_IMPLEMENTATION_CLASSES = "impl_separate"; 

	// In case of boolean entry positive and negative value representations
	private static final String[] BOOLEAN_VALUES_TRUE  = { "true", "yes", "ano" };
	private static final String[] BOOLEAN_VALUES_FALSE = { "false", "no", "ne" };

	/// Output printer
	private PrintStream out = System.out;
	
	/**
	 * Reads configuration from file and create its runtime representation. 
	 * Command line parameters have bigger precedence and overwrites values from configuration file.
	 * 
	 * @param argv String array with configuration values. Use as command line input.
	 * @param fileName File where configuration stored. If null no configuration file read.
	 * @param out Stream where print (error) output 
	 */
	public Configuration(String[] argv, String fileName, PrintStream out) {
		this.out = out;
		if (fileName != null) {
			try {
				BufferedReader in = new BufferedReader(new FileReader(fileName));
				String str;
				int line = 1;
				while ((str = in.readLine()) != null) {
					processConfigurationLine(line, str);
					line++;
				}
				in.close();
			} catch (FileNotFoundException e) {
				System.out.println("Configuration file not found.");
			} catch (IOException e) {
				out.println("Error reading configuration file.");
				out.println(e.getMessage());
				e.printStackTrace(out);
			}
		}
		
		// Process string array (command line arguments)
		int line = -1;
		for(String str : argv) {
			processConfigurationLine(line, str);
			line--;
		}
		
		// Check if all mandatory options has been set
		checkMandatoryOptions();
	}

	/**
	 * Process one line with configuration and set internal variables according line
	 * @param str Line to process. If negative processing command lines parameters.
	 */
	private void processConfigurationLine(int line, String str) {
		// parse line
		
		// Check for white lines of comments
		StringBuffer lineBuff = new StringBuffer(str);
		while (lineBuff.length() > 0 && isWhiteSpace(lineBuff.charAt(0))) {
			lineBuff.deleteCharAt(0);
		}
		// Ignore empty lines and comments
		if ((lineBuff.length() == 0) || (lineBuff.indexOf(SeparatorComment) == 0)) {
			return;
		}
		
		// Check for correct line formating
		int positionIDValSeparator = lineBuff.indexOf(SeparatorIDValue);
		if (positionIDValSeparator == -1) {
			if (line > 0) { 
				out.println("Invalid configuration file format at line " + Integer.toString(line));
			} else {
				out.println("Invalid configuration format at command line parameter" + Integer.toString(-line));
				
			}
			ConfigurationError = true;
			return;
		}
		
		String identifierPart = lineBuff.substring(0, positionIDValSeparator);
		String valuePart = lineBuff.substring(positionIDValSeparator+1);
		if (DEBUG) { out.println("OriginalLine=" + str + "\nIdentifierPart=" + identifierPart + "\nvaluePart=" + valuePart); }

		// Identifier based switch - parse separately each configuration choice 
		if (identifierPart.equals(IDComponentName)) {
			ComponentName = valuePart;
			setFlagComponentName = true;
			
		} else if (identifierPart.equals(IDComponentImplementationClass)) {
			String[] separatedPairs = valuePart.split(SeparatorMultipleValues);
			ComponentImplementationClasses = Arrays.asList(separatedPairs);
			setFlagComponentImplementationClass = true;
			
		} else if (identifierPart.equals(IDComponentProvidedInterfaces)) {
			if (!valuePart.isEmpty()) {
				String[] separatedPairs = valuePart.split(SeparatorMultipleValues);
				for(String pairValue : separatedPairs) {
					if (parsePairValue(line, pairValue, ComponentProvidedInterfaces)) {
						// If pair error stop processing this line;
						break;
					}
				}
			}
			setFlagComponentProvidedInterfaces = true;
			
		} else if (identifierPart.equals(IDComponentRequiredInterfaces)) {
			if (!valuePart.isEmpty()) {
				String[] separatedPairs = valuePart.split(SeparatorMultipleValues);
				for(String pairValue : separatedPairs) {
					if (parsePairValue(line, pairValue, ComponentRequiredInterfaces)) {
						// If pair error stop processing this line;
						break;
					}
				}
			}
			setFlagComponentRequiredInterfaces = true;
			
		} else if (identifierPart.equals(IDComponentReentrancyLimit)) {
			int parsedValue = -1;
			try {
				parsedValue = Integer.parseInt(valuePart);
			} catch(NumberFormatException e) {
				ConfigurationError = true;
				out.println("Error number expected as parameter for entry " + IDComponentReentrancyLimit);
				return;
			}
			if (parsedValue <= 0) {
				ConfigurationError = true;
				out.println("Error invalid range of value (" + IDComponentReentrancyLimit + "). Use positive number.");
				return;
			}
			ComponentReentrancyLimit = parsedValue;
			setFlagComponentReentrancyLimit = true;
			
		} else if (identifierPart.equals(IDEnvValueSets)) {
			EnvValueSets = valuePart;
			setFlagEnvValueSets = true;
			
		} else if (identifierPart.equals(IDEnvTargetDir)) {
			String tmpEnvTargetDir = valuePart;
			
			// Adds ending '\' character if missing
			if (tmpEnvTargetDir.lastIndexOf( File.separator) != (tmpEnvTargetDir.length() - 1) ) {
				tmpEnvTargetDir = tmpEnvTargetDir + File.separator;
			}
			
			// Create output directory

			// Test if output directory exists or create them
			File targetDir = new File(tmpEnvTargetDir);
			if (targetDir.isFile()) {
				// Target directory exists as file --> can't be created as directory
				out.println("Error can't create target directory : " + targetDir);
				ConfigurationError = true;
				return;
			}
			// Target directory exists as dir or not exists at all
			try {
				targetDir.mkdirs();
			} catch (Exception e) { // Catches IOException and SecurityException
				out.println("Error while creating target dir : " + EnvTargetDir + "..." + targetDir.getAbsolutePath());
				out.println(e);
				ConfigurationError = true;
				return;
			}
			EnvTargetDir = tmpEnvTargetDir;
			setFlagEnvTargetDir = true;

		} else if (identifierPart.equals(IDEnvProtocol)) {
			// Protocol file parsing			
			String tmpEnvProcol = valuePart;
			if (!checkFileExists(tmpEnvProcol)) {
				out.println("Error - not existing protocol file : " + tmpEnvProcol);
				ConfigurationError = true;
				return;
			}
			EnvProtocol = tmpEnvProcol;
			setFlagEnvProtocol = true;
			
		} else if (identifierPart.equals(IDEnvRequierdItfcSettingMethod)) {
			EnvRequierdItfcSettingMethod = parseEnumRequiertItfsSettingMethod(valuePart);
			setFlagEnvRequierdItfcSettingMethod = true;
			
		} else if (identifierPart.equals(IDEnvProvidedItfsAccesMethod)) {
			EnvProvidedItfsAccesMethod = parseEnumProvidedInterfaceAccessMethod(valuePart);
			setFlagEnvProvidedItfsAccesMethod = true;

		} else if (identifierPart.equals(IDEnvGenerate)) {
			EnvGenerate = parseBoolean(valuePart, EnvGenerate);
			setFlagEnvGenerate = true;
		}
	}
	
	/**
	 * Test existence of file.
	 * 
	 * @param fileName File to test.
	 * @return Returns true if regular file with given name exists and is readable.
	 */
	private static boolean checkFileExists(String fileName) {
		File testedFile = new File(fileName);
		return testedFile.isFile() && testedFile.canRead();
	}

	/**
	 * Test if given character is whitespace and thus can be ignored.
	 *
	 * @param c Character to test
	 * @return True if given character is space tab or newline.
	 */
	private boolean isWhiteSpace(char c) {
		return (c == ' ') || (c == '\t') || (c == '\n'); 
	}
	
	/**
	 * Parse pair in format FirstEntry-SecondEntry into two parts and add this pair into result map.
	 *
	 * @param line Number of processed line. For error reporting purposes only.
	 * @param pair String with pair to parse.
	 * @param resultMap Map where add result.
	 * @return true if error found (invalid pair format), false otherwise
	 */
	private boolean parsePairValue(int line, String pair, Map<String, String> resultMap) {
		int positionPairValueSeparator = pair.indexOf(SeparatorPairValue);
		if (positionPairValueSeparator == -1) {
			if (line > 0) {
				out.println("Error at line " + Integer.toString(line) + " -  invalid value part format (pair: " + pair + ") ");
			} else {
				out.println("Error at " + Integer.toString(-line) + " command line parameter - invalid value part format (pair: " + pair + ") ");
			}
			ConfigurationError = true;
			return true;
		}
		String firstPairEntry = pair.substring(0, positionPairValueSeparator);
		String secondPairEntry = pair.substring(positionPairValueSeparator + 1);
		if (DEBUG) { out.println("OriginalPair=" + pair + "\nfirstPairEntry=" + firstPairEntry + "\nsecondPairEntry=" + secondPairEntry); }

		resultMap.put(firstPairEntry, secondPairEntry);
		return false;
		
	}

	/**
	 * Converts string into enum entries. Comparing ignore letter case.
	 *  
	 * @param value String with value we want to convert.
	 * @return Returns {@link RequiredInterfaceSettingMethod#SOFA2_CLIENT} if value contains {@link #RISM_SOFA2_CLIENT}, {@link RequiredInterfaceSettingMethod#SETTERS} otherwise.
	 */
	private RequiredInterfaceSettingMethod parseEnumRequiertItfsSettingMethod(String value) {
		if ( value.equalsIgnoreCase(RISM_SETTERS)) {
			return RequiredInterfaceSettingMethod.SETTERS; 
		}

		if ( value.equalsIgnoreCase(RISM_SOFA2_CLIENT)) {
			return RequiredInterfaceSettingMethod.SOFA2_CLIENT; 
		}

		if ( value.equalsIgnoreCase(RISM_FRACTAL)) {
			return RequiredInterfaceSettingMethod.FRACTAL; 
		}
		
		if (value.equalsIgnoreCase(RISM_FIELDS)) {
			return RequiredInterfaceSettingMethod.FIELDS;
		}

		// default
		return RequiredInterfaceSettingMethod.SETTERS;
	}

	/**
	 * Converts value into string representations. This function is inverse to {@link #parseEnumRequiertItfsSettingMethod}. 
	 *
	 * @param value Enum value to convert.
	 * @return String representation of given value that can be used in configuration files.
	 */
	private static String stringRequiertItfsSettingMethod(RequiredInterfaceSettingMethod value) {
		switch (value) {
		case /*RequiredInterfaceSettingMethod.*/SETTERS : return RISM_SETTERS;
		case /*RequiredInterfaceSettingMethod.*/SOFA2_CLIENT : return RISM_SOFA2_CLIENT;
		case /*RequiredInterfaceSettingMethod.*/FRACTAL : return RISM_FRACTAL;
		case /*RequiredInterfaceSettingMethod.*/FIELDS : return RISM_FIELDS;
		default: return "Unknown RequiredInterfaceSettingMethod value";
		}
	}
	
	/**
	 * Converts string into enum entries. Comparing ignore letter case.
	 *  
	 * @param value String with value we want to convert.
	 * @return Returns {@link ProvidedInterfaceAccessMethod#COMPONENT_CLASS} if value contains {@link #PIAM_COMPONENT_CLASS}, {@link ProvidedInterfaceAccessMethod#SEPARATE_CLASSES} otherwise.
	 */
	private static ProvidedInterfaceAccessMethod parseEnumProvidedInterfaceAccessMethod(String value) {
		if ( value.equalsIgnoreCase(PIAM_COMPONENT_CLASS)) {
			return ProvidedInterfaceAccessMethod.COMPONENT_CLASSES; 
		}

		if ( value.equalsIgnoreCase(PIAM_SEPARATE_CLASSES)) {
			return ProvidedInterfaceAccessMethod.SEPARATE_CLASSES; 
		}
		
		if ( value.equalsIgnoreCase(PIAM_SEPARATE_IMPLEMENTATION_CLASSES)) {
			return ProvidedInterfaceAccessMethod.SEPARATE_IMPLEMENTATION_CLASSES;
		}
		// default
		return ProvidedInterfaceAccessMethod.COMPONENT_CLASSES;
	}

	/**
	 * Converts string into boolean. Check if string is in given expected list of values.
	 * @see Configuration#BOOLEAN_VALUES_TRUE holds possible values in string to get true.
	 * @see Configuration#BOOLEAN_VALUES_FALSE holds possible values in string to get false.
	 *  
	 * @param value String to convert
	 * @param defaulValue Default value to return if unknown value in given string is preset
	 * @return 
	 */
	private static boolean parseBoolean(String value, boolean defaulValue) {
		// Check if true,... is specified
		for(String truePossitive : BOOLEAN_VALUES_TRUE) {
			if (truePossitive.equals(value)) {
				return true;
			}
		}
		
		// Check if false,... is specified
		for(String truePossitive : BOOLEAN_VALUES_FALSE) {
			if (truePossitive.equals(value)) {
				return false;
			}
		}
		return defaulValue;
	}
	
	/**
	 * Converts value into string representations. This function is inverse to {@link #parseEnumProvidedInterfaceAccessMethod}. 
	 *
	 * @param value Enum value to convert.
	 * @return String representation of given value that can be used in configuration files.
	 */
	private static String stringProvidedInterfaceAccessMethod(ProvidedInterfaceAccessMethod value) {
		switch (value) {
		case /*ProvidedInterfaceAccessMethod.*/COMPONENT_CLASSES : return PIAM_COMPONENT_CLASS;
		case /*ProvidedInterfaceAccessMethod.*/SEPARATE_CLASSES : return PIAM_SEPARATE_CLASSES;
		case /*ProvidedInterfaceAccessMethod.*/SEPARATE_IMPLEMENTATION_CLASSES : return PIAM_SEPARATE_IMPLEMENTATION_CLASSES;
		default: return "Unknown ProvidedInterfaceAccessMethod value";
		}
	}
	
	/**
	 * Text if all mandatory options has been set.
	 */
	private void checkMandatoryOptions() {

		if (setFlagComponentName==false) {
			out.println("Error - Mandatory option \"" + IDComponentName + "\" not specified by given configuration (both file and command line parameters).");
			ConfigurationError = true;
		}

		if (setFlagComponentImplementationClass==false) {
			out.println("Error - Mandatory option \"" + IDComponentImplementationClass + "\" not specified by given configuration (both file and command line parameters).");
			ConfigurationError = true;
		}

		if (setFlagEnvValueSets==false) {
			out.println("Error - Mandatory option \"" + IDEnvValueSets + "\" not specified by given configuration (both file and command line parameters).");
			ConfigurationError = true;
		}

		if (setFlagEnvProtocol==false) {
			out.println("Error - Mandatory option \"" + IDEnvProtocol + "\" not specified by given configuration (both file and command line parameters).");
			ConfigurationError = true;
		}

	}


	
	/**
	 * @return the componentName
	 */
	public String getComponentName() {
		return ComponentName;
	}

	/**
	 * @return Set of classes specified in componentImplementationClasses
	 */
	public List<String> getComponentImplementationClasses() {
		return ComponentImplementationClasses;
	}

	/**
	 * @return the componentProvidedInterfaces
	 */
	public Map<String, String> getComponentProvidedInterfaces() {
		return ComponentProvidedInterfaces;
	}

	/**
	 * @return the componentRequiredInterfaces
	 */
	public Map<String, String> getComponentRequiredInterfaces() {
		return ComponentRequiredInterfaces;
	}
	
	public int getComponentReentrancyLimit() {
		return ComponentReentrancyLimit;
	}

	/**
	 * @return the envValueSets
	 */
	public String getEnvValueSets() {
		return EnvValueSets;
	}

	/**
	 * @return the envTargetDir
	 */
	public String getEnvTargetDir() {
		return EnvTargetDir;
	}

	/**
	 * @return the envProcol
	 */
	public String getEnvProtocol() {
		return EnvProtocol;
	}
	
	/**
	 * @return the envGenerate
	 */
	public boolean getEnvGenerate() {
		return EnvGenerate;
	}
	

	
	/**
	 * @return the envRequierdItfcSettingMethod
	 */
	public RequiredInterfaceSettingMethod getEnvRequierdItfcSettingMethod() {
		return EnvRequierdItfcSettingMethod;
	}

	/**
	 * @return the envProvidedItfsAccesMethod
	 */
	public ProvidedInterfaceAccessMethod getEnvProvidedItfsAccesMethod() {
		return EnvProvidedItfsAccesMethod;
	}

	
	/**
	 * @return the configurationError
	 */
	public boolean isConfigurationError() {
		return ConfigurationError;
	}


	/**
	 * Prints parsed configuration into output. This output can be used as configuration file.
	 */
	public void printConfiguration() {
		out.println(IDComponentName + SeparatorIDValue + ComponentName);
		out.print(IDComponentImplementationClass + SeparatorIDValue);
		printConfigurationEntriesList(ComponentImplementationClasses);

		out.print(IDComponentProvidedInterfaces + SeparatorIDValue);
			printConfigurationPairs(ComponentProvidedInterfaces);

		out.print(IDComponentRequiredInterfaces + SeparatorIDValue);
			printConfigurationPairs(ComponentRequiredInterfaces);
		out.println(IDComponentReentrancyLimit + SeparatorIDValue + ComponentReentrancyLimit);

			out.println(IDEnvValueSets + SeparatorIDValue + EnvValueSets);
		out.println(IDEnvTargetDir + SeparatorIDValue + EnvTargetDir);
		out.println(IDEnvProtocol + SeparatorIDValue + EnvProtocol);
		
		out.println(IDEnvRequierdItfcSettingMethod + SeparatorIDValue + stringRequiertItfsSettingMethod(EnvRequierdItfcSettingMethod));
		out.println(IDEnvProvidedItfsAccesMethod + SeparatorIDValue + stringProvidedInterfaceAccessMethod(EnvProvidedItfsAccesMethod));

		out.println(IDEnvGenerate + SeparatorIDValue + EnvGenerate);

		if (ConfigurationError) {
			out.println("# Configuration contains errors");
		} else {
			out.println("# Configuration doen't contain errors");
		}
	}	
	
	/**
	 * Helper function for {@link #printConfiguration()}. Prints given list of entries in correct format.
	 * @param entryList List of entries to print out.
	 */
	private void printConfigurationEntriesList(List<String> entryList) {
		boolean first = true;
		for(String key : entryList) {
			if (!first) {
				out.print(SeparatorMultipleValues);
				first = false;
			}
			out.print(key);
		}
		out.println();
	}
	
	/**
	 * Helper function for {@link #printConfiguration()}. Prints given pair map in correct format.
	 * @param pairMap Map with parsed pair values that have to be printed out.
	 */
	private void printConfigurationPairs(Map<String, String> pairMap) {
		Set<String> pairKeySet = pairMap.keySet();
		boolean first = true;
		for(String key : pairKeySet) {
			if (!first) {
				out.print(SeparatorMultipleValues);
				first = false;
			}
			out.print(key + SeparatorPairValue + pairMap.get(key));
		}
		out.println();
	}
	
	
	/** For testing purposes */
	public static void main(String[] args) {
		String configurationFile = args[0];
		String[] overwritingsParameters = new String[args.length-1];
		// Removing first parameter ... Coping args[1 .. end] into overwritingsParameters[0 ++]
		for(int i = 1; i < args.length; i++) {
			overwritingsParameters[i-1] = args[i];
		}

		Configuration envGenConfig = new Configuration(overwritingsParameters, configurationFile, System.out);
		System.out.println("ParsedConfiguration");
		envGenConfig.printConfiguration();
	}
}
