package org.ow2.dsrg.fm.tbpjava.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

/**
 * Extremely simple classloader 
 *   which everytime when used, look into given directory and tries
 *   if class no in this directory.
 * 
 * Note: Used for loading generated stub classes.
 * Note: It's needed because standard Java classloaders cannot 
 *   load generated classes directly. It's not possible to actualize theirs classpath during runtime.
 * 
 * @author alf
 * 
 */
public class ClassloaderDir extends ClassLoader {
	private static final boolean DEBUG = false; /// Enables/disables debug output
	private String dir; /// Directory where lookup class files (base directory of package structure)
	
	private static final int MAX_CLASS_FILE_SIZE = 16 * 1024 * 1024;
	
	/**
	 * 
	 * @param dir Directory where class files are located. 
	 * @param parent Parent class loader where search for classes at first.
	 */
	public ClassloaderDir(String dir, ClassLoader parent) {
		super(parent);
		this.dir = dir;
	}
	
	public Class<?> findClass(String name) {
		byte[] b = loadClassData(name);
		if (b == null) {
			// Class not found
			return null;
		}
		return defineClass(name, b, 0, b.length);
	}

	/** 
	 * Gets class file in memory buffer 
	 * 
	 * @param name Full class name
	 * @return Buffer with class file content.
	 */
	private byte[] loadClassData(String name) {
		String fileName = dir + '/' + name.replace('.', '/') + ".class";
		if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":loadClassData - using fileName: " + fileName); }
		
		File fi = new File(fileName);
		if (fi.length() <= 0 || fi.length() > MAX_CLASS_FILE_SIZE) {
			// File to small or too big
			if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":loadClassData - using file size " + fi.length()); }
			return null;
		}
		byte[] buffer = new byte[(int) fi.length()];

		FileInputStream is;
		try {
			is = new FileInputStream(fileName);
		} catch (FileNotFoundException e) {
			return null;
		}
		try {
			is.read(buffer);
			is.close();
		} catch (IOException e) {
			if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":loadClassData - problem during reding class file"); }
			e.printStackTrace();
			return null;
		}
		return buffer;
	}

}
