package org.ow2.dsrg.fm.tbpjava.envgen;

import java.io.PrintStream;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbpjava.utils.Type2String;
import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.parsed.ParsedComponentSpecification;
import org.ow2.dsrg.fm.tbplib.parsed.TBPLimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAccept;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAlternative;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAssignment;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedEmit;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedIf;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeBinaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeLeafNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNull;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeSequence;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeUnaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedMethodDeclaration;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallel;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallelOr;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionBinaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionLeafNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionNaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionNull;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionUnaryNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedRepetition;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedReturn;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSequence;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSwitch;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSync;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedThreadContainerNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedValue;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedWhile;
import org.ow2.dsrg.fm.tbplib.parsed.TBPUnlimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.TBPParsedVisitor;

/**
 * <p>Checks whether all accepts and emits in provisions/reactions are correct.
 *  It means whether interface with given name is defined in configuration,
 *  and whether method with given name exists.
 *  
 *  <p>If error found methods return true if no problem found false is returned.
 * 
 * @author alf
 *
 */
public class VisitorTypeCheck implements TBPParsedVisitor<Boolean> {

	private Configuration config; /// Parsed configuration
	private PrintStream   out;

	private Set<String> processedProvisionMethods  = new HashSet<String>(); /// Holds strings "interfaceName.methodName" for methods that was successfully tested. Used in final testing whether all method are used in specification.
	private Set<String> declaredProvidedMethods  = new HashSet<String>(); /// Holds strings "interfaceName.methodName" for methods that was declared in the reaction section. Used in final testing whether all method from provided interface are defined.

	private Map<String, Class<?>> providedInterface2Class = new HashMap<String, Class<?>>(); /// Buffer that maps interface class names (e.g. java.util.map) to classes. If class is null, than class cannot be found, error was reported and succeeding errors should not be reported.
	private Map<String, Class<?>> requiredInterface2Class = new HashMap<String, Class<?>>(); /// Buffer that maps interface class names (e.g. java.util.map) to classes. If class is null, than class cannot be found, error was reported and succeeding errors should not be reported.
	private Set<String> invalidInterfaceNames  = new HashSet<String>(); /// Holds names of invalid provided interfaces (to prevent repetitive error reports)
	private Set<String> invalidInterfaceClassNames  = new HashSet<String>(); /// Holds names classes that cannot be load by Class.forName() (to prevent repetitive error reports)
	private Set<String> invalidInterfaceMethodName  = new HashSet<String>(); /// Holds strings "className.methodName" for methods that was not declared by interface (to prevent repetitive error reports)
	private Set<String> duplicateInterfaceMethodName  = new HashSet<String>(); /// Holds strings "className.methodName" for methods that was declared by interface more then once (with different parameters) (to prevent repetitive error reports)

	/**
	 * @param config Used configuration.
	 * @param out Stream where print results
	 */
	public VisitorTypeCheck(Configuration config, PrintStream out) {
		assert config != null;
		this.config = config;
		this.out = out;
	}
	/**
	 * TODO
	 * @param spec Parsed specification to check.
	 * @return true if error found and reported, false otherwise 
	 */
	public boolean completeCheck(ParsedComponentSpecification spec) {
		boolean result = false;
		result |= checkInterfaces();
		result |= checkSpecification(spec);
		result |= checkCompleteness();
		return result;
	}
	
	/** 
	  * Checks used provided and required interfaces, whether doesn't hold methods with same names.
	  * <br>
	  * <br>Note: Overloaded methods with same names cannot be distinguish by the TBP protocol.
	  * 
	  * @return True if multiple methods with same name detected, false if no problem found.
	  */
	public boolean checkInterfaces() {
		boolean result = false;
		for(Map.Entry<String, String> iface : config.getComponentProvidedInterfaces().entrySet()) {
			result |= checkInterface(iface.getKey(), iface.getValue());
		}
		for(Map.Entry<String, String> iface : config.getComponentRequiredInterfaces().entrySet()) {
			result |= checkInterface(iface.getKey(), iface.getValue());
		}
		return result;
	}
	
	private boolean checkInterface(String interfaceName, String interfaceClassName) {
		boolean result = false;

		// Find Class of provided interface 
		Class<?> interfaceClass = null;
		// Not in buffer
		try {
			//interfaceClass = Class.forName(Type2String.removeGenerics(interfaceClassName));
			interfaceClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(interfaceClassName));
			if (interfaceClass ==null) {
				throw new ClassNotFoundException();
			}
		} catch (ClassNotFoundException e) {
			if (!invalidInterfaceClassNames.contains(interfaceClassName)) {
				invalidInterfaceClassNames.add(interfaceClassName);
				out.println("Error - cannot load class (" + Type2String.removeGenerics(interfaceClassName) + ") where the " + interfaceName + " interface is defined.");
			}
			return true;
		}

		// Process one by one declared methods in the interface
		for(Method testedMethod : interfaceClass.getMethods()) {
			String id = interfaceClassName + '.' + testedMethod.getName();
			for(Method otherMethod : interfaceClass.getMethods()) {
			
				if (testedMethod.getName().equals( otherMethod.getName())
				    && (!testedMethod.equals(otherMethod)) ) {
					// Method with same name, but with different parameters
					if (!duplicateInterfaceMethodName.contains(id)) {
						duplicateInterfaceMethodName.add(id);
						out.println("Error - Interface " + interfaceName + " that is defined in class (" + Type2String.removeGenerics(interfaceClassName) + ") contains more methods with same name " + testedMethod.getName() + ". Such interfaces are not supported by TBP.");
					}
					result = true;
				}
				
				
				//TODO test whether other interfaces doesn't define same method with same parameters
				// This is possible be implementation .. but not supported by TBP
				//  ... in JPF there is not way how to distinguish to which interface events belongs. 
			}
		}
		return result;
	}
	
	/**
	 * Checks whether all method from provided interfaces are used in provisions and 
	 *   whether reaction section defines section for each method from interface.
	 * <br>
	 * Have to be called after {@link #checkSpecification(ParsedComponentSpecification)}
	 * <br>
	 * <br>Note: Prints only informative warnings.
	 *  
	 *  @return Always false.
	 */
	public boolean checkCompleteness() {
		//TODO
		return false;
	}
	
	
	/**
	 * Main checking procedure checks whole specification for used method and interface names.
	 * For more details see {@link VisitorTypeCheck}
	 * 
	 * @param spec Parsed specification to check.
	 * @return true if error found and reported, false otherwise 
	 */
	public boolean checkSpecification(ParsedComponentSpecification spec) {
		assert spec != null;
		
		boolean result = false;
		
		for(TBPParsedProvisionContainerNode node : spec.getProvisions()) {
			result |= node.visit(this);
		}

		for(TBPParsedImperativeNode node : spec.getReactions()) {
			result |= node.visit(this);
		}

		for(TBPParsedImperativeNode node : spec.getThreads()) {
			result |= node.visit(this);
		}
		
		return result;
	}
	/**
	 * Calls {@link TBPParsedNode#visit(TBPParsedVisitor) on all childrends of given node.
	 * <br>
	 * <br>Note: Always process all children.
	 * 
	 * @param node Node whose children should be visited.
	 * @return true if any of children report error (true), false if on error found
	 */
	private boolean processChilds(TBPNode node) {
		assert node != null;
		
		boolean result = false;
		for(TBPNode child : node.getChildren()) {
			assert child != null;
			TBPParsedNode parsedChild = (TBPParsedNode)child;
			result |= parsedChild.visit(this);
		}
		return result;
	}

	private enum providedEvenTypes {
		PROVIDED_CALL, /// Call on provided interface in provision section.
		PROVIDED_DECLARATION /// Declaration of reaction on provided event.
	}
	
	private boolean testAccept(String interfaceName, String methodName, providedEvenTypes event, boolean reportError) {
		// Check if interface exists
		String interfaceClassName = config.getComponentProvidedInterfaces().get( interfaceName);

		if (interfaceClassName == null) {
			if (reportError && !invalidInterfaceNames.contains(interfaceName)) {
				out.println("Error - unknown provided interface name " + interfaceName);
				invalidInterfaceNames.add(interfaceName);
			}
			return true;
		}
		
		// Find Class of provided interface 
		Class<?> interfaceClass = providedInterface2Class.get(interfaceClassName);
		if (interfaceClass == null) {
			// Not in buffer
			try {
				interfaceClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(interfaceClassName));
				//interfaceClass = Class.forName(Type2String.removeGenerics(interfaceClassName));

			} catch (ClassNotFoundException e) {
				if (reportError && !invalidInterfaceClassNames.contains(interfaceClassName)) {
					invalidInterfaceClassNames.add(interfaceClassName);
					out.println("Error - cannot load class (" + Type2String.removeGenerics(interfaceClassName) + ") where " + interfaceName + " interface is defined.");
					providedInterface2Class.put(interfaceClassName, null);
				}
				return true;
			}
			providedInterface2Class.put(interfaceClassName, interfaceClass);
		}
		
		// Find called Method
		Method calledMethod = null;
		for(Method m : interfaceClass.getMethods()) {
			if (m.getName().equals( methodName)) {
				calledMethod = m;
				break; // Stop searching
			}
		}
		if (calledMethod == null) {
			if (reportError && !invalidInterfaceMethodName.contains(interfaceClassName + '.' + methodName)) {
				invalidInterfaceMethodName.add(interfaceClassName + '.' + methodName);
				out.println("Error - provided intergace " + interfaceName + " (" + interfaceClassName + ")  doesn't contain method with name " + methodName + ".");
			}
			return true;
		}
		if (reportError) {
			if (event == providedEvenTypes.PROVIDED_CALL) {
				processedProvisionMethods.add(interfaceName + '.' + methodName);
			} else if (event == providedEvenTypes.PROVIDED_DECLARATION) {
				declaredProvidedMethods.add(interfaceName + '.' + methodName);
			} else {
				throw new RuntimeException("Unknown enum " + providedEvenTypes.class.getName() + " entry " + event);
			}
		}
		return false;
	}

	private boolean testEmit(String interfaceName, String methodName, boolean reportErrors) {
		// Check if interface exists
		String interfaceClassName = config.getComponentRequiredInterfaces().get( interfaceName);

		if (interfaceClassName == null) {
			if (reportErrors && !invalidInterfaceNames.contains(interfaceName)) {
				out.println("Error - unknown required interface name " + interfaceName);
				invalidInterfaceNames.add(interfaceName);
			}
			return true;
		}
		
		// Find Class of provided interface 
		Class<?> interfaceClass = requiredInterface2Class.get(interfaceClassName);
		if (interfaceClass == null) {
			// Not in buffer
			try {
				//interfaceClass = Class.forName(Type2String.removeGenerics(interfaceClassName));
				interfaceClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(interfaceClassName));
			} catch (ClassNotFoundException e) {
				if (reportErrors && !invalidInterfaceClassNames.contains(interfaceClassName)) {
					invalidInterfaceClassNames.add(interfaceClassName);
					out.println("Error - cannot load class (" + Type2String.removeGenerics(interfaceClassName) + ") where required interface " + interfaceName + " is defined.");
					requiredInterface2Class.put(interfaceClassName, null);
				}
				return true;
			}
			requiredInterface2Class.put(interfaceClassName, interfaceClass);
		}
		
		// Find called Method
		Method calledMethod = null;
		for(Method m : interfaceClass.getMethods()) {
			if (m.getName().equals( methodName)) {
				calledMethod = m;
				break; // Stop searching
			}
		}
		if (calledMethod == null) {
			if (reportErrors && !invalidInterfaceMethodName.contains(interfaceClassName + '.' + methodName)) {
				invalidInterfaceMethodName.add(interfaceClassName + '.' + methodName);
				out.println("Error - provided intergace " + interfaceName + " (" + interfaceClassName + ")  doesn't contain method with name " + methodName + ".");
			}
			return true;
		}
		
		return false;
	}

	@Override
	public Boolean visitLimitedReentrancy(TBPLimitedReentrancy node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedAccept(TBPParsedAccept node) {
		return testAccept(node.getInterface(), node.getMethod(), providedEvenTypes.PROVIDED_CALL, true);
	}


	@Override
	public Boolean visitParsedAlternative(TBPParsedAlternative node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedAssignment(TBPParsedAssignment node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedEmit(TBPParsedEmit node) {
		return testEmit(node.getInterface(), node.getMethod(), true);
	}

	@Override
	public Boolean visitParsedIf(TBPParsedIf node) {
		// Condition is not problem cannot contains any call (emit action)
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedImperativeBinaryNode(
			TBPParsedImperativeBinaryNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedImperativeBinaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedImperativeLeafNode(
			TBPParsedImperativeLeafNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedImperativeLeafNode(...) - Unexpected behavior: this class shoudn't be dirrectly in parse tree, only descendants -> maybe some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedImperativeNaryNode(
			TBPParsedImperativeNaryNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedImperativeNaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedImperativeNull(TBPParsedImperativeNull node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedImperativeSequence(
			TBPParsedImperativeSequence node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedImperativeUnaryNode(
			TBPParsedImperativeUnaryNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedImperativeUnaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedMethodDeclaration(TBPParsedMethodDeclaration node) {
		boolean result = processChilds(node);
		if ( testAccept(node.getInterface(), node.getMethod(), providedEvenTypes.PROVIDED_DECLARATION, false) == true) {
			// Not provided interface definition, check whether it isn't required interface
			if (testEmit(node.getInterface(), node.getMethod(), false) ==  false) {
				// Required interface call
				return result;
			}
		}

		result |= testAccept(node.getInterface(), node.getMethod(), providedEvenTypes.PROVIDED_DECLARATION, true);
		return result;
	}

	@Override
	public Boolean visitParsedNull(TBPParsedProvisionNull node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedParallel(TBPParsedParallel node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedParallelOr(TBPParsedParallelOr node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedProvisionBinaryNode(
			TBPParsedProvisionBinaryNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedProvisionBinaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedProvisionContainerNode(
			TBPParsedProvisionContainerNode node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedProvisionLeafNode(TBPParsedProvisionLeafNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedProvisionLeafNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedProvisionNaryNode(TBPParsedProvisionNaryNode node) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedProvisionNaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + node.getClass().getName());
	}

	@Override
	public Boolean visitParsedProvisionUnaryNode(
			TBPParsedProvisionUnaryNode parsedProvisionUnaryNode) {
		throw new RuntimeException(this.getClass().getSimpleName() + ":visitParsedProvisionNaryNode(...) - Unexpected behavior: visitor of abstract class called -> some child doesn't override visit method\n\tProblematic child : " + parsedProvisionUnaryNode.getClass().getName());
	}

	@Override
	public Boolean visitParsedRepetition(TBPParsedRepetition node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedReturn(TBPParsedReturn node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedSequence(TBPParsedSequence node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedSwitch(TBPParsedSwitch node) {
		boolean result = processChilds(node); // all cases and default one
		if (node.getValue() != null) {
			result |= node.getValue().visit(this);
		}
		return result;
	}

	@Override
	public Boolean visitParsedSync(TBPParsedSync node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedThreadContainerNode(
			TBPParsedThreadContainerNode node) {
		return processChilds(node);
	}

	@Override
	public Boolean visitParsedValue(TBPParsedValue node) {
		MethodCall mc = node.getMethodCall();
		if (mc != null) {
			return this.testEmit( mc.getInterface(), mc.getMethod(), true);
		} else {
			// Value is variable not method call 
			return false;
		}
	}

	@Override
	public Boolean visitParsedWhile(TBPParsedWhile node) {
		// Condition is not problem cannot contains any call (emit action)
		return processChilds(node);
	}

	@Override
	public Boolean visitUnlimitedReentrancy(TBPUnlimitedReentrancy node) {
		return processChilds(node);
	}

}
