package org.ow2.dsrg.fm.tbpjava.envgen;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbpjava.utils.Type2String;

/**
 * Generates stub classes for required interfaces.
 */
public class StubGenerator {
	private static final boolean DEBUG = false; // was true

	private String reqItfsName; /// TBP Interface name
	private Configuration config; /// Configuration of generator
	
	private PrintStream infoStream; /// Stream where prints errors and other info
	
	private String stubClassName; /// Contains name of generated stub class. Without package name.
	private String stubClassFullName; /// Contains full name of generated stub class. With package name.

	public static final String CLASS_NAME_SUFFIX = "_stub"; /// Suffix added into {@link #stubClassName} 

	public static final String METHOD_PARAMETER_PREFIX = "p"; /// Prefix used for parameter names in generated methods
	
	public static final String PROPERTY_OBTAIN_VALUES = "envValues";
	public static final String PROPERTY_OBTAIN_VALUES_TYPE = "EnvValueSets";
	
	
	
	/**
	 * Creates stub generator that generates one stub file. Stub file is created by calling {@link #generateStub()} method.
	 * 
	 * @param config Configuration of environment generator.
	 * @param reqItfsName Name of required interface for which file stub should be generated. 
	 * @param outStream Stream were print errors and other results.
	 */
	public StubGenerator(Configuration config, String reqItfsName, PrintStream outStream) {
		this.config = config;
		this.reqItfsName = reqItfsName;
		this.infoStream = outStream;
		
		stubClassFullName = getGeneratedClassFullName(reqItfsName, config);
		stubClassName = getClassName(stubClassFullName);
	}
	
	
	/**
	 * Generates stub file.
	 * @return Gets full name with package part of generated class. Or null if generation failed
	 */
	public String generateStub() {
		PrintStream stubStream = null; /// Stream used for generating resulting stub file
		
		// Generate name of file where stub will be generated
		String stubFileName = config.getEnvTargetDir() + stubClassFullName.replace('.', '/') + ".java";

		try {
			generateDirectoryStructureForFile(stubFileName);
			stubStream = new PrintStream(stubFileName);
		} catch (FileNotFoundException e) {
			infoStream.println("Generating stub " + stubClassName + " failed. File creation error.");
			if (DEBUG) { e.printStackTrace(infoStream); }

			return null;
		}
		
		Indenter indenter = new Indenter(stubStream); 

		// Obtaining "Class" for interface
		String reqitfClassName = config.getComponentRequiredInterfaces().get(reqItfsName);
		if (reqitfClassName == null) {
			infoStream.println("Generating stub " + stubClassName + " failed. Name of interface for required interface " + reqItfsName + " can't be resolved. ");
			stubStream.close();
			return null;
		}
		
		Class<?> reqitfClass = null;
		try {
			reqitfClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(reqitfClassName));
		} catch (ClassNotFoundException e) {
			infoStream.println("Generating stub " + stubClassName + " failed. Interface class " + reqitfClassName + " not found.");
			if (DEBUG) { e.printStackTrace(infoStream); }

			stubStream.close();
			return null;
			
		}
		Map<String, String> params =  new HashMap<String, String>();
		params.put(PROPERTY_OBTAIN_VALUES, PROPERTY_OBTAIN_VALUES_TYPE);
		
		// Output code generation
		genInitialClassCode(stubStream, indenter, stubClassName, reqitfClassName, reqitfClass.getTypeParameters(), config);
		genProperties(stubStream, indenter, params);
		genConstructor(stubStream, indenter, stubClassName, params);
		for(Method method : reqitfClass.getMethods()) {
			genMethod(stubStream, indenter, method, config.getComponentName(), reqitfClassName);
		}
		genEndingClassCode(stubStream, indenter);
		stubStream.close();
		
		return stubClassFullName;
	}
	
	/**
	 * Gets package part of full class name. (Remove all after last dot character, dot including)
	 * 
	 * @param objectFullName Full class name.
	 * @return Gets package part of full class name. Returns null if no package part (no dot in name).
	 */
	public static String getPackage(String objectFullName) {
		if (objectFullName == null) {
			return null;
		}
		
		objectFullName = Type2String.removeGenerics(objectFullName);
		
		int pos = objectFullName.lastIndexOf('.');
		if (pos == -1) {
			// "." not found ... no package part --> default package
			return null;
		}
		return objectFullName.substring(0, pos);
	}
	
	public static String getClassName(String objectFullName) {
		objectFullName = Type2String.removeGenerics(objectFullName);
		int pos = objectFullName.lastIndexOf('.');
		if (pos == -1) {
			// "." not found ... no package part --> name consists only from class name
			return objectFullName;
		}
		return objectFullName.substring(pos+1);
		
	}

	/**
	 * Prints code with imports and class declaration.
	 *
	 * @param stubStream Stream where print generated code. Point to resulting file.
	 * @param indenter Shared class for output indenting.
	 * @param stubClassName Name of generated stub class.
	 * @param reqItfsClassName Name of required interface (not class name), we generate stub.
	 * @param genericTypes Generic types used in header both Stub class and Required interface. 
	 * @param config Configuration of environment generator.
	 */
	private static void genInitialClassCode(final PrintStream stubStream, final Indenter indenter,
			final String stubClassName, final String reqItfsClassName, final Type[] genericTypes, final Configuration config) {

		String pkgReqitf = getPackage(reqItfsClassName);
		if (pkgReqitf != null) {
			indenter.indent(); stubStream.println("package " + getPackage(reqItfsClassName) + ";");
		} else {
			// default package 
			indenter.indent(); stubStream.println("// default package;");
		}
		
		indenter.indent(); stubStream.println();
		//indenter.indent(); stubStream.println("import gov.nasa.jpf.jvm.Verify;");
		indenter.indent(); stubStream.println("import org.ow2.dsrg.fm.tbpjava.envgen.EnvValueSets;");
		indenter.indent(); stubStream.println();
		for(String compImplClass : config.getComponentImplementationClasses()) {
			String pkgComponent = getPackage(compImplClass);
			if (pkgComponent != null) {
				indenter.indent(); stubStream.println("import " + pkgComponent + ".*;");
			} else {
				indenter.indent(); stubStream.println("// Component implementation in default package ... no import;");
			}
		}
		indenter.indent(); stubStream.println();
		indenter.indent(); stubStream.println();
		indenter.indent(); stubStream.println("public class " + stubClassName + Type2String.getGenericTypeDefinition(genericTypes) + " implements " + Type2String.removeGenerics(reqItfsClassName) + Type2String.getGenericsTypeNames(genericTypes) + " {");
		indenter.addLevel();
		indenter.indent(); stubStream.println();
	}

	/**
	 * Prints code declaring properties in created stub class.
	 * 
	 * @param stubStream Stream where print generated code. Point to resulting file.
	 * @param indenter Shared class for output indenting.
	 * @param properties List of properties to declare. First string of entry is declared variable name, second is declared variable type.  
	 */
	private static void genProperties(PrintStream stubStream, Indenter indenter, Map<String, String> properties) {
		for(Entry<String, String> entry : properties.entrySet()) {
			final String propName = entry.getKey();
			final String className = entry.getValue();
			indenter.indent(); stubStream.println("private " + className + " " + propName + ";");
			indenter.indent(); 
		}
	
	}

	/**
	 * Prints code for stub class constructor with given list of parameters. Generated constructors set 
	 *  stub class properties with same names as parameters.
	 * 
	 * @param stubStream Stream where print generated code. Point to resulting file.
	 * @param indenter Shared class for output indenting.
	 * @param stubClassName Name of generated stub class.
	 * @param parameters List of parameters to declare. First string of entry is declared variable name, second is declared variable type.  
	 */
	private static void genConstructor(PrintStream stubStream, Indenter indenter, String stubClassName, Map<String, String> parameters) {
		Set<String> propertyNames = parameters.keySet();

		indenter.indent(); stubStream.println();
		indenter.indent(); 

		// Print constructor header
		stubStream.print("public " + stubClassName + "(");
		boolean firstParam = true;
		for(String propName : propertyNames) {
			if ( firstParam ) {
				firstParam = false;
			} else {
				stubStream.print(", ");
			}
			stubStream.print( parameters.get(propName) + " " + propName);
		}
		stubStream.println(") {");

		indenter.addLevel();

		// Setting properties code
		for(String propName : propertyNames) {
			indenter.indent(); stubStream.println("this." + propName + " = " + propName + ";");
		}
		
		indenter.removeLevel();
		indenter.indent(); stubStream.println("}");
		
	}
	
	/**
	 * Prints code for one method of stub class. Generated method ignore given parameters and returns
	 *  any from objects (with proper type) in given EnvValueDB class.
	 * 
	 * @param stubStream Stream where print generated code. Point to resulting file.
	 * @param indenter Shared class for output indenting.
	 * @param method Method from given required interface we generate stub class method. This method is used for gaining name, parameter a return value types.  
	 * @param componentName Tested component name form TBP specification.
	 * @param itfsName Name of required (from TBP Specification )interface, we generate stub for.
	 */
	private static void genMethod(PrintStream stubStream, Indenter indenter, Method method, String componentName, String itfsName) {
		indenter.indent(); stubStream.println();
		
		Annotation[] annotations = method.getAnnotations();
		for(Annotation annot : annotations) {
			indenter.indent(); stubStream.println(annot.toString());
		}
		// Print method header
		int methodModifiers = method.getModifiers();
		// remove abstract modifier
		methodModifiers = methodModifiers & (~Modifier.ABSTRACT);
		indenter.indent(); 
			stubStream.print( Modifier.toString(methodModifiers) + " ");
			stubStream.print( Type2String.getGenericTypeDefinition(method.getTypeParameters()) + " ");
			stubStream.print( Type2String.getTypeName(method.getGenericReturnType()) + " ");
			stubStream.print( method.getName() + "(");
			// Print parameters
			Type[] params = method.getGenericParameterTypes();
			for(int i = 0; i < params.length; i++) {
				if ( i > 0 ) {
					// Not first parameter
					stubStream.print(", ");
				}
				stubStream.print(Type2String.getTypeName(params[i]) + " " + METHOD_PARAMETER_PREFIX + Integer.toString(i));
			}
			stubStream.print(")");
			// Print Throws clause
			Type[] exceptions = method.getGenericExceptionTypes();
			if ( exceptions.length > 0 ) {
				stubStream.print(" throws");
				for(int i1 = 0; i1 < exceptions.length; i1++) {
					if ( i1 > 0 ) {
						// Not first parameter
						stubStream.print(",");
					}
					stubStream.print(" " + Type2String.getTypeName(exceptions[i1]));
				}
			}

			stubStream.println(" {");
			
		indenter.addLevel();
		indenter.indent(); 
			stubStream.print("return" + genMethodReturnValue(method, componentName, itfsName));
			stubStream.print(";");
			stubStream.println();
		indenter.removeLevel();
		indenter.indent(); stubStream.println("}");
	}

	/**
	 * Creates string with command for retrieving return value of method.
	 * 
	 * @param method Method from given required interface we generate stub class method. This method is used for gaining name, parameter a return value types.  
	 * @param componentName Tested component name form TBP specification.
	 * @param itfsName Name of required (from TBP Specification )interface, we generate stub for.
	 * @return String with command that gets return value for specified method. 
	 */	
	private static String genMethodReturnValue(Method method, String componentName, String itfsName) {
		String result = genCodeObtainingValue( method.getGenericReturnType(), componentName, itfsName, method.getName()); 
		if (result.isEmpty() == false) {
			result = ' ' + result;
		}
		return result;
	}

	/**
	 * Creates string with command for retrieving return value of method.
	 * 
	 * @param type Method from given required interface we generate stub class method. This method is used for gaining name, parameter a return value types.  
	 * @param componentName Tested component name form TBP specification.
	 * @param itfsName Name of required (from TBP Specification )interface, we generate stub for.
	 * @param methodName Name of method that wants value. (Method that in which will generated code take place).
	 * @return String with command that gets return value for specified method. 
	 */	
	public static String genCodeObtainingValue(Type type, String componentName, String itfsName, String methodName) {
	
		StringBuffer retVal = new StringBuffer();
		
		if (type.equals(void.class)) {
			// No return value -> get empty String
			return "";
		}
		if (envValSetUseObject(type)) {
			// getObject method call ... object (and not string) ...
			
			// Overcasting
			retVal.append('(');
			retVal.append(Type2String.getTypeName(type));
			retVal.append(')');
			retVal.append(' ');
		}

		retVal.append(PROPERTY_OBTAIN_VALUES);
		retVal.append('.');
		if (type.equals(boolean.class)) {
			retVal.append("getBoolean");
		} else if (type.equals(byte.class)) {
			retVal.append("getByte");
		} else if (type.equals(short.class)) {
			retVal.append("getShort");
		} else if (type.equals(int.class)) {
			retVal.append("getInt");
		} else if (type.equals(long.class)) {
			retVal.append("getLong");
		} else if (type.equals(char.class)) {
			retVal.append("getChar");
		} else if (type.equals(float.class)) {
			retVal.append("getFloat");
		} else if (type.equals(double.class)) {
			retVal.append("getDouble");
		} else if (type.equals(String.class)) {
			retVal.append("getString");
		} else {
			retVal.append("getObject");
		}
		
		// Generate parameters
		retVal.append("(\"");
		if (envValSetUseObject(type)) {
			// This first parameter is used only for "getObject" method 
			retVal.append(Type2String.getTypeName(type));
			retVal.append("\", \"");
		}
		retVal.append(componentName);
		retVal.append("\", \"");
		retVal.append(itfsName);
		retVal.append("\", \"");
		retVal.append(methodName);
		retVal.append("\")");
		return retVal.toString();
	}
	/**
	 * Test if given parameter represent primitive type.
	 * @param type Type to check.
	 * @return True if type parameter represents primitive type
	 */
	private static boolean isTypePrimitive(Type type) {
		return 
			type.equals(boolean.class) ||
			type.equals(byte.class) ||
			type.equals(short.class) ||
			type.equals(int.class) ||
			type.equals(long.class) ||
			type.equals(char.class) ||
			type.equals(float.class) ||
			type.equals(double.class);
	}
	/**
	 * Check if given type should be obtained be calling {@link EnvValueSets#getObject(String, String, String, String)}.
	 * It means there in no specialized method for given type.
	 * 
	 * @param type Type to check
	 * @return Return true if given type should be obtained from EnvValueSets class by generic getObjectMethod
	 */
	private static boolean envValSetUseObject(Type type) {
		return ! (isTypePrimitive(type) || type.equals(String.class)); 
	}
	
	/**
	 * Prints closing bracket for stub class definition.
	 *  
	 * @param stubStream Stream where print generated code. Point to resulting file.
	 * @param indenter Shared class for output indenting.
	 */
	private static void genEndingClassCode(PrintStream stubStream, Indenter indenter) {
		indenter.indent(); stubStream.println();
		indenter.removeLevel();
		indenter.indent(); stubStream.println("}");
		indenter.indent(); stubStream.println();
	}
	

	/**
	 * Generate full name for generated stub class.
	 * @param reqItfsName Full name of interface 
	 * @return Return name of generated stub class
	 */
	public static String getGeneratedClassFullName(String reqItfsName, Configuration config) {
		String reqitfClassPackageName = getPackage(config.getComponentRequiredInterfaces().get(reqItfsName));
		if (reqitfClassPackageName == null) {
			// default package
			reqitfClassPackageName = "";
		} else {
			// not default package
			reqitfClassPackageName = reqitfClassPackageName + ".";
		}
		return reqitfClassPackageName + reqItfsName + CLASS_NAME_SUFFIX;
	}

	/**
	 * Creates missing directories for given file. (Recursively creates missing directory path).
	 * @param fileName Name of file you are trying create (Directory is extracted from this name)
	 */
	public static void generateDirectoryStructureForFile(String fileName) {
		File file = new File(fileName);
		File dir = file.getParentFile(); // Directory part of path
		dir.mkdirs();
	}
}

