package org.ow2.dsrg.fm.tbpjava.envgen;

import java.util.List;

import org.ow2.dsrg.fm.tbplib.parsed.*;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.TBPParsedCheckingVisitor;

/**
 * Visitor that converts given parsed AST provision 
 * subtree into string form.
 *
 * @author Alfifi
 *
 */
public class ProvisionToString extends TBPParsedCheckingVisitor<StringBuffer> {

	public static final String INVALID_INPUT = "???";
	public static final String NULL = "null";
	
	// Formating output
	public enum Style {
		STYLE_SPACY,  /// Add spaces to create more readable code
		STYLE_COMPACT /// Uses only necessary spaces
	}
	
	private Style spaceStyle;
	private boolean printAnnotations; 
	
	public ProvisionToString(Style spaceStyle, boolean printAnnotations) {
		this.spaceStyle = spaceStyle;
		this.printAnnotations = printAnnotations || true;
	}
	

	private void addSpace(StringBuffer buffer) {
		if (spaceStyle==Style.STYLE_SPACY) {
			buffer.append(' ');
		}
	}

	private StringBuffer processLeafNode(TBPParsedProvisionLeafNode node, String nodeName) {
		StringBuffer result = new StringBuffer();
		if (node == null) {
			result.append(INVALID_INPUT);
		} else {
			if (printAnnotations && node.getAnnotation() != null) {
				result.append(node.getAnnotation().toString());
				result.append(' ');
			}
			result.append(nodeName);
		}
		
		return result;
	
	}
	
	private StringBuffer processUnaryOperator(TBPParsedProvisionUnaryNode node, String operatorName) {
		StringBuffer result = new StringBuffer();
		if (node == null) {
			result = new StringBuffer();
			result.append(INVALID_INPUT);
			return result;
		} 

		// Process subtree
		TBPParsedNode childNode = (TBPParsedNode) node.getChild();
		
		if (childNode == null) {
			result.append(INVALID_INPUT);
			addSpace(result);
		} else {  
			// Generate parenthesis around subtree if needed
			if (childNode instanceof TBPParsedProvisionLeafNode) {
				result.append(childNode.visit(this));
				addSpace(result);
			} else {
				result.append('(');
				addSpace(result);
				result.append( childNode.visit(this));
				addSpace(result);
				result.append(')');
			}
		}
		if (printAnnotations && node.getAnnotation() != null) {
			// Add annotation just before unary suffix operator
			result.append(node.getAnnotation().toString());
			result.append(' ');
		}
		result.append(operatorName);
		return result;
	}

	// Special version for provision node .. uses prefix suffix, doesn't uses parenthesis, doesn't process Annoations
	// Used only for ContainerNode
	private StringBuffer processUnaryPrefixSuffix(TBPParsedProvisionUnaryNode node, String prefixString, String suffixString) {
		StringBuffer result = new StringBuffer();
		if (node == null) {
			result = new StringBuffer();
			result.append(INVALID_INPUT);
			return result;
		} 

		result.append(prefixString);
		addSpace(result);

		// Process subtree
		TBPParsedNode childNode = (TBPParsedNode) node.getChild();
		
		if (childNode == null) {
			result.append(INVALID_INPUT);
		} else {
			result.append( childNode.visit(this));
		}

		addSpace(result);
		result.append(suffixString);
		return result;
	}
	
	// There is no shared common predecessor of Binary and NAry node (processing can be same)
	private StringBuffer processNAryOperator(TBPParsedNode node, String operatorName) {
		StringBuffer result = new StringBuffer();

		if (node == null) {
			// Invalid input
			result = new StringBuffer();
			result.append(INVALID_INPUT);
			return result;
		};

		if (printAnnotations && node.getAnnotation() != null) {
			result.append(node.getAnnotation().toString());
			addSpace(result);
			result.append('(');
		}

		// Process subtree
		int childCount = node.getChildCount();
		
		// For all childrens
		for(int i = 0; i < childCount; i++) {
			TBPParsedNode childNode = (TBPParsedNode) node.getChild(i);
			if (childNode == null) {
				result.append(INVALID_INPUT);
			} else {
				// Generate parenthesis around subtree if needed ... child not leaf and uses different operator
				if (childNode instanceof TBPParsedProvisionLeafNode ||
						childNode.getClass().equals(node.getClass()) ) { // <- linearize binary operators into NAry
					result.append(childNode.visit(this));
				} else {
					result.append('(');
					addSpace(result);
					result.append(childNode.visit(this));
					addSpace(result);
					result.append(')');
				}
			}
			
			// Add operator if not last
			if (i < childCount - 1) {
				addSpace(result);
				result.append(operatorName);
				addSpace(result);
			}
		}
		
		if (printAnnotations && node.getAnnotation() != null) {
			result.append(')');
		}
		return result;
	}
	
	public static String methodCall2String(MethodCall mc) {
		if (mc == null) {
			return "";
		}
		
		StringBuffer result = new StringBuffer(20);
		result.append('?');
		result.append(mc.getFullname());
		// Process parameters
		result.append('(');
		List<String> params = mc.getParamDecl();
		if (params != null && params.isEmpty() == false) {
			boolean first = true;
			for(String param : params) {
				if (first) {
					first = false;
				} else {
					result.append(", ");
				}
				
				result.append(param);
			}
		}
		result.append(')');
		
		// Process return value
		String retVal = mc.getReturnValue();
		if (retVal != null && retVal.isEmpty() == false) {
			result.append(':');
			result.append(retVal);
		}

		return result.toString();
	}
	
	
	@Override
	public StringBuffer visitLimitedReentrancy(TBPLimitedReentrancy node) {
		int limit = 0;
		if (node != null) {
			limit = node.getLimit();
		}
		
		return processUnaryOperator(node, "|" + Integer.toString(limit));
	}

	@Override
	public StringBuffer visitParsedAlternative(TBPParsedAlternative node) {
		return processNAryOperator(node, "+");
	}

	@Override
	public StringBuffer visitParsedNull(TBPParsedProvisionNull node) {
		return processLeafNode(node, NULL);
	}

	@Override
	public StringBuffer visitParsedParallel(TBPParsedParallel node) {
		return processNAryOperator(node, "|");
	}

	@Override
	public StringBuffer visitParsedParallelOr(TBPParsedParallelOr node) {
		return processNAryOperator(node, "||");
	}

	@Override
	public StringBuffer visitParsedProvisionContainerNode(TBPParsedProvisionContainerNode node) {
		String provisionName = "";
		if (node != null) {
			provisionName = node.getName();
		}

		String annotation = "";
		if (printAnnotations && node.getAnnotation() != null) {
			annotation = " " + node.getAnnotation().toString();
		}

		return processUnaryPrefixSuffix(node, "provision " + provisionName + annotation + " {", "}");
	}

	@Override
	public StringBuffer visitParsedAccept(TBPParsedAccept node) {
		String callingCode = INVALID_INPUT;
		if (node != null) {
			callingCode = methodCall2String(node.getMethodCall());
		}
		return processLeafNode(node, callingCode);
	}

	@Override
	public StringBuffer visitParsedRepetition(TBPParsedRepetition node) {
		return processUnaryOperator(node, "*");
	}

	@Override
	public StringBuffer visitParsedSequence(TBPParsedSequence node) {
		return processNAryOperator(node, ";");
	}

	@Override
	public StringBuffer visitUnlimitedReentrancy(TBPUnlimitedReentrancy node) {
		return processUnaryOperator(node, "|*");
	}
}
