package org.ow2.dsrg.fm.tbpjava.envgen;

import java.util.Map;
import java.util.HashMap;

import gov.nasa.jpf.jvm.Verify;


/**
 * Implementations of this interface define sets of values that are used as sources for non-deterministic selection of method parameters and return values during verification of components with JPF.
 */
public abstract class EnvValueSets {
	/// Mark is in final generated code will be generic types used.
	static final boolean USE_GENERIC_TYPE = false; 
	
	/**
	 * Maps strings "typeName:componentName:javaItfName:methodName" to an array of objects with given type.
	 * ComponentName, javaItfName and methodName parts of name are optional (i.e. can be empty)
	 */
	private final Map<String, Object> valueSets = new HashMap<String, Object>();
	
	public static final String TYPE_NAME_BOOLEAN = "Boolean";
	public static final String TYPE_NAME_BYTE = "Byte";
	public static final String TYPE_NAME_SHORT = "Short";
	public static final String TYPE_NAME_INT = "Int";
	public static final String TYPE_NAME_LONG = "Long";
	public static final String TYPE_NAME_CHAR = "Char";
	public static final String TYPE_NAME_FLOAT = "Float";
	public static final String TYPE_NAME_DOUBLE = "Double";
	public static final String TYPE_NAME_STRING = "String";
	
	public static final boolean IMPLICIT_RETURN_VALUE_BOOLEAN = false;
	public static final byte IMPLICIT_RETURN_VALUE_BYTE = 0;
	public static final short IMPLICIT_RETURN_VALUE_SHORT = 0;
	public static final int IMPLICIT_RETURN_VALUE_INT = 0;
	public static final long IMPLICIT_RETURN_VALUE_LONG = 0;
	public static final char IMPLICIT_RETURN_VALUE_CHAR = ' ';
	public static final float IMPLICIT_RETURN_VALUE_FLOAT = 0;
	public static final double IMPLICIT_RETURN_VALUE_DOUBLE = 0.0;
	public static final String IMPLICIT_RETURN_VALUE_STRING = "";
	public static final Object IMPLICIT_RETURN_VALUE_OBJECT = null;
	
	/**
	 * Value to be specified in the "methodName" entry if given value should be used in call of component constructor.
	 */
	public static final String COMPONENT_CONSTRUCTOR_PARAMETER = "__CONSTRUCTOR__";

	private final Object[] getValueSet(String typeName, String componentName, String javaItfName, String methodName) {
		Object[] objs;

		// try to find a set of values specific to a method
		objs = (Object[]) valueSets.get(typeName+":"+componentName+":"+javaItfName+":"+methodName);
		if (objs == null) {
			// try to find a set of values specific to an interface
			objs = (Object[]) valueSets.get(typeName+":"+componentName+":"+javaItfName+":");
			if (objs == null) {
				// try to find a set of values specific to a component
				objs = (Object[]) valueSets.get(typeName+":"+componentName+"::");
				if (objs == null) {	
					// try to find a set of values specific to an application
					objs = (Object[]) valueSets.get(typeName+":::");
				}
			}
		}
		
		return objs;
	}

	private final void addValueSet(String typeName, String componentName, String javaItfName, String methodName, Object[] objs) {
		valueSets.put(typeName+":"+componentName+":"+javaItfName+":"+methodName, objs);
		valueSets.put(typeName+":"+componentName+":"+javaItfName+":", objs);
		valueSets.put(typeName+":"+componentName+"::", objs);
		valueSets.put(typeName+":::", objs);
	}
	

	public final synchronized boolean getBoolean(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_BOOLEAN, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_BOOLEAN; 
		}
	
		Boolean obj = (Boolean) objs[Verify.random(objs.length-1)];
		boolean b = obj.booleanValue();
		return b;
	}
		
	public final synchronized void putBooleanSet(String componentName, String javaItfName, String methodName, boolean[] vals) {
		if (vals == null) {
			return;
		}

		Boolean[] objs = new Boolean[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Boolean(vals[i]);
		}
	
		addValueSet(TYPE_NAME_BOOLEAN, componentName, javaItfName, methodName, objs);
	}
	
	public final synchronized byte getByte(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_BYTE, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_BYTE; 
		}
	
		Byte obj = (Byte) objs[Verify.random(objs.length-1)];
		byte b = obj.byteValue();
		return b;
	}
		
	public final synchronized void putByteSet(String componentName, String javaItfName, String methodName, byte[] vals) {
		if (vals == null) {
			return;
		}

		Byte[] objs = new Byte[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Byte(vals[i]);	
		}
	
		addValueSet(TYPE_NAME_BYTE, componentName, javaItfName, methodName, objs);
	}

	public final synchronized short getShort(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_SHORT, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_SHORT; 
		}
		
		Short obj = (Short) objs[Verify.random(objs.length-1)];
		short s = obj.shortValue();
		return s;
	}
		
	public final synchronized void putShortSet(String componentName, String javaItfName, String methodName, short[] vals) {
		if (vals == null) {
			return;
		}

		Short[] objs = new Short[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Short(vals[i]);
		}
	
		addValueSet(TYPE_NAME_SHORT, componentName, javaItfName, methodName, objs);
	}

	public final synchronized int getInt(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_INT, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_INT; 
		}
	
		Integer obj = (Integer) objs[Verify.random(objs.length-1)];
		int i = obj.intValue();
		return i;
	}
		
	public final synchronized void putIntSet(String componentName, String javaItfName, String methodName, int[] vals) {
		if (vals == null) {
			return;
		}

		Integer[] objs = new Integer[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Integer(vals[i]);
		}
	
		addValueSet(TYPE_NAME_INT, componentName, javaItfName, methodName, objs);
	}
	
	public final synchronized long getLong(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_LONG, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_LONG; 
		}
	
		Long obj = (Long) objs[Verify.random(objs.length-1)];
		long l = obj.longValue();
		return l;
	}
		
	public final synchronized void putLongSet(String componentName, String javaItfName, String methodName, long[] vals) {
		if (vals == null) {
			return;
		}

		Long[] objs = new Long[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Long(vals[i]);
		}
		
		addValueSet(TYPE_NAME_LONG, componentName, javaItfName, methodName, objs);
	}
	
	public final synchronized char getChar(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_CHAR, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_CHAR; 
		}
	
		Character obj = (Character) objs[Verify.random(objs.length-1)];
		char c = obj.charValue();
		return c;
	}
		
	public final synchronized void putCharSet(String componentName, String javaItfName, String methodName, char[] vals) {
		if (vals == null) {
			return;
		}

		Character[] objs = new Character[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Character(vals[i]);
		}
	
		addValueSet(TYPE_NAME_CHAR, componentName, javaItfName, methodName, objs);
	}

	public final synchronized float getFloat(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_FLOAT, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_FLOAT; 
		}
	
		Float obj = (Float) objs[Verify.random(objs.length-1)];
		float f = obj.floatValue();	
		return f;
	}
		
	public final synchronized void putFloatSet(String componentName, String javaItfName, String methodName, float[] vals) {
		if (vals == null) {
			return;
		}

		Float[] objs = new Float[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Float(vals[i]);
		}
	
		addValueSet(TYPE_NAME_FLOAT, componentName, javaItfName, methodName, objs);
	}
	
	public final synchronized double getDouble(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_DOUBLE, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_DOUBLE; 
		}
	
		Double obj = (Double) objs[Verify.random(objs.length-1)];
		double d = obj.doubleValue();
		return d;
	}
		
	public final synchronized void putDoubleSet(String componentName, String javaItfName, String methodName, double[] vals) {
		if (vals == null) {
			return;
		}

		Double[] objs = new Double[vals.length];
		for (int i = 0; i < vals.length; i++) {
			objs[i] = new Double(vals[i]);
		}
	
		addValueSet(TYPE_NAME_DOUBLE, componentName, javaItfName, methodName, objs);
	}

	public final synchronized String getString(String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(TYPE_NAME_STRING, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_STRING; 
		}
	
		String s = (String) objs[Verify.random(objs.length-1)];
		return s;
	}
		
	public final synchronized void putStringSet(String componentName, String javaItfName, String methodName, String[] vals) {
		if (vals == null) {
			return;
		}

		addValueSet(TYPE_NAME_STRING, componentName, javaItfName, methodName, vals);
	}

	public final synchronized Object getObject(String typeName, String componentName, String javaItfName, String methodName) {
		Object[] objs = getValueSet(typeName, componentName, javaItfName, methodName);
		
		// default value
		if ((objs == null) || (objs.length == 0)) {
			return IMPLICIT_RETURN_VALUE_OBJECT; 
		}
	
		Object o = objs[Verify.random(objs.length-1)];
		return o;
	}
		
	public final synchronized void putObjectSet(String componentName, String javaItfName, String methodName, Object[] vals) {
		if (vals == null || vals.length == 0) {
			return;
		}

		putObjectSet(vals[0].getClass().getSimpleName(), componentName, javaItfName, methodName, vals);
	}

	public final synchronized void putObjectSet(String typeName, String componentName, String javaItfName, String methodName, Object[] vals) {
		if (vals == null) {
			return;
		}

		addValueSet(typeName, componentName, javaItfName, methodName, vals);
	}

}
