package org.ow2.dsrg.fm.tbpjava.envgen;

import java.io.PrintStream;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import org.ow2.dsrg.fm.tbpjava.EnvGenerator;
import org.ow2.dsrg.fm.tbpjava.EnvGenerator.EnvironmentDescription;
import org.ow2.dsrg.fm.tbpjava.envgen.ProvisionToString.Style;
import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbpjava.utils.LineCouningOutputStream;
import org.ow2.dsrg.fm.tbpjava.utils.Type2String;
import org.ow2.dsrg.fm.tbplib.parsed.TBPLimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAccept;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAlternative;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallel;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedParallelOr;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionNull;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedRepetition;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSequence;
import org.ow2.dsrg.fm.tbplib.parsed.TBPUnlimitedReentrancy;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.TBPParsedCheckingVisitor;

/**
 * Generates resulting environment class for given ProvisionContainerNode by calling {@link #generateFile()}. 
 * Translates trees of parsed provision nodes into java code of corresponding environment.
 * <br>
 * Based on visitor pattern. Subtrees with same types of nodes are processed by special routines. (Makes n-ary operators from binary) 
 */
public class CodeGenerator extends TBPParsedCheckingVisitor<Object> {
	private static final boolean DEBUG = false; // was true

	private static final boolean EXCEPTIONS_ACCEPT_TRY_CATCH_BLOCKS = false; /// If true, surround calls on required interfaces by try - catch block where catch all checked exceptions
	private static final boolean REPETITION_AS_WHILE = false; /// How to generate code for the reperetion operator (whether as FOR cycle or while cycle)

	private boolean errorFound = false; /// Flag that marks that we found error during processing subtree
	
	private Indenter indenter; /// Class used for printing correct indentation in outFile
	private LineCouningOutputStream outFile; /// Stream used for printing command into result file 

	private PrintStream infoStream = System.out; /// Stream used for printing processing info and errors 
	
	private Configuration config = null; /// Configuration of whole Environment generator
	private boolean printAnnotations = true; /// Flag sets if in result generated protocol should be preserved annotations from protocol description.
	private ProvisionToString prov2String = null; /// Object used for converting provision tree into text representation that is outputed into result souce code comments.
	private EnvironmentDescription envDesc = null; /// Where code generator store mappings with parameters of provision calls. For each generated provided call, {@link CodeGenerator} stores line of this call and parameters from the protocol / and expected return values
	
	private int cntThread = 0;     /// Number of created thread variables
	private int cntOrParallel = 0; /// Number of create temporary variables for or parallel or limited reentrancy .. has to share same variable, --> use same counter
	private int cntAlternative = 0;
	private int cntRepetition = 0; // Number of created temporary variables in repetition operator 

	public static final String AlternativeVariableName = "alternativeChoice";
	public static final String ParallelOrVariableName  = "parallel";
	public static final String RepetitionVariableName  = "repetition";

	public static final String EnvValueSetsPropertyName = "envValues"; /// Name of property in generated class where instance of {@link EnvValueSets#} is stored
	
	public static final int REPETITION_LIMIT = 3; /// Number of loops in that are unroll repetition operators.  
	
	
	public CodeGenerator(PrintStream infoStream, LineCouningOutputStream outFile, Indenter indenter, EnvironmentDescription envDesc, Configuration config) {
		this.infoStream = infoStream;
		this.outFile = outFile;
		this.indenter = indenter;
		this.envDesc = envDesc;
		this.config = config;
		this.prov2String = new ProvisionToString(Style.STYLE_COMPACT, printAnnotations);
	}
	
	public static String patchName(String src) {
        src = src.replaceAll("<", "_");
        src = src.replaceAll(">", "_");
        src = src.replaceAll(" ", "_");
        src = src.replaceAll("\\.", "_");
        return src;
    }

	public void generateCode(TBPParsedProvisionContainerNode node) {
		if (node == null) {
			return;
		}
		
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();

		((TBPParsedNode)node.getChild()).visit(this);
		
		indenter.restoreIndentLevel(originalLevel); 
	}
	
	/**
	 * Getter for errorFound property. Returns true is any error take place during processing TBPParsed tree. 
	 * 
	 * @return Getter for errorFound property.
	 */
	public boolean wasError() {
		return errorFound;
	}
	
	private void printAnnotation(TBPParsedNode node) {
		if (printAnnotations == false) {
			return;
		}
		
		if (node.getAnnotation() == null) {
			return;
		}

		String annotations = node.getAnnotation().toString();
		if (annotations.trim().isEmpty() == true) {
			return;
		}

		indenter.indent();
		outFile.println("// " + annotations);
	}
	
	///////////////////////////////////////////////////
	// Routines related to work with threads
	//////////////////////////////////////////////////
	/**
	 * Creates new thread variable. Thread body contains subprotocol of given node
	 * 
	 * @param node Node which subtree is processed in new thread run method.
	 * @return Number of created thread variable 
	 */
	private int threadProcessNodeInNewThread(TBPParsedNode node) {
		int myThreadNum = ++cntThread;
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.indent(); outFile.println("Thread t" + Integer.toString(myThreadNum) + " = new Thread() {");
		indenter.addLevel();
		indenter.indent(); outFile.println("public void run() {"); 
		indenter.indent(); outFile.println("try {");
		indenter.addLevel();
		// Process inner thread action 
		node.visit(this);
		indenter.removeLevel();
		indenter.indent(); outFile.println("} catch (Throwable e) {");
		indenter.addLevel();
		// Check way how to handle exceptions
		if (EnvGenerator.EXCEPTIONS_STOP_ON_EXCEPTION) {
			indenter.indent(); outFile.println("throw new RuntimeException(e);");
		} else {
			indenter.indent(); outFile.println("System.out.println(\"Error - Unhandled Exception occured\");");
			indenter.indent(); outFile.println("System.out.println(e);");
		}
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
		indenter.removeLevel();
		indenter.indent(); outFile.println("} // end of run method");
		indenter.restoreIndentLevel(originalLevel);
		indenter.indent(); outFile.println("}; // end of class for thread t" + Integer.toString(myThreadNum));
		
		return myThreadNum;
	}

	/**
	 * Creates copy of original thread. Create new thread variable that behaves same as original thread.
	 * 
	 * @param originalThreadNum Number of variable with thread to copy
	 * @return Variable number of newly created thread
	 */
	@SuppressWarnings("unused")
	private int threadCopyThread(int originalThreadNum) {
		int myThreadNum = ++cntThread;
		indenter.indent(); outFile.println("Thread t" + myThreadNum + " = t" + originalThreadNum + ".getClass().getConstructor(this.getClass()).newInstance(this);");

		return myThreadNum;
	}

	/**
	 * Generates join code for given thread variable.
	 * 
	 * @param threadNum Number of thread variable to process
	 */
	private void threadNodeJoinBlock(int threadNum) {
		// indenter.indent(); out.println("try {");
		indenter.indent(); outFile.println("t" + Integer.toString(threadNum) + ".join();");
		// indenter.indent(); out.println("} catch (InterruptedException e) {}");
	}


	@Override
	/**
	 * Process alternative operator from grammar. "+"
	 * Introduce one variable with "random" value, and series of "if blocks". 
	 * Choosing of correct if block is based on value of previously created variable.
	 * <br>
	 * Linearize multiple neighboring alternatives nodes into one block (as it is n-ary operator)
	 *
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitParsedAlternative(TBPParsedAlternative node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();

		int myAlternative = ++cntAlternative;
		indenter.indent(); outFile.println("// visitParsedAlternative");
		indenter.indent(); outFile.println("// " + node.visit(prov2String));
		printAnnotation(node);
		
		// Introducing new choosing variable
		int depth = visitParsedAlternativeDepthCount(node);
		indenter.indent(); outFile.println("int " + AlternativeVariableName + Integer.toString(myAlternative) + " = Verify.random(" + Integer.toString(depth) + ");");
		indenter.indent(); outFile.println();

		// Recursively create "if clauses" for each alternative 
		visitParsedAlternativeGenerateIfBlock(node, myAlternative, 0);
		// Correctly ends last if block 
		outFile.println();
		indenter.restoreIndentLevel(originalLevel);
		return null;
	}

	/**
	 * Generates if blocks for given node and it's descendants. 
	 * <br>
	 * Code generation is delegated into {@link #visitParsedAlternativeGenerateIfBlockCode}. 
	 * This method walk recursively down (preorder) the tree for "alternative type" descendants, process others.  
	 * 
	 * @param node Node to process
	 * @param myAlternative Number of alternative variable
	 * @param processed  Previously processed alternatives count. Number of previously used values in alternative variables 
	 */
	private int visitParsedAlternativeGenerateIfBlock(TBPParsedAlternative node, int myAlternative, int processed) {
		// Process left child
		if (node.getLeft() instanceof TBPParsedAlternative) {
			// recurse
			processed = visitParsedAlternativeGenerateIfBlock((TBPParsedAlternative)(node.getLeft()), myAlternative, processed);
		} else {
			// generate "if block" for node 
			visitParsedAlternativeGenerateIfBlockCode((TBPParsedNode)node.getLeft(), myAlternative, processed);
			processed++;
		}

		// Process right child
		if (node.getRight() instanceof TBPParsedAlternative) {
			// recurse
			processed = visitParsedAlternativeGenerateIfBlock((TBPParsedAlternative)(node.getRight()), myAlternative, processed);
		} else {
			// generate "if block" for node 
			visitParsedAlternativeGenerateIfBlockCode((TBPParsedNode)node.getRight(), myAlternative, processed);
			processed++;
		}

		return processed;
	}

	/**
	 * Prints if block code into output. And recursively generate code for subtree of node.
	 * @param node
	 * @param myAlternative
	 * @param processed
	 */
	private void visitParsedAlternativeGenerateIfBlockCode(TBPParsedNode node, int myAlternative, int processed) {
		if (processed == 0) {
			// first if block ... indent before if needed
			// for other } else if block, indentation is printed as end of previous step before "}" character 
			indenter.indent();
		} else {
			// not first if block ... using "else if" to make state space smaller
			outFile.print(" else ");
		}

		// if clause for selection correct branch  
		outFile.println("if ( " + AlternativeVariableName + Integer.toString(myAlternative) + " == " + Integer.toString(processed) + " ) {");
		// if clause body ... subprotocol of given node
		node.visit(this);
		indenter.indent(); outFile.print("}");

	}

	/**
	 * Goes recursively down through node subtree (of "alternative") and counts leaf nodes.
	 * 
	 * @param node Root of tree where leaves are counted
	 * @return Number of nodes that are different than alternative
	 */
	private int visitParsedAlternativeDepthCount(TBPParsedAlternative node) {
		int count = 0;
		if (node.getLeft() instanceof TBPParsedAlternative) {
			// node is alternative --> recursively calculate number of leaves in this subtree
			count +=visitParsedAlternativeDepthCount((TBPParsedAlternative)(node.getLeft()));
		} else {
			// New leaf node ... one new alternative 
			count += 1;
		}

		if (node.getRight() instanceof TBPParsedAlternative) {
			// node is alternative --> recursively calculate number of leaves in this subtree
			count +=visitParsedAlternativeDepthCount((TBPParsedAlternative)(node.getRight()));
		} else {
			// New leaf node ... one new alternative 
			count += 1;
		}

		return count;
	}

	/**
	 * Process and-parallel operator from grammar. "|"
	 * <br>
	 * For each leaf create separate thread. Then starts all threads and wait for theirs results. 
	 * <br>
	 * Linearize multiple neighboring parallel nodes into one block (as it is n-ary operator)
	 *
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	@Override
	public Object visitParsedParallel(TBPParsedParallel node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();
		
		indenter.indent(); outFile.println("// visitParsedParallel");
		indenter.indent(); outFile.println("// " + node.visit(prov2String));
		printAnnotation(node);

		// Create thread variables
		List<Integer> threads = visitParsedParallelGenerateThreads(node);
		// Start all threads
		visitParsedParallelGenerateStartBlock(threads);
		// Waits for all threads
		visitParsedParallelGenerateJoinBlock(threads);
		
		indenter.restoreIndentLevel( originalLevel);
		return null;
	}


	/**
	 * Generates thread variables with anonymous thread classes for leaves in subtree of given node. 
	 * <br>
	 * Code generation is delegated into {@link #threadProcessNodeInNewThread}. 
	 * This method walk recursively down (preorder) the tree for "and-parallel type" descendants, process others types of nodes.  
	 * 
	 * @param node Node to process
	 * @return List with numbers of created thread variables created in subtree of node.
	 */
	private List<Integer> visitParsedParallelGenerateThreads(TBPParsedParallel node) {
		// Initialize empty result list
		List<Integer> retVal = new ArrayList<Integer>();

		// Process left child
		if (node.getLeft() instanceof TBPParsedParallel) {
			// recursive process left node and register created variables
			TBPParsedParallel left = (TBPParsedParallel)(node.getLeft()); 
			List<Integer> childThreads = visitParsedParallelGenerateThreads(left);
			retVal.addAll(childThreads);
		} else {
			// leaf found ... create thread variable
			int threadNum = threadProcessNodeInNewThread((TBPParsedNode) node.getLeft());
			retVal.add(threadNum);
		}

		// Process right child
		if (node.getRight() instanceof TBPParsedParallel) {
			// recursive process left node and register created variables
			TBPParsedParallel right = (TBPParsedParallel)(node.getRight()); 
			List<Integer> childThreads = visitParsedParallelGenerateThreads(right);
			retVal.addAll(childThreads);
		} else {
			// leaf found ... create thread variable
			int threadNum = threadProcessNodeInNewThread((TBPParsedNode) node.getRight());
			retVal.add(threadNum);
		}

		return retVal;
	}

	/**
	 * Prints block that starts created threads from given list.
	 * 
	 * @param threadNums List with thread numbers to start.
	 */
	private void visitParsedParallelGenerateStartBlock(List<Integer> threadNums) {
		for( int threadNum : threadNums) {
			indenter.indent(); outFile.println("t" + Integer.toString(threadNum) + ".start();");
		}
	}

	/**
	 * Prints block that waits until all started threads stop work.
	 * 
	 * @param threadNums List with thread numbers we have to wait.
	 */
	private void visitParsedParallelGenerateJoinBlock(List<Integer> threadNums) {
		for( int threadNum : threadNums) {
			threadNodeJoinBlock(threadNum);
		}
	}

	
	@Override
	/**
	 * Process parallel or operator from grammar. "||"
	 * <br>
	 * For each leaf create separate thread. Then "randomly" select one thread that will start. (At least one thread has to be started)
	 * In start block starts previously selected thread and "randomly" starts threads. 
	 * <br>
	 * Linearize multiple neighboring parallel nodes into one block (as it is n-ary operator).
	 * Because limited reentrancy is special case of parallel or operator, they are processes together as one tree.
	 *
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitParsedParallelOr(TBPParsedParallelOr node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();

		int myOrParallel = ++cntOrParallel;
		indenter.indent(); outFile.println("// visitParsedParallelOr");
		indenter.indent(); outFile.println("// " + node.visit(prov2String));
		printAnnotation(node);

		// Create variable with thread that will be started
		int depth = visitParsedParallelOrDepthCount(node);
		indenter.indent(); outFile.println("int " + ParallelOrVariableName + Integer.toString(myOrParallel) + " = Verify.random(" + Integer.toString(depth) + ");" );

		List<Integer> threads = visitParsedParallelOrGenerateThreads(node);
		visitParsedParallelOrGenerateStartBlock(threads, myOrParallel);
		visitParsedParallelOrGenerateJoinBlock(threads);
		
		indenter.restoreIndentLevel(originalLevel);
		return null;
	}

	/**
	 * Goes recursively down through node subtree (of "Parallel or" and "Limited reentrancy") and "counts" leaf nodes.
	 * <br> 
	 * Because limited reentrancy is special case of parallel or operator, they are processes together as one tree.
	 * <br>
	 * Note: Limited reentrancy multiply number of nodes in subtree. Action can by run at most "limit" times in parallel ... so we need "limit" number of threads.
	 *  
	 * @param node Root of tree where leaves are counted
	 * @return Number of nodes in subtree that are different than "parallel or". (Number of needed thread variables for subtree) 
	 */
	private int visitParsedParallelOrDepthCount(TBPParsedParallelOr node) {
		int retVal = 0;
		// Process left subtree
		if (node.getLeft() instanceof TBPParsedParallelOr) { // concatenated parallel Or 
			// recursively counts subtree nodes
			retVal += visitParsedParallelOrDepthCount((TBPParsedParallelOr)(node.getLeft()));
		} else if (node.getLeft() instanceof TBPLimitedReentrancy) {
			// recursively counts subtree nodes
			TBPLimitedReentrancy left = (TBPLimitedReentrancy)(node.getLeft());
			retVal += visitLimitedReentrancyDepthCount(left);
		} else {
			// New leaf node
			retVal += 1; // Node to process
		}

		if (node.getRight() instanceof TBPParsedParallelOr) { // // concatenated parallel Or
			// recursively counts subtree nodes
			retVal += visitParsedParallelOrDepthCount((TBPParsedParallelOr)(node.getRight()));
		} else if (node.getRight() instanceof TBPLimitedReentrancy) {
			// recursively counts subtree nodes
			TBPLimitedReentrancy right = (TBPLimitedReentrancy)(node.getRight());
			retVal += visitLimitedReentrancyDepthCount(right);
		} else {
			retVal += 1; //Node to process
		}
		return retVal;
	}

	/**
	 * Generates thread variables with anonymous thread classes for subtree of given node. 
	 * <br>
	 * Code generation is delegated into {@link #threadProcessNodeInNewThread}. 
	 * This method walks recursively down (preorder) the tree for "parallel or" and "limited reentrancy" type descendants, 
	 * process others types of nodes.  
	 * 
	 * @param node Node to process
	 * @return List with numbers of generated thread variables.
	 * <br>
	 * Note: Returned list is needed in Limited reentrancy nodes, because we need to know which threads should be copied 
	 */
	private List<Integer> visitParsedParallelOrGenerateThreads(TBPParsedParallelOr node) {
		List<Integer> retVal = new ArrayList<Integer>();
		if (node.getLeft() instanceof TBPParsedParallelOr) { // concatenated 
			// Recursively process left descendant and collect generated variables
			TBPParsedParallelOr left = (TBPParsedParallelOr)(node.getLeft());
			List<Integer> createdThreads = visitParsedParallelOrGenerateThreads(left); 
			retVal.addAll(createdThreads); 
		} else if (node.getLeft() instanceof TBPLimitedReentrancy) {
			// Recursively process left descendant and collect generated variables
			TBPLimitedReentrancy left = (TBPLimitedReentrancy)(node.getLeft());
			List<Integer> createdThreads = visitLimitedReentrancyGenerateThreads(left);
			retVal.addAll(createdThreads); 
		} else {
			// Not recurse, node to execute
			int newThreadNum = threadProcessNodeInNewThread((TBPParsedNode) node.getLeft());
			retVal.add( new Integer(newThreadNum));
		}

		if (node.getRight() instanceof TBPParsedParallelOr) { // concatenated 
			// Recursively process right descendant and collect generated variables
			TBPParsedParallelOr right = (TBPParsedParallelOr)(node.getRight());
			List<Integer> createdThreads = visitParsedParallelOrGenerateThreads(right);
			retVal.addAll(createdThreads); 
		} else if (node.getRight() instanceof TBPLimitedReentrancy) {
			// Recursively process left descendant and collect generated variables
			TBPLimitedReentrancy right = (TBPLimitedReentrancy)(node.getRight());
			List<Integer> createdThreads = visitLimitedReentrancyGenerateThreads(right);
			retVal.addAll(createdThreads); 
		} else {
			// Not recurse, node to execute
			int newThreadNum = threadProcessNodeInNewThread((TBPParsedNode) node.getRight());
			retVal.add( new Integer(newThreadNum));
		}

		return retVal;
	}

	/**
	 * Prints block that "randomly" starts threads from given list. 
	 * Generated block ensures that at least one thread specified by value of given variableOrNum will be started.   
	 * 
	 * @param threadVariables List with thread numbers for start condition generation 
	 * @param variableOrNum Number of {@link #ParallelOrVariableName} used for specifying started thread.
	 */
	private void visitParsedParallelOrGenerateStartBlock(List<Integer> threadVariables, int variableOrNum) {
		int processed = 0;
		for(int threadNum : threadVariables) {
			indenter.indent(); outFile.println("if ( (" + ParallelOrVariableName + Integer.toString(variableOrNum) + " == " + Integer.toString(processed) + ") || (Verify.getBoolean()) ) {");
			indenter.addLevel();
			indenter.indent(); outFile.println("t" + Integer.toString(threadNum) + ".start();");
			indenter.removeLevel();
			indenter.indent(); outFile.println("}");
			processed++;
		}
	}

	/**
	 * Prints block that waits until all started threads stop work.
	 * 
	 * @param threadVariables List with thread numbers we have to wait.
	 */
	private void visitParsedParallelOrGenerateJoinBlock(List<Integer> threadVariables) {
		for(int threadNum : threadVariables) {
			threadNodeJoinBlock(threadNum);
		}
	}

	@Override
	/**
	 * Process sequence operator from grammar. ";".
	 * <br>
	 * Sequentially process first left and then right subtree.
	 *  
	 * @param node Node in parsed AST tree to process
	 * @return Always null. 
	 */
	public Object visitParsedSequence(TBPParsedSequence node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		
		if (!(node.getParent() instanceof TBPParsedSequence)) {
			indenter.addLevel();
			indenter.indent(); outFile.println("// visitParsedSequence");
			indenter.indent(); outFile.println("// " + node.visit(prov2String));
		}
		printAnnotation(node);
		((TBPParsedNode)node.getLeft()).visit(this);
		((TBPParsedNode)node.getRight()).visit(this);

		indenter.restoreIndentLevel(originalLevel);
		return null;
	}

	@Override
	/**
	 * Process repetition operator from grammar. "*".
	 * <br>
	 * Generates "randomly" length repeated cycle with body representing repeated subtree.
	 *  
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitParsedRepetition(TBPParsedRepetition node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();

		indenter.indent(); outFile.println("// visitParsedRepetition");
		indenter.indent(); outFile.println("// " + node.visit(prov2String));
		printAnnotation(node);
		if (REPETITION_AS_WHILE) {
			indenter.indent(); outFile.println("while (Verify.getBoolean()) {");
		} else {
			String varName = RepetitionVariableName + Integer.toString(cntRepetition++);
			indenter.indent(); outFile.println("for(byte " + varName + " = 0; " + varName + " < " + REPETITION_LIMIT + "; " + varName + "++) {");
			indenter.addLevel(); 
			indenter.indent(); outFile.println("if (Verify.getBoolean()) { break; }"); 
			indenter.removeLevel();
		}

		((TBPParsedNode)node.getChild()).visit(this);
		indenter.indent(); outFile.println("}");

		indenter.restoreIndentLevel(originalLevel);
		return null;
	}


	@Override
	/**
	 * Process limited reentrancy operator from grammar. "|n".
	 * <br>
	 * For each leaf in (or-parallel and limited reentrancy) subtree create separate thread. Than create (n-1) copies of created threads. 
	 * Then "randomly" select one thread that will start. (At least one thread has to be started)
	 * In start block starts previously selected thread and "randomly" starts threads. 
	 * <br>
	 * Linearize cascade of limited reentrancy into one block (as one operator use)
	 * Because limited reentrancy is special case of parallel or operator, they are processes together as one tree.
	 *  
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitLimitedReentrancy(TBPLimitedReentrancy node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();

		int myOrParallel = ++cntOrParallel;

		indenter.indent(); outFile.println("// visitLimitedReentrancy " + node.getLimit());
		indenter.indent(); outFile.println("// " + node.visit(prov2String));
		printAnnotation(node);
		int depth = visitLimitedReentrancyDepthCount(node);
		indenter.indent(); outFile.println("int " + ParallelOrVariableName + Integer.toString(myOrParallel) + " = Verify.random(" + Integer.toString(depth) + ");");

		List<Integer> threads = visitLimitedReentrancyGenerateThreads(node);

		visitParsedParallelOrGenerateStartBlock(threads, myOrParallel);
		visitParsedParallelOrGenerateJoinBlock(threads);
		
		indenter.restoreIndentLevel(originalLevel);
		return null;
	}

	/**
	 * Goes recursively down through node subtree (of "Parallel or" and "Limited reentrancy") and "counts" leaf nodes.
	 * <br> 
	 * Because limited reentrancy is special case of parallel or operator, they are processes together as one tree.
	 * <br>
	 * Note: Limited reentrancy multiply number of nodes in subtree. Action can by run at most "limit" times in parallel ... so we need "limit" number of threads.
	 *  
	 * @param node Root of tree where leaves are counted
	 * @return Number of nodes in subtree that are different than "parallel or". (Number of needed thread variables for subtree) 
	 */
	private int visitLimitedReentrancyDepthCount(TBPLimitedReentrancy node) {
		if (node.getChild() instanceof TBPParsedParallelOr) { // concatenated 
			return node.getLimit() * visitParsedParallelOrDepthCount((TBPParsedParallelOr)(node.getChild()));
		} else if (node.getChild() instanceof TBPLimitedReentrancy) {
			TBPLimitedReentrancy left = (TBPLimitedReentrancy)(node.getChild());
			return node.getLimit() * visitLimitedReentrancyDepthCount(left);
		}

		return node.getLimit();
	}

	/**
	 * Generates thread variables with anonymous thread classes for leaves in subtree of given node. 
	 * <br>
	 * For first instances of thread is code generation is delegated into {@link #threadProcessNodeInNewThread}.
	 * Remaining instances of theads (n-1) are copies of previously generated threads. 
	 *
	 * This method walks recursively down (preorder) the tree for "parallel or" and "limited reentrancy" type descendants, 
	 * process others types of nodes.  
	 *  
	 * @param node Node to process
	 * @return List with numbers of created thread variables created in subtree of node.
	 * <br>
	 * Note: Returned list is needed in Limited reentrancy nodes, because we need to know which threads should be copied 
	 */
	private List<Integer> visitLimitedReentrancyGenerateThreads(TBPLimitedReentrancy node) {
		// Generates as much same thread as is needed
		int limit = node.getLimit();
		List<Integer> retVal = new ArrayList<Integer>();

		for(int i = 0; i < limit; i++) { 
			if (node.getChild() instanceof TBPParsedParallelOr) { // concatenated 
				TBPParsedParallelOr child = (TBPParsedParallelOr)(node.getChild());
				retVal.addAll(visitParsedParallelOrGenerateThreads(child));
			} else if (node.getChild() instanceof TBPLimitedReentrancy) {
				TBPLimitedReentrancy child = (TBPLimitedReentrancy)(node.getChild());
				retVal.addAll(visitLimitedReentrancyGenerateThreads(child));
			} else {
				// Not recurse, subtree to execute
				TBPParsedNode child = (TBPParsedNode)(node.getChild());
				retVal.add(threadProcessNodeInNewThread(child));
			}
		}
		return retVal;
	}

// Thread reusing version ... each thread is defined only once others are created by "newInstance" call
//		private List<Integer> visitLimitedReentrancyGenerateThreads(TBPLimitedReentrancy node) {
//			//TBPParsedNode child = (TBPParsedNode)(node.getChild());
//			int limit = node.getLimit();
//			List<Integer> originalThreadNums;
//
//			// Gets thread to be copied 
//			if (node.getChild() instanceof TBPParsedParallelOr) { // concatenated 
//				TBPParsedParallelOr child = (TBPParsedParallelOr)(node.getChild());
//				originalThreadNums = visitParsedParallelOrGenerateThreads(child);
//			} else if (node.getChild() instanceof TBPLimitedReentrancy) {
//				TBPLimitedReentrancy child = (TBPLimitedReentrancy)(node.getChild());
//				originalThreadNums = visitLimitedReentrancyGenerateThreads(child);
//			} else {
//				// Not recurse, subtree to execute
//				TBPParsedNode child = (TBPParsedNode)(node.getChild());
//				int newThreadNum = threadProcessNodeInNewThread(child);
//
//				originalThreadNums = new ArrayList<Integer>();
//				originalThreadNums.add(newThreadNum);
//			}
//			
//			
//			// Creates copies of threads
//			List<Integer> retVal = new ArrayList<Integer>();
//			retVal.addAll(originalThreadNums);
//			for(Integer originalThread : originalThreadNums) {
//				// counts ... 1 is generated as original threads ... so start index is 2
//				for(int count = 2; count <= limit; count++) {
//					int newCreatedThreadNum = threadCopyThread(originalThread);
//					retVal.add(newCreatedThreadNum);
//				}
//			}
//			return retVal;
//		}

	@Override
	/**
	 * Process unlimited reentrancy operator. 
	 * Prints internal error output because unlimited reentrancy operator have to be converted into limited reentrancy one.
	 *
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitUnlimitedReentrancy(TBPUnlimitedReentrancy node) {
		errorFound = true;
		infoStream.println("Internal error - unlimited reentrancy wasn't changes into limited reantrancy node");

		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();
		printAnnotation(node);
		indenter.indent(); outFile.println("throw new RuntimeException(\"Unprocessed unlimited reentrancy mode. Internal environment generator error.\");");
		indenter.restoreIndentLevel(originalLevel);
		return null;
	}

	@Override
	/**
	 * Process NULL action from grammar. Doe'nt generate any executable code. 
	 *
	 * @param node Node in parsed AST tree to process
	 * @return Always null.
	 */
	public Object visitParsedNull(TBPParsedProvisionNull node) {
		indenter.addLevel();
		printAnnotation(node);
		indenter.indent(); outFile.println("// NULL");
		indenter.removeLevel(); 
		return null;
	}


	@Override
	public Object visitParsedAccept(TBPParsedAccept node) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();
		indenter.addLevel();
		indenter.indent(); outFile.println("// visitParsedAccept " + node.getFullname());
		printAnnotation(node);
		
		
		// Check if interface exists
		if (config.getComponentProvidedInterfaces().containsKey( node.getInterface()) == false) {
			infoStream.println("Error - unknown provided interface name " + node.getInterface() + ". Cannot generate correct calling code.");
			indenter.indent(); outFile.println("throw new RuntimeException(\"Call on provided interface cannot be generated\");");
			indenter.restoreIndentLevel(originalLevel);
			errorFound = true;
			return null;
		}
		
		// Find name class where interface is defined
		String interfaceClassName = config.getComponentProvidedInterfaces().get( node.getInterface());
		if (interfaceClassName == null) {
			infoStream.println("Error - unknown name for interface name " + node.getInterface() + ". Cannot generate correct calling code.");
			indenter.indent(); outFile.println("throw new RuntimeException(\"Call on provided interface cannot be generated\");");
			indenter.restoreIndentLevel(originalLevel);
			errorFound = true;
			return null;
		}
		
		// Find Class of provided interface 
		Class<?> interfaceClass = null;
		try {
			interfaceClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(interfaceClassName));
		} catch (ClassNotFoundException e) {
			infoStream.println("Error - cannot load Class (" + Type2String.removeGenerics(interfaceClassName) + ") where " + node.getInterface() + " interface is defined.");
			if (DEBUG) { e.printStackTrace(infoStream); }
			indenter.indent(); outFile.println("throw new RuntimeException(\"Call on provided interface cannot be generated\");");
			indenter.restoreIndentLevel(originalLevel);
			errorFound = true;
			return null;
		}
		
		// Find property name for given interface
		String interfacePropertyName = envDesc.provItfs2envFieldName.get(node.getInterface());
		if (interfacePropertyName == null) {
			infoStream.println("Internal Error - no property name for provided interface " + node.getInterface() + ".");
			indenter.indent(); outFile.println("throw new RuntimeException(\"Call on provided interface cannot be generated\");");
			indenter.restoreIndentLevel(originalLevel);
			errorFound = true;
			return null;
			
		}
		
		// Find called Method
		Method calledMethod = null;
		for(Method m : interfaceClass.getMethods()) {
			if (m.getName().equals( node.getMethod())) {
				// Note names of classes in interface have to different. No overloading.
				calledMethod = m;
				break; // Stop searching
			}
		}
		if (calledMethod == null) {
			infoStream.println("Error - provided interface " + node.getInterface() + " (" + interfaceClassName + ") doesn't contain method with name " + node.getMethod() + ".");
			indenter.indent(); outFile.println("throw new RuntimeException(\"Call on provided interface cannot be generated\");");
			indenter.restoreIndentLevel(originalLevel);
			errorFound = true;
			return null;
		}
		
		visitParsedAcceptExceptionsTryBlock(calledMethod);
		
		// Generate method call
		indenter.indent(); outFile.println( interfacePropertyName + "." + node.getMethod() + "(");
		int line = outFile.getLine();
		indenter.addLevel();
		indenter.addLevel();
		// Generate method call code parameters
		Type[] params = calledMethod.getGenericParameterTypes();
		for(int i = 0; i < params.length; i++) {
			// Reusing code from stub generator for obtaining values from envSets
			//TODO replace interfaceClassName by Interface name!!
			indenter.indent(); outFile.print( StubGenerator.genCodeObtainingValue(params[i], config.getComponentName(), interfaceClassName, node.getMethod()));
			if ( i <  params.length-1 ) {
				// Not last parameter
				outFile.println(", ");
			} else {
				outFile.println();
			}
		}
		indenter.removeLevel();
		indenter.indent(); outFile.println(");");
		indenter.removeLevel();
		
		visitParsedAcceptExceptionsCatchBlock(calledMethod);
		indenter.restoreIndentLevel(originalLevel);

		// Add mappings into EnvironmentDescription
		envDesc.line2ProvidedCall.put(line, node);
		return null;
	}

	private void visitParsedAcceptExceptionsTryBlock(Method calledMethod) {
		if (!EXCEPTIONS_ACCEPT_TRY_CATCH_BLOCKS) {
			return;
		}
		// Check if method call can return any exception
		if (calledMethod.getGenericExceptionTypes().length > 0) {
			indenter.indent(); outFile.println("try {");
			indenter.addLevel();
		}
	}
	private void visitParsedAcceptExceptionsCatchBlock(Method calledMethod) {
		if (!EXCEPTIONS_ACCEPT_TRY_CATCH_BLOCKS) {
			return;
		}

		// Check if method call can return any exception
		if (calledMethod.getGenericExceptionTypes().length == 0) {
			return;
		}
		indenter.removeLevel();
		for(Type t : calledMethod.getExceptionTypes()) {
			indenter.indent(); outFile.println("} catch (" + Type2String.getTypeName(t) + " ex) {");
			indenter.addLevel();
			indenter.indent(); outFile.println("ex.printStackTrace();");
			indenter.removeLevel();
		}
		indenter.indent(); outFile.println("}");
	}
	
	static public boolean isRepetitionOperAsWhile() {
		return REPETITION_AS_WHILE;
	}
	
	static public String getStaticConfiguration() {
		if (REPETITION_AS_WHILE) {
			return "Repetetion operator is generated as while cycle";
		} else {
			return "Repetetion operator is generated as for cycle with up to " + REPETITION_LIMIT + " iterations";
		}
	}
}
