package org.ow2.dsrg.fm.tbpjava.checker;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Is used for manipulation with variable evaluations - getting and setting values. 
 *
 * For each variable from specification holds mapping from all its possible values to integers.   
 *
 */
public class VariablesMapper {
	private static final boolean DEBUG = false;
	private static final boolean DEBUG_CONSTRUCTOR = DEBUG;
	private static final boolean DEBUG_SET_VALUE = false; // was true
	
	private static Integer variableMapperIDsCounter = 0; 
	
	
	public static final int STATE_ARRAY_OFSET_MAPPER_ID = 0; /// Position where in array is stored ID of owning mapper
	public static final int STATE_ARRAY_OFSET_HASH_CODE = 1; /// Position where in array is stored record with hash value
	public static final int STATE_ARRAY_OFSET_STATE_ID = 2;  /// Position StatesMapper saves state number
	public static final int STATE_ARRAY_OFSET_VARIABLE_START = 3; /// Position where in array starts records about variables values.

	/* 
	 * Note: 
	 * 	First array entry is ID of component (mapper), 
	 * 	Second entry is hash (XOR) code of all values for faster comparation
	 */
	
	//private LTSAComponentSpecification component;
	private int variableMapperID;
	
	private class VariableRecord {
		public int variableIndex;
		public TypeDef variableType;
	}
	
	private Map<Typedef, TypeDef> type2CheckerType; /// Maps variable types into extended types needed for checker.

	private Map<String, VariableRecord> variables; /// Maps variables names into record about them
	private List<String> variablesIndexes; /// Maps indexes in array into variables names
	
	private int stateArraySize; /// How many entries have arrays that holds state.
	
	private final int[] initialArray;
	
	/**
	 * Creates new object that maps variables from given component into array. 
	 *
	 * @param component Component to process.
	 */
	public VariablesMapper(LTSAComponentSpecification component) {
		assert component !=  null;
		
		synchronized (variableMapperIDsCounter) {
			variableMapperID = variableMapperIDsCounter;
			variableMapperIDsCounter++;
		}
		
		
		//this.component = component;
		
		Map<String, TBPResolvedVardef> varDefs = component.getVardefs();
		int variablesCount = varDefs.size();
		this.stateArraySize = variablesCount + STATE_ARRAY_OFSET_VARIABLE_START;

		// Initialize Checker types
		type2CheckerType = new HashMap<Typedef, TypeDef>( variablesCount + variablesCount / 2);
		for(Typedef type : component.getTypes()) {
			type2CheckerType.put(type, new TypeDef(type));
		}
		
		// Initialize variable mappings variables
		variables = new HashMap<String, VariableRecord>( variablesCount + variablesCount / 2);
		variablesIndexes = new ArrayList<String>(variablesCount + STATE_ARRAY_OFSET_VARIABLE_START);
		int firstEmptyArrayIndex = STATE_ARRAY_OFSET_VARIABLE_START;

		// Initialize empty items in variablesIndexes. Not start on index 0
		for(int i = 0; i < STATE_ARRAY_OFSET_VARIABLE_START; i++) {
			variablesIndexes.add(null);
		}
		
		for(String variableName : varDefs.keySet()) {

			VariableRecord newVarRecord = new VariableRecord();
			newVarRecord.variableIndex = firstEmptyArrayIndex;
			Typedef type = varDefs.get(variableName).getType();
			newVarRecord.variableType = type2CheckerType.get(type); 
			
			variables.put(variableName, newVarRecord);
			variablesIndexes.add(variableName);
			firstEmptyArrayIndex++;
		}
		
		assert firstEmptyArrayIndex == stateArraySize;
		
		// Prepare initial array
		initialArray = stateArrayInitalize();

		for(String varName : varDefs.keySet()) {
				if (DEBUG_CONSTRUCTOR) {System.out.print("VariablesMapper::VariablesMapper - setting initial value for " + varName); }
			String varInitialValue = varDefs.get(varName).getInitialValue(); 
				if (DEBUG_CONSTRUCTOR) {System.out.println(" ... " + varInitialValue); }
			stateArrayInitializeSetValue(initialArray, varName, varInitialValue);
		}
		
	}

	private int[] stateArrayInitalize() {
		int[] result =  new int[stateArraySize];
		result[STATE_ARRAY_OFSET_MAPPER_ID] = variableMapperID;
		return result;
	}
	
	/**
	 * Sets variable value into state array. This is initializing routine and works differently with has value. 
	 * 
	 * @param stateArray Array where set value
	 * @param varName Name of variable to set value
	 * @param varInitialValue What value set
	 */
	private	void stateArrayInitializeSetValue(int[] stateArray, String varName, String varInitialValue) {
		VariableRecord vr = variables.get(varName);
		assert vr != null; // Test for (In)valid variable name 
		
		TypeDef.TypeEntryRecord ter = vr.variableType.getValueRecord(varInitialValue);
		assert ter != null; // Test for (in)valid variable value
		
		stateArray[vr.variableIndex] = ter.value;
		stateArray[STATE_ARRAY_OFSET_HASH_CODE] ^= ter.xorValue;
	}
	
	/**
	 * Sets (changes) variable value in state array. 
	 * 
	 * @param stateArray Array where set value
	 * @param varName Name of variable to set value
	 * @param varInitialValue What value set
	 */
	public boolean stateArraySetValue(int[] stateArray, String varName, String varValue) {
		assert stateArray[STATE_ARRAY_OFSET_MAPPER_ID] == variableMapperID;

		VariableRecord vr = variables.get(varName);
		if (vr == null) { // Test for Invalid variable name
			if (DEBUG_SET_VALUE) { System.out.println("ERROR - Assigning value for invalid variable."); }
			return false;  
		}

		// Obtaining new value informations
		TypeDef.TypeEntryRecord ter = vr.variableType.getValueRecord(varValue);
		if (ter == null) { // Test for invalid variable value
			if (DEBUG_SET_VALUE) { System.out.println("ERROR - Assigning invalid value into variable."); }
			return false; 
		}
		
		// Obtaining current values XOR
		int usedXOR = vr.variableType.getXorValue( stateArray[vr.variableIndex]);
		
		// Setting new values
		stateArray[vr.variableIndex] = ter.value;
		stateArray[STATE_ARRAY_OFSET_HASH_CODE] ^= usedXOR; // Removing old XOR value
		stateArray[STATE_ARRAY_OFSET_HASH_CODE] ^= ter.xorValue; // Setting now XOR
		
		return true;
	}

	/**
	 * Gets variable value in textual form from state array. 
	 * 
	 * @param stateArray Array where set value
	 * @param varName Name of variable to set value
	 * @param varInitialValue What value set
	 * @return Value of selected variable in state array, or null if invalid variable name of value 
	 */
	public String stateArrayGetValue(int[] stateArray, String varName) {
		assert stateArray[STATE_ARRAY_OFSET_MAPPER_ID] == variableMapperID;

		VariableRecord vr = variables.get(varName);
		if (vr == null) { // Test for Invalid variable name
			return null;  
		}
		return vr.variableType.getEnumName( stateArray[vr.variableIndex]);
	}
	
	public static boolean stateEquals(int[] state1, int[] state2) {
		if (state1[STATE_ARRAY_OFSET_HASH_CODE] != state2[STATE_ARRAY_OFSET_HASH_CODE]) {
			return false;
		} 
		if (state1[STATE_ARRAY_OFSET_MAPPER_ID] != state2[STATE_ARRAY_OFSET_MAPPER_ID]) { 
			return false;
		}
			
		if (state1 == state2) {
			return true;
		}
		
		boolean result = true;
		for(int i = STATE_ARRAY_OFSET_VARIABLE_START; i < state1.length /* && result */; i++) {
			result &= state1[i]==state2[i];
		}
		return result;
	}
	
	public int[] getInitialState() {
		return initialArray;
	}

	public final static int getStateArrayKey(int[] stateArray) {
		return stateArray[STATE_ARRAY_OFSET_STATE_ID];
	}

	public final static void setStateArrayNewKey(int[] stateArray, int newID) {
		stateArray[STATE_ARRAY_OFSET_STATE_ID] = newID;
	}
	
	/**
	 * Converts variable evaluation from state array into string.
	 * @param state Variables evaluation to convert.
	 * @return String with variable names and values and internal states
	 */
	public String DEBUG_stateArrayToString(int[] state) {
		if (state == null) {
			return "[null]";
		}
		if (state[STATE_ARRAY_OFSET_MAPPER_ID] != variableMapperID) {
			return "[invalid state owner]";
		}
		StringBuffer sb = new StringBuffer();
		sb.append("[");
		sb.append("hash=" + state[STATE_ARRAY_OFSET_HASH_CODE]);
		sb.append(", stateID=" + state[STATE_ARRAY_OFSET_STATE_ID]);
		
		for(String varName : variablesIndexes) {
			// Skip first invalid entries
			if (varName == null) continue;
			
			sb.append(", ");
			sb.append(varName);
			sb.append('=');
			String value = stateArrayGetValue(state, varName);
			if (value==null) {
				value="!!INVALID!!";
			}
			sb.append(value);
		}
		
		sb.append("]");
		return sb.toString();
	}

}
