package org.ow2.dsrg.fm.tbpjava.checker;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.util.Typedef;


/**
 * This class holds evaluation of component variables (from specification).
 * 
 * This means that for any variable from specification Variables part holds mapping into exactly one value.
 * Values are represented as strings for easier debugging. 
 *  
 * <br>
 * Note: Immutable class
 * Note: For better performance very good practice that for any variable value there is only one string instance that hold this string representation. (falster state comparation)
 *
 * @author Alfifi
 */
final public class VariablesEvaluation implements Serializable {

	private static final long serialVersionUID = 7526438036848650079L;
	
	private static final double CONST_MAP_GROWING = 1.5; 
	private static final int INITIAL_XOR_VALUE = 0xf0f0f0f0; /// Initial value for property xorValuesHash
	
	private static final Random rnd = new Random(); /// Used to randomly generate java hash codes
	
	private final LTSAComponentSpecification comp; /// Component for which we represents values;
	private final StatesMapper stateMapper; /// Instance that maps {@link VariablesEvaluation} with {@link AutomatonState}. This is needed to know for setting values to track automaton state membership.

	private final Map<String, String> var2value; /// Maps variables to it's values.
	private int xorValuesHash = INITIAL_XOR_VALUE; /// hashCode of variable value for easier comparison if evaluations are same. 
	
	private int DEBUG_variableEvaluationID; /// Unique instance ID. Only for debugging purposes.
	private static int DEBUG_variablesEvaluationIDCounter = 0; /// Counter for assigning instance IDs.
	
	private final int javaHashCode = rnd.nextInt(); 

	/** 
	 * Creates variable evaluation with initial variable values.
	 * 
	 * @param comp Component for which represent variables. Cann't be null.
	 */
	public VariablesEvaluation(LTSAComponentSpecification comp, StatesMapper stateMapper) {
		assert comp != null;
		this.comp = comp;
		this.stateMapper = stateMapper;
		this.DEBUG_variableEvaluationID = DEBUG_variablesEvaluationIDCounter++;
		
		Map<String, TBPResolvedVardef> initialVars = comp.getVardefs();
		assert initialVars != null; // Invalid specification
		
		var2value = new HashMap<String, String>((int)(initialVars.size() * CONST_MAP_GROWING));

		// Set initial values
		for(Entry<String, TBPResolvedVardef> entry : initialVars.entrySet()) {
			final String varName = entry.getKey();
			final String varValue = entry.getValue().getInitialValue(); 
			
			assert varValue != null; // Each variable has to have assigned initial value
			
			var2value.put(varName, varValue);
			xorValuesHash ^= varValue.hashCode();
		}
	}
	
	/** 
	 * Makes clone of source instance.
	 * <br>
	 * Note: This clone is not mapped to any state in {@link StatesMapper}. 
	 *   It's needed to call {@link StatesMapper#cloneMappings(VariablesEvaluation, VariablesEvaluation)} 
	 *   to map instance with same {@link AutomatonState} as source.
	 *
	 * Note: It's needed to change {@link #DEBUG_variableEvaluationID} before
	 *   call {@link StatesMapper#cloneMappings(VariablesEvaluation, VariablesEvaluation)}.
	 *
	 * @param source VariablesEvaluation to clone. Cann't be null. 
	 */
	protected VariablesEvaluation(VariablesEvaluation source) {
		assert source != null;
		
		comp = source.comp;
		stateMapper = source.stateMapper;
		DEBUG_variableEvaluationID = source.DEBUG_variableEvaluationID;
		
		var2value = new HashMap<String, String>(source.var2value);
		xorValuesHash = source.xorValuesHash;
	}
	
	/**
	 * Gets value associated to variable.
	 * 
	 * @param name Name of variable. Cannot be null.
	 * @return Gets value of variable name. Null if variable with name doesn't exists.
	 */
	public String getVariableValue(String name) {
		assert name != null;
		return var2value.get(name);
	}
	
	/**
	 * Return value of variable as constant reference
	 * 
	 * <br>Note: Usable for parameter processing.
	 * 
	 * @param name Name of variable. Cannot be null.
	 * @return Gets value of variable name. Null if variable with name doesn't exists.
	 */
	public ConstantRef getVariableValueReference(String name) {
		assert name != null;
		String variableValue = getVariableValue(name);
		if (variableValue == null) {
			// Unknown variable value
			return null;
		}
		
		Typedef variableType = getVariableType(name);
		assert variableType != null;
		
		return new ConstantRef( variableValue, variableType);
	}
	
	/**
	 * Gets type of variable with given name.
	 * 
	 * <br>Note: Not usable on parameters, only for represented values (from specification var section)
	 * 
	 * @param name Name of variable.
	 * @return Return type of variable with given name or null if invalid name given.
	 */
	public Typedef getVariableType(String name) {
		assert name != null;
		
		TBPResolvedVardef variableDef = comp.getVardefs().get(name); 
		if (variableDef == null) {
			// Unknown variable name -> return null
			return null;
		}
		return variableDef.getType();
	}
	
	/**
	 * Change variables evaluation.
	 * Returned Variable evaluation has same mappings {@link StatesMapper} as current one and is in MAPPED state.
	 * Can be called only if {@link VariablesEvaluation} is in MAPPED state. (see {@link StatesMapper} for more info about {@link VariablesEvaluation} states)
	 * <br>
	 * Note: Doesn't check if new value is from permitted one (doesn't check type compatibility).
	 * 
	 * @param name Variable name to change. Cann't be null.
	 * @param value Variable value to assign. Cann't be null. 
	 * @return Gets instance that has same mapping as this except variable name that has specified value. 
	 *   Get's null if variables name is invalid or value is null.
	 */
	public VariablesEvaluation setVariableValue(String name, String value) {
		assert name != null;
		assert value != null;

		VariablesEvaluation result = new VariablesEvaluation(this);
		result.DEBUG_variableEvaluationID = DEBUG_variablesEvaluationIDCounter++; // Assign new ID because this is different mapping
		
		String oldValue = var2value.get(name);
		if (oldValue == null) {
			// Invalid variable name
			return null;
		}
		
		result.var2value.put(name, value);
		result.xorValuesHash ^= oldValue.hashCode(); // Remove old hashcode
		result.xorValuesHash ^= value.hashCode(); // Add new hash code;
		// Can be called AFTER we calculate the xorValue !!!!
		stateMapper.cloneMappings(this, result);
		
		return result;
	}
	
	/**
	 * Compare if other {@link VariablesEvaluation} has same values assigned to variables.
	 *  
	 * @param other Instance to compare.
	 * @return true is variables has same values, false otherwise.
	 */
	public boolean equalsValues(VariablesEvaluation other) {
		if (other == this) {
			return true;
		} else if (other == null) {
			return false;
		} else if ( other.xorValuesHash != xorValuesHash) {
			return false;
		} else if (other.comp != comp) {
			return false; // Comparison of states from different specification does't give sense 
		};
		
		for(String name : var2value.keySet()) {
			String myValue = var2value.get(name);
			String otherValue = other.var2value.get(name);
			if (!myValue.equals(otherValue)) {
				return false; // Different assignment
			}
		}
		
		return true;
	}
	
	/**
	 * @return Gets internal hash value that is related to the current mapping.
	 * <be>Note: Can be used to calculate real hash values of the objects that holds {@link VariablesEvaluation}.
	 */
	public int getChecherHashCode() {
		return xorValuesHash;
	}
	
	/**
	 * Check other is equal to this instance. Only same instances equals.
	 * <br>
	 * Note: Doesn't check variables values (nor other state), because we doesn't join same evaluations together.
	 * Note: To compare if assigned values are same call {@link #equalsValues(VariablesEvaluation)}.
	 * 
	 * @param other Instance to compare
	 * @return true it other is same instance.
	 */
	final public boolean equals(Object other) {
		return other == this;
	}
	
	final public int hashCode() {
		return javaHashCode;
	}

	/**
	 * Converts variable evaluation into string.
	 * @return String with variable names and values and internal states
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("[");
		sb.append("ID=" + DEBUG_variableEvaluationID);
		sb.append(", hash=" + xorValuesHash);
		
		for(String name : var2value.keySet()) {
			sb.append(", ");
			sb.append(name);
			sb.append('=');
			sb.append(var2value.get(name));
		}
		
		sb.append("]");
		return sb.toString();
	}
	
	/**
	 * @return Gets internal unique instance identifier. Note: Only for debugging purposes.
	 */
	public int DEBUG_get_variableEvaluationID() {
		return DEBUG_variableEvaluationID;
	}
}
