package org.ow2.dsrg.fm.tbpjava.checker;

import org.ow2.dsrg.fm.tbplib.resolved.LastCallRef;

/**
 * Used for connecting automaton state (from one thread) to variable evaluations arrays 
 *   and for storing return values from automatons.
 *   
 *  <br>
 *  Note: Immutable class.
 *
 */
final public class ThreadStateStack {
	static public enum ReturnValues {
		RET_VAL_SET,
		RET_VAL_NOT_SET,
		/**
		 * Used to represent the "result" of the undefined emit call. 
		 * It means that no return value is specified and 
		 * if the {@link LastCallRef} is used to assign value into the variable,
		 *   then all possible return values permitted by assigned type are tried.
		 * Note: Associated value entry contains null if this type is used.
		 */
		RET_VAL_UNDEFINED_EMIT
	}
	
	private final AutomatonState as;
	private final ThreadStateStack previous;
	private final String returnValue;
	private final ReturnValues returnValueType;

	private final String lastCallReturnValue; /// Return value of the last call in given thread
	private final ReturnValues lastCallReturnValueType; /// Type of return value ... if set or not
	
	
	final private int hashCode;
	final private int HASH_CONSTANT = 83; 

	/**
	 * Used to create entry (used in stack) for thread specific automatons states.
	 * Created entry doesn't have set returnValue.
	 * 
	 * Note: {@link ThreadStateStack} created by this constructor represents new call.
	 * 
	 * @param as State. Cann't be null.
	 * @param previous Previous entry in (thread specific) stack.
	 */
	public ThreadStateStack(AutomatonState as, ThreadStateStack previous) {
		assert as != null;
		assert as.getEdge() == null; // Call (start state without previous edge). Whether realy used only to represents calls

		this.as = as;
		
		// Check if events come from same thread
		if (previous != null && as.getThreadNum() == previous.as.getThreadNum()) {
			this.previous = previous; 
		} else {
			this.previous = null;
		}
		
		this.returnValue = null;
		this.returnValueType = ReturnValues.RET_VAL_NOT_SET;
		
		this.lastCallReturnValue = null;
		this.lastCallReturnValueType = ReturnValues.RET_VAL_NOT_SET;
		
		this.hashCode = calculateHashCode();
	}
	
	/** 
	 * Makes copy of source {@link ThreadStateStack} but with set return value.
	 * 
	 * Note: This action take place during processing action in edge -> {@link #asEdgeProcessed} is set true.
	 * @param source Original ThreadStateStack entry to clone. Cannot be null.
	 * @param retVal Return value to set. Cann't be null.
	 */
	public ThreadStateStack(ThreadStateStack source, String retVal) {
		assert source != null;
		assert retVal != null;

		// Actualize Automaton state
		this.as = source.as;
		
		// Copy all other settings
		this.previous = source.previous;

		this.lastCallReturnValue = source.lastCallReturnValue;
		this.lastCallReturnValueType = source.lastCallReturnValueType;
		
		this.returnValue = retVal;
		this.returnValueType = ReturnValues.RET_VAL_SET;

		this.hashCode = calculateHashCode();
	}
	
	/** 
	 * Makes copy of source {@link ThreadStateStack} but with different {@link #AutomatonState}.
	 * 
	 * <br>Note: Used by {@link StatesMapper#setMappings(AutomatonState, VariablesEvaluation)}
	 * 
	 * @param source Original ThreadStateStack entry to clone. Cannot be null.
	 * @param as Automaton state to use in this instance.
	 * @param i Ignored parameter used to determine different version of constructors. 
	 */
	public ThreadStateStack(ThreadStateStack source, AutomatonState as, int i) {
		assert source != null;
		assert as != null;

		// Actualize Automaton state
		this.as = as;
		
		// Copy all other settings
		this.previous = source.previous;

		this.lastCallReturnValue = source.lastCallReturnValue;
		this.lastCallReturnValueType = source.lastCallReturnValueType;
		
		this.returnValue = source.returnValue;
		this.returnValueType = source.returnValueType;

		this.hashCode = calculateHashCode();
	}
	
	/**
	 * Make copy of source ThreadStateStack, but actualize 
	 *   {@link ThreadStateStack#lastCallReturnValue} according returningTss
	 *   return value.
	 *   
	 * <br>Note: Used to process return from functions. (Emit edges)
	 * <br>Note: Instead this constructor use {@link #processReturnEvent(ThreadStateStack)}
	 * 
	 * @param source {@link ThreadStateStack} to copy. Cann't be null.
	 * @param newLastCallReturnValueType Type of the return value to set.
	 * @param newLastCallReturnValue Return value to set.
	 */
	protected ThreadStateStack(ThreadStateStack source, ReturnValues newLastCallReturnValueType, String newLastCallReturnValue) {
		assert source != null;
		
		this.as = source.as;
		this.previous = source.previous;
		
		this.returnValue = source.returnValue;
		this.returnValueType = source.returnValueType;

		// Requirements on the newly set values
		assert (
				((newLastCallReturnValueType == ReturnValues.RET_VAL_NOT_SET) && (newLastCallReturnValue == null)) ||
				((newLastCallReturnValueType == ReturnValues.RET_VAL_SET) && (newLastCallReturnValue != null)) ||
				((newLastCallReturnValueType == ReturnValues.RET_VAL_UNDEFINED_EMIT) && (newLastCallReturnValue == null))
			);
		
		this.lastCallReturnValue = newLastCallReturnValue;
		this.lastCallReturnValueType = newLastCallReturnValueType;

		this.hashCode = calculateHashCode();
	}

	/**
	 * Gets copy of instance but returned {@link ThreadStateStack} has actualized {@link #lastCallReturnValue}
	 *  to return values from returningTss.
	 * 
	 * <br>Note: Is used to handle return events. (From functions)
	 * <br>Note: For more detail see {@link #StatesMapper(ThreadStateStack, ThreadStateStack)}
	 *
	 * @param returningTss {@link ThreadStateStack} where take return value. Cann't be null.
	 * @return Copy of instance with actualized lastCallReturnValue property
	 */
	public ThreadStateStack processReturnEvent(ThreadStateStack returningTss) {
		assert returningTss != null;
		return new ThreadStateStack(this, returningTss.getReturnValueType(), returningTss.getReturnValue());
	}
	
	/** 
	 * Gets copy of instance but with different {@link #AutomatonState}.
	 * 
	 * @param as Automaton state to use in this instance.
	 * @return Copy with different associated Automaton state.
	 */
	
	public ThreadStateStack changeAutomatonState(AutomatonState as) {
		assert as != null;
		return new ThreadStateStack(this, as, 1);
	}
	
	/**
	 * Gets copy of current instance but with last call return value set to {@link ReturnValues#RET_VAL_UNDEFINED_EMIT}
	 * @return Copy of the instance with modified cast call entries.
	 */
	public ThreadStateStack changeLastCallRetValTypeUndefEmit() {
		assert as != null;
		return new ThreadStateStack(this, ReturnValues.RET_VAL_UNDEFINED_EMIT, null);
	}
	
	public final AutomatonState getAutomatonState() {
		return as;
	}
	
	public final ThreadStateStack getPrevious() {
		return previous;
	}
	
	public final String getReturnValue() {
		return returnValue;
	}
	
	public final ReturnValues getReturnValueType() {
		return returnValueType;
	}
	
	public final String getLastCallReturnValue() {
		return lastCallReturnValue;
	}

	public final ReturnValues getLastCallReturnValueType() {
		return lastCallReturnValueType;
	}
	
	public final int hashCode() {
		return hashCode;
	}
	
	public String toString() {
		StringBuffer sb = new StringBuffer();
		
		sb.append("as=");
		  sb.append(as.toString());
		sb.append(", hash=");
		  sb.append(hashCode());
		  
		sb.append(", prevHash=");
		  sb.append(previous==null?"null":previous.hashCode());
		  
		sb.append(", retVal=");
		  sb.append(returnValue);
		
		sb.append(", retValType=");
		  sb.append(returnValueType);
		  
		sb.append(", lastCallretVal=");
		  sb.append(lastCallReturnValue);

		sb.append(", lastCallRetValType=");
		  sb.append(lastCallReturnValueType);

	  return sb.toString();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		ThreadStateStack other = (ThreadStateStack) obj;
		if (other.hashCode() != hashCode()) {
			return false;
		}
		
		if (as == null) {
			if (other.as != null) {
				return false;
			}
		} else if (!as.equals(other.as)) {
			return false;
		}
		if (lastCallReturnValue == null) {
			if (other.lastCallReturnValue != null) {
				return false;
			}
		} else if (!lastCallReturnValue.equals(other.lastCallReturnValue)) {
			return false;
		}
		if (lastCallReturnValueType == null) {
			if (other.lastCallReturnValueType != null) {
				return false;
			}
		} else if (!lastCallReturnValueType
				.equals(other.lastCallReturnValueType)) {
			return false;
		}
		if (returnValue == null) {
			if (other.returnValue != null) {
				return false;
			}
		} else if (!returnValue.equals(other.returnValue)) {
			return false;
		}
		if (returnValueType == null) {
			if (other.returnValueType != null) {
				return false;
			}
		} else if (!returnValueType.equals(other.returnValueType)) {
			return false;
		}
		if (previous == null) {
			if (other.previous != null) {
				return false;
			}
		} else if (!previous.equals(other.previous)) {
			return false;
		}
		return true;
	}

	/**
	 * Calculate hash code for object.
	 * 
	 * Note: All properties (except {@link #hashCode}) has to be set before this call.
	 * 
	 * @return hashCode value.
	 */
	final private int calculateHashCode() {
		int result = 0;
		result = (result * HASH_CONSTANT) + as.hashCode();
		result = (result * HASH_CONSTANT) + (previous != null ? previous.hashCode() : 0);
		result = (result * HASH_CONSTANT) + (returnValue != null ? returnValue.hashCode() : 0);
		result = (result * HASH_CONSTANT) + returnValueType.hashCode();
		result = (result * HASH_CONSTANT) + (lastCallReturnValue != null ? lastCallReturnValue.hashCode() : 0);
		result = (result * HASH_CONSTANT) + lastCallReturnValueType.hashCode();
		return result;
	}
	
	
	
}
