package org.ow2.dsrg.fm.tbpjava.checker;


import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.Checker;
import org.ow2.dsrg.fm.tbpjava.EnvGenerator.EnvironmentDescription;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAccept;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;

/**
 * Map automatons ({@link AutomatonState}) of one threads together.
 * <br>
 * Process incoming events and moves by automatons.
 *
 */

//TODO poradek v navratovych hodnotach
//TODO
//TODO: How should it be with Provided call recursion -> it means if one provided method calls another provided call.
//TODO: Same from what if THREAD_section thread calls some provided method ... is it permited or not?
// Forbidden (report error)

public class ThreadAutomatons {
	private static final boolean DEBUG = false; // Set to true if you want to see debug prints from this object

	/**
	 * Specify type of thread that instance of {@link ThreadAutomatons} represent.
	 * 
	 */
	public enum ThreadType {
		ENVIRONMENT_THREAD, /// Represented thread is created by environment. It's standard thread, that is expected to call provided interface and then required one. 
		THREAD_SECTION, /// Represent thread that was created by component. (Either in constructor or in any provided method). This thread is expected to call methods from required interfaces. 
		NONE_EVENT_THREAD, /// Any other thread that cannot emit any events. Typically created by objects used as parameters. 
	}

	/**
	 * Specify behavior of Thread automaton instance.
	 * Note: Unlike {@link ThreadType} this type can change in {@link ThreadAutomatons#prev} list.
	 */
	public enum AutomatonsType {
		THREAD_SECTION, /// This instance represents thread section events. (If any event on provided interface occur, then descendant has {@link #PROVISION_EVENTS} type 
		PROVISION_EVENTS, /// This instance expect event on provided interface as first
		NONE_EVENTS /// This instance don't expect any events. Incoming event cause error.
	}

	private final LTSAComponentSpecification specification; /// Parsed specification of processed component
	private final StatesMapper varStatesMap; /// Used for mapping between automatons states and sets of variables arrays (variablesState)
	private final AutomatonsMoves mover; /// Move by automaton states according evaluations

	private final ThreadType threadType; // What type of thread instance represents
	private final int threadNum; /// Number of thread whose events processing
	/**
	 * Stack with all (unpaired)events that the thread is currently inside.
	 * 
	 * <br>Note: If not "don't represent thread" than in the bottom there is provided method called from the environment.
	 *   then in the JPF + generated environment case
	 *     - subsequent provided calls takes place in stack and eventually in the top required call.
	 *   in the "whole component application" mode
	 *     - there are provided calls and required calls intermixed.
	 * 
	 * <br>Note: Simulates "call stack" of represented thread. 
	 */
	private final Stack<EventItem> eventStack = new Stack<EventItem>(); 
	
	private int reqItfsCallDepth = 0; /// Calculate depth 
	private int provItfsCallDepth = 0; /// Calculate depth 

	private final ThreadAutomatons prev; /// Make linked list to previous versions with older state. (Simulate stack) 
	private final AutomatonsType type; /// Represent type of instance. Specify which types of events are expected.

	private int eventParserRoundNum; /// Hold value from eventParser ... count of processEvent calls (to identify when was object created)  

	private final EnvironmentDescription envDesc; /// Description of the current environment. 
	
	private Set<AutomatonState> automStates = new HashSet<AutomatonState>(); /// Main property, holds associated {@link AutomatonState} of one thread
	
	// Error reporting
	private boolean errorFlag = false;
	private String errorString = null;

	/**
	 * Create new {@link ThreadAutomatons} instance.
	 * This constructor is intended to create representative for newly created thread. 
	 * 
	 * @param specification Parsed specification of component that is processed.
	 * @param varStatesMap Mapper used for mapping {@link VariablesEvaluation} to {@link AutomatonState}s
	 * @param threadType Specify type of thread. (Standard / Thread section / No events)
	 * @param eventParserRoundNum Number of round (number of calls {@link EventParser#processEvents(List)}) in which object was created
	 * @param threadNum Number of created thread
	 */
	public ThreadAutomatons(LTSAComponentSpecification specification, StatesMapper varStatesMap, ThreadType threadType, int eventParserRoundNum, int threadNum, EnvironmentDescription envDesc) {
		if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + "." + ThreadAutomatons.class.getSimpleName() + " -   creating TA for new thread"); }

		this.specification = specification;
		this.varStatesMap = varStatesMap;
		this.prev = null;
		this.threadType = threadType;
		this.type = getDefaultAutomatonsTypeForThreadType(threadType);
		this.eventParserRoundNum = eventParserRoundNum;
		this.threadNum = threadNum;
		this.envDesc = envDesc;
		this.mover = new AutomatonsMoves(specification, varStatesMap);
		
		if (type == AutomatonsType.THREAD_SECTION) {
			// Initialize thread section - add start states for all automatons
			if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + "." + ThreadAutomatons.class.getSimpleName() + "(...) -   creating thread section start states"); }
			// For each thread save start state and add variables mapping for them
			Map<String, LTSAAutomaton> threadMap = specification.getThreads();
			for(Entry<String, LTSAAutomaton> threadMapEntry : threadMap.entrySet()) {
				final LTSAAutomaton threadAutomaton = threadMapEntry.getValue();
				// Create AutomatonState for start state
				AutomatonState threadStartAS =  new AutomatonState(threadAutomaton, threadAutomaton.getStart(), threadNum);
				automStates.add(threadStartAS);
				varStatesMap.setMappingsAllStatesAsCall(threadStartAS);
				// Actively move from start state
				moveAutomatonsForward();
			}

		}
	}

	/**
	 * Makes new empty instance that is descendant of given {@link ThreadAutomatons} object.
	 * 
	 * @param prev Predecessor of created instance.
	 */
	protected ThreadAutomatons(ThreadAutomatons prev, int eventParserRoundNum) {
		assert prev != null;
		if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + "." + ThreadAutomatons.class.getSimpleName() + "(prev=" + prev + ", evPRN=" + eventParserRoundNum + ") -   new successort TA"); }

		this.specification = prev.specification;
		this.varStatesMap = prev.varStatesMap;
		this.threadNum = prev.threadNum;
		this.mover = prev.mover;
		this.prev = prev;
		this.type = ThreadAutomatons.getSuccessorAutomatonsType(prev.type);
		this.threadType = prev.threadType;
		this.eventParserRoundNum = eventParserRoundNum;
		this.envDesc = prev.envDesc;
	}

	/**
	 * Cloning constructor.
	 * Makes new instance that is copy of given source instance. 
	 * The only difference is in {@link #eventParserRoundNum}.
	 * 
	 * <br>Note: Doesn't clone stored instances in {@link #automStates}. Makes only new {@link HashSet} with same entries. 
	 * 
	 * @param source Source {@link ThreadAutomatons} to be cloned.
	 * @param eventParserRoundNum Number of round (number of calls {@link EventParser#processEvents(List)}) in which object was created
	 * @param ignored This parameter is ignored. Has to be always true. Used only meaning is to distinguish between other constructors.
	 */
	protected ThreadAutomatons(ThreadAutomatons source, int eventParserRoundNum, boolean ignored) {
		assert source != null;
		assert ignored == true;
		if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + "." + ThreadAutomatons.class.getSimpleName() + "(source=" + source + ", evPRN=" + eventParserRoundNum + ", ignored=" + ignored + ") -   new clone TA"); }

		this.specification = source.specification;
		this.varStatesMap = source.varStatesMap;
		this.threadNum = source.threadNum;
		this.mover = source.mover;
		this.prev = source.prev;
		this.type = source.type;
		this.threadType = source.threadType;
		this.eventParserRoundNum = eventParserRoundNum;

		this.errorFlag = source.errorFlag;
		this.errorString = source.errorString;

		this.reqItfsCallDepth = source.reqItfsCallDepth;
		this.provItfsCallDepth = source.provItfsCallDepth;
		this.eventStack.addAll( source.eventStack);
		
		this.envDesc = source.envDesc;
		
		automStates = new HashSet<AutomatonState>(source.automStates);

	}

	/**
	 * Makes {@link ThreadAutomatons} deep copy. It's used for 
	 * Only difference is eventParserRoundNum that is set according parameter.
	 * 
	 * @param eventParserRoundNum Current event parse call count. (Time when clone was created)
	 */
	public ThreadAutomatons clone(int eventParserRoundNum) {
		if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + ".clone(evPRN=" + eventParserRoundNum + ");"); }
		return new ThreadAutomatons(this, eventParserRoundNum, true);
	}

	/**
	 * Getter for eventParserRoundNum property. Gets call count of {@link EventParser#processEvents(List)}
	 *   where his object was created (by cloning or by constructor).
	 *
	 * @return Get number of call {@link EventParser#processEvents(List)}.
	 */
	public int getEventParserRoundNum() {
		return eventParserRoundNum; 
	}
	
	public ThreadAutomatons getPrevThreadAutomaton() {
		return prev;
	}
	
	/**
	 * Gets true if any error occur during processing of events.
	 * @return Gets true if any error occur during processing of events.
	 */
	public final boolean getError() {
		return errorFlag;
	}
	
	public final String getErrorString() {
		return errorString;
	}
	
	public ThreadAutomatons processEventProvidedCall(EventItem event, int eventParserRoundNum) {
		if (DEBUG) { System.out.println("DEBUG -     " +  ThreadAutomatons.class.getSimpleName() + ".processEventProvidedCall(event=" + event + ", eventPRN=" + eventParserRoundNum + ")"); }
		
		if (type == AutomatonsType.NONE_EVENTS) {
			String errorStr = "Provided call event (event=" + event + ") from thread where no events expected (threadNum=" + threadNum + ", threadType=" + threadType + ")";
			setError(errorStr);
			throw new RuntimeException(errorStr);
		}

		if (provItfsCallDepth > 0) {
			// Subsequent provided call (provided call while handling other provided call)
			//  (we done "in-lining")
			provItfsCallDepth++;
			eventStack.push(event);
			
			return this;
		}
		// Initial provided call from the environment (is handled by new instance of ThreadAutomaton)
		ThreadAutomatons result = new ThreadAutomatons(this, eventParserRoundNum); 

		// Initialize automaton states
		TBPParsedAccept call = envDesc.line2ProvidedCall.get(event.line);
		if (call == null) {
			throw new RuntimeException(Checker.ERROR_PREFIX + "Unexpected provided call " + event.fullName + " on line " + event.line);
		}

		Binding binding = findAutomaton(specification, event, call);
		LTSAAutomaton automaton = specification.getReactions().get(binding);

		AutomatonState startState = new AutomatonState(automaton, automaton.getStart(), event.thread);
		result.automStates.add(startState);

		// Update variables states mapping
		result.varStatesMap.setMappingsAllStatesAsCall(startState);

		result.moveAutomatonsForward(); // Actively move with automaton states
		result.provItfsCallDepth++;
		result.eventStack.push(event);

		return result;
	}
	
	public ThreadAutomatons processEventRequiredCall(EventItem event, int currentEventParserRoundNum) {
		if (DEBUG) { System.out.println("DEBUG -     " +  ThreadAutomatons.class.getSimpleName() + ".processEventRequiredCall(" + event + ", currentEventParserRoundNum=" + currentEventParserRoundNum + ")"); }

		if (type == AutomatonsType.NONE_EVENTS) {
			String errorStr = "Required call event (event=" + event + ") from thread where no events expected (threadNum=" + threadNum + ", threadType=" + threadType + ")";
			setError(errorStr);
			throw new RuntimeException(errorStr);
		}

		reqItfsCallDepth++;
		eventStack.push(event);
		if (reqItfsCallDepth > 1) {
			return this; // No error found
		}
		
		
		automStates = mover.moveAutomatonsEnterMutexes(automStates); // Enter into mutexes
		automStates = mover.filterStates(automStates, event); // Remove states that doesn't represent current event. (Bad path in automatons)
		// Moving in automatons take place after return (in current layer)

		//Check if any automaton state remain -> 
		//  if no state remains there is no automaton trace that 
		//  is equivalent with currently executed event trace
		// -> implementation violates specification
		if (automStates.isEmpty() == true) {
			setError("Implementation violates specification - no possible automaton move found");
			return this;
		}
		
		
		// Map calling states into start states of called automaton
		ThreadAutomatons newLayer = new ThreadAutomatons(this, currentEventParserRoundNum);
		newLayer.reqItfsCallDepth++;
		newLayer.eventStack.push(event);
		final boolean startStateFound = mover.mapCallingStatesWithCalledStartState(this, automStates, newLayer.automStates);
		
		
		if (startStateFound == true) {
			// Move actively with start states
			newLayer.moveAutomatonsForward();
			return newLayer; // Any state to process next calls was added into new newLayer ... let new layer process incoming calls 
		} 

		return this;

	}

	public ThreadAutomatons processEventRequiredReturn(EventItem event, int currentEventParserRoundNum) {
		if (DEBUG) { System.out.println("DEBUG -     " +  ThreadAutomatons.class.getSimpleName() + ".processEventRequiredReturn( event=" + event + ", currentEventParserRoundNum=" + currentEventParserRoundNum + ")"); }

		if (type == AutomatonsType.NONE_EVENTS) {
			String errorStr = "Required return event (event=" + event + ") from thread where no events expected (threadNum=" + threadNum + ", threadType=" + threadType + ")";
			setError(errorStr);
			throw new RuntimeException(errorStr);
		}

		if (reqItfsCallDepth <= 0) {
			setError("Illegal required return - not paired with any call (more returns than calls)"); 
			return this; // Error no return expected
		} 
		
		
		// Return on undefined emit (processed by this layer) -> check if events are related 
		EventItem callEvent = eventStack.pop();
		reqItfsCallDepth--;
		
		if (callEvent.isAssociatedEvent(event) == false) {
			setError("Illegal required return - not paired with stored call (different return than call)"); 
		}

		if (reqItfsCallDepth == 0 && !eventStack.isEmpty()) {
			// Last required return -> we should move the automatons
			//  This ThreadAutomaton doesn't represent "Automaton of called required method"
			
			// Transform AS from Unprocessed to EdgeActionProcessed state 
			automStates = mover.edgeActicionProcess(automStates);
			// Move actively from call state ... to new states
			this.moveAutomatonsForward();
			return this;
		
		} else if (reqItfsCallDepth == 0 && eventStack.isEmpty()) {
			// ThreadAutomaton represent (automaton of) Required method
			//  Note: This is not typical case (similar to Provided call)
			
			// reqItfsCallDepth == 0
			// eventStack.isEmpty() == true;
			// Check return from defined emit -> 
			//   check if event paired with calling event (stored in previous layer)
			//   move by automatons
			//   process return values
			//   update reqItfsCallDepth in previous layer 

			if (prev == null) {
				// No previous known state from call come... no calling layer
				setError("Illegal required return - not associated call found (return on unknown call)"); 
				return null;
			}

			EventItem prevLastCallEvent = prev.eventStack.pop();
			prev.reqItfsCallDepth--;
			if (prevLastCallEvent.isAssociatedEvent(event) == false) {
				setError("Illegal required return - not asociated with call (return=" + event + ", call=" + prevLastCallEvent); 
				return null;
			}
			
			
			automStates = mover.filterStates(automStates, event);
			// Remains only return states

			//Check if any automaton state remain -> 
			//  if no state remains there is no automaton trace that 
			//  is equivalent with currently executed event trace
			// -> implementation violates specification
			if (automStates.isEmpty() == true) {
				setError("Implementation violates specification - no possible automaton move found");
				return this;
			}

			// After moving of automatons should be all stored states final.
			//  --> update mappings and remove this layer
			ThreadAutomatons resultLayer = prev;
			// Null means this is first provided call for given thread -> no layer where continue after return

			if (resultLayer.getEventParserRoundNum() != currentEventParserRoundNum) {
				// Predecessor (prev property) is from different (older) CheckerEventParser.processEvents call 
				//  -> It means prev can by stored in stack (EventParset.state) for further usage 
				// (so we cannot modify them) -> we have to make copy 
				resultLayer = resultLayer.clone(currentEventParserRoundNum);
			}
			
			// process return values
			mover.processReturnValues(this, this.automStates, event.thread);

			// Transform AS from Unprocessed to EdgeActionProcessed state 
			resultLayer.automStates = mover.edgeActicionProcess(resultLayer.automStates);
			
			// Return was processed move forward to new state
			resultLayer.moveAutomatonsForward();
			
			return resultLayer;
		}
	
		return this;

	}
	
	public ThreadAutomatons processEventProvidedReturn(EventItem event, int currentEventParserRoundNum) {
		if (DEBUG) { System.out.println("DEBUG -     " +  ThreadAutomatons.class.getSimpleName() + ".processEventProvidedReturn(" + event + ")"); }
		
		if (type == AutomatonsType.NONE_EVENTS) {
			String errorStr = "Provided return event (event=" + event + ") from thread where no events expected (threadNum=" + threadNum + ", threadType=" + threadType + ")";
			setError(errorStr);
			throw new RuntimeException(errorStr);
		}

		EventItem provCallEvent = eventStack.pop();
		provItfsCallDepth--;
		if (provCallEvent.isAssociatedEvent(event) == false) {
			setError("Illegal provided return - not asociated with call (return=" + event + ", call=" + provCallEvent); 
			return null;
		}

		if (provItfsCallDepth > 0) {
			return this;
		}
		//Return events has bad lines .. we use associated call events :-)
		automStates = mover.moveAutomatonsEnterMutexes(automStates);

		TBPParsedAccept call = envDesc.line2ProvidedCall.get(provCallEvent.line);
		if (call == null) {
			throw new RuntimeException("Return from invalid call (invalid call line) - " + provCallEvent.line);
		}

		automStates = mover.filterStates(automStates, event);

		automStates = mover.filterStatesOnReturnValue(automStates, event, call.getReturnValue());

		// After filtering only return event remains

		//Check if any automaton state remain -> 
		//  if no state remains there is no automaton trace that 
		//  is equivalent with currently executed event trace
		// -> implementation violates specification
		if (automStates.isEmpty() == true) {
			setError("Implementation violates specification - no possible automaton move found");
			return this;
		}

		ThreadAutomatons resultLayer = prev;
		// Null means this is first provided call for given thread -> no layer where continue after return 

		if (resultLayer != null && resultLayer.getEventParserRoundNum() != currentEventParserRoundNum) {
			// Predecessor (prev property) is from different (older) CheckerEventParser.processEvents call 
			//  -> It means prev can by stored in stack (EventParset.state) for further usage 
			// (so we cannot modify them) -> we have to make copy 
			resultLayer = resultLayer.clone(currentEventParserRoundNum);
		}

		varStatesMap.setMappingsAllStatesAsReturn(event.thread);

		return resultLayer;
	}
	
	/**
	 * @return True if it's possible that thread ends and protocol is not violated.
	 *   It means exist automaton in final state or we return from all provided events.
	 * 
	 * Note: Final states are recently used mainly for thread section automatons to mark end of execution. Reaction automatons typically ends with return edge.  
	 */
	public boolean isEndState() {
		return ((type == AutomatonsType.THREAD_SECTION) && (AutomatonsMoves.isEndStateInSet(automStates))) ||
		       ((type == AutomatonsType.PROVISION_EVENTS) && (automStates.size() == 0 && prev == null)) || // initial environment thread representant
		       ((type == AutomatonsType.PROVISION_EVENTS) && (AutomatonsMoves.isEndStateInSet(automStates))) || // failsafe for environment
		       ( type == AutomatonsType.NONE_EVENTS );
	}
	
	/*
	 * Set error flag and error string. 
	 * <br>For internal purposes only. Intended only for {@link AutomatonsMoves} 
	 */
	/* default */ void setError(String errorString) {
		// Store only first error
		if (errorFlag == false) {
			this.errorFlag = true;
			this.errorString = errorString;
		}
	}
	
	/**
	 * Moves actively saved states forward. Update saved automatons states.
	 * <br>Note: See {@link AutomatonsMoves#moveAutomatons(Set)}
	 * <br>Note: After each call {@link #moveAutomatonsForward} should come {@link AutomatonsMoves#filterStates(Set, EventItem)}. Don't call twice.
	 * 
	 * @param threadNum Specify thread that caused moving.
	 */
	private void moveAutomatonsForward() {
		automStates = mover.moveAutomatons(automStates);

		if (automStates.isEmpty() == true) {
			setError("Implementation violates specification - no possible automaton move found");
		}
	}

	/**
	 * Get for given threadType associated default value of Automaton type that will be used
	 *  as type of Automaton on the bottom of the {@link #prev} list.
	 * 
	 * @param threadType to process
	 * @return {@link AutomatonsType} associated to given threadType
	 */
	static private AutomatonsType getDefaultAutomatonsTypeForThreadType(ThreadType threadType) {
		if (threadType == ThreadType.ENVIRONMENT_THREAD) {
			return AutomatonsType.PROVISION_EVENTS;
		}
		
		if (threadType == ThreadType.THREAD_SECTION) {
			return AutomatonsType.THREAD_SECTION;
		}
		if (threadType == ThreadType.NONE_EVENT_THREAD) {
			return AutomatonsType.NONE_EVENTS;
		}

		throw new RuntimeException("New unknown " + ThreadType.class.getSimpleName() + " entry type " + threadType);
	}

	/**
	 * Gets {@link AutomatonsType} for newly created successor.
	 * 
	 * <br>Note: Only thread section change type into standard PROVISION_EVENTS type.
	 * 
	 * @param prevAutomatonsType {@link AutomatonType} of predecessor.
	 * @return {@link AutomatonsType} that should have successor of {@link ThreadAutomatons} that has given {@link AutomatonType}.
	 */
	static private AutomatonsType getSuccessorAutomatonsType(AutomatonsType prevAutomatonsType) {
		if (prevAutomatonsType == AutomatonsType.PROVISION_EVENTS) {
			return AutomatonsType.PROVISION_EVENTS;
		}
		
		if (prevAutomatonsType == AutomatonsType.THREAD_SECTION) {
			return AutomatonsType.PROVISION_EVENTS;
		}
		if (prevAutomatonsType == AutomatonsType.NONE_EVENTS) {
			return AutomatonsType.NONE_EVENTS;
		}

		throw new RuntimeException("New unknown " + AutomatonsType.class.getSimpleName() + " entry type " + prevAutomatonsType);
	}
	
	/**
	 * @return Gets type of represented thread.
	 */
	final public ThreadAutomatons.ThreadType getThreadType() {
		return threadType;
	}
	
	// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	
	private final static Binding findAutomaton(LTSAComponentSpecification specification, EventItem event, TBPParsedAccept call) {
		if (DEBUG) { System.out.println("DEBUG - ThreadAutomatons.findAutomaton( specification=" + specification + ", event=" + event + ", call=" + call + ");"); }

		Binding result = null;
		String searchedMethod = event.fullName;
		for(Binding binding : specification.getReactions().keySet()) {
			if (searchedMethod.equals( binding.getFullname()) == true) {
				// Name matched -> now we check parameters
				
				final boolean sameParameters = checkParameters(binding.getValues(), call.getMethodCall().getParamDecl());
				if (sameParameters) {
					result = binding;
					if (DEBUG) { System.out.println("DEBUG - ThreadAutomatons.findAutomaton - findMatch - " + result); }
				}
			}
		}
		return result;
	}

	/**
	 * Compares two parameter lists. Check whether the use same parameter values.
	 * @param params1 Parameter list from Binding.
	 * @param params2 Parameter list from the TBP spec
	 * @return true if values of all parameter are same, false otherwise
	 */
	private final static boolean checkParameters(List<Reference> params1, List<String> params2) {
		if (params1 == null || params2 == null) {
			return false;
		}
		
		if (params1.size() != params2.size()) {
			return false;
		}
		
		for(int i = 0; i < params1.size(); i++) {
			if (params1.get(i).getName().equals(params2.get(i)) == false) {
				return false;
			}
		}
		return true;
	}
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append( ThreadAutomatons.class.getSimpleName());
		sb.append("\n");
		sb.append("  Internal state: myHash=" + hashCode());
		sb.append(", threadNum=" + threadNum);
		sb.append(", type=" + type);
		sb.append("\n");
		sb.append("    errorFlag=" + errorFlag);
		sb.append(", errorString=" + errorString);

		sb.append("\n");
		sb.append("    eventParserRoundNum=" + eventParserRoundNum);
		sb.append(", prev=" + (prev!=null?prev.hashCode():"null"));
		
		sb.append("\n");
		sb.append("    provItfsCallDepth=" + provItfsCallDepth);
		sb.append(", reqItfsCallDepth=" + reqItfsCallDepth);
		sb.append("\n");
		sb.append("    eventStack=" + eventStack);
		sb.append("\n");

		sb.append("  Dumping states:\n");
		sb.append(DEBUG_StatesToString(4, automStates, varStatesMap));
		return sb.toString();
	}
	
	public static String DEBUG_StatesToString(int indent, Set<AutomatonState> states, StatesMapper varStatesMap) {
		StringBuffer sb = new StringBuffer();
		for(AutomatonState as : states) {
			DEBUG_indetStringBuffer(sb, indent);
			sb.append(as);
			sb.append("\n");
			
			Set<VariablesEvaluation> automatonStates = varStatesMap.DEBUG_getAssociatedMappings(as);
			if (automatonStates != null) {
				for(VariablesEvaluation state : automatonStates) {
					DEBUG_indetStringBuffer(sb, indent +2);
					sb.append(state);
					sb.append('\n');
				}
			}
		}
		
		return sb.toString();
	}
	
	/**
	 * Adds indent spaces into string buffer. If no buffer given creates new empty one.
	 * 
	 * @param sb String buffer where add spaces. If given it's used as return value.
	 * @param indent How many spaces append.
	 * @return Return string buffer where was added indent spaces.
	 */
	public static StringBuffer DEBUG_indetStringBuffer(StringBuffer sb, int indent) {
		if (sb == null) {
			sb = new StringBuffer();
		}
		for(int i = 0; i < indent; i++) {
			sb.append(' ');
		}
		
		return sb;
	}

//	/**
//	 * Makes new ThreadAutomaton.
//	 * 
//	 * @param specification Parsed TBP specification of checked component.
//	 * @param eventParserRoundNum Number of round in which object was created
//	 * @param clone Specify if automaton is newly created one or is created by cloning. It's is used to determine if initialization (in THREAD_SECTION case) should by done. 
//	 */
//	protected ThreadAutomatons(LTSAComponentSpecification specification, StatesMapper varStatesMap, ThreadAutomatons prev, ThreadType type, int eventParserRoundNum, int threadNum, boolean clone) {
//		this.specification = specification;
//		this.varStatesMap = varStatesMap;
//		this.prev = prev;
//		this.type = type;
//		this.eventParserRoundNum = eventParserRoundNum;
//		this.threadNum = threadNum;
//		
//		this.mover = new AutomatonsMoves(specification, varStatesMap);
//		
//		// Thread automatons initialization (only for bottom ... first layer)
//		// Note: clone parameter blocks multiple initialization if cloning of TreadAutomaton in first layer take place
//		if (type == automatonsType.THREAD_SECTION && prev == null && !clone) {
//			if (DEBUG) { System.out.println("DEBUG -     " + ThreadAutomatons.class.getSimpleName() + "." + ThreadAutomatons.class.getSimpleName() + " -   creating thread section start states"); }
//			// For each thread save start state and add variables mapping for them
//			Map<String, LTSAAutomaton> threadMap = specification.getThreads();
//			for(Entry<String, LTSAAutomaton> threadMapEntry : threadMap.entrySet()) {
//				final LTSAAutomaton threadAutomaton = threadMapEntry.getValue();
//				// Create AutomatonState for start state
//				AutomatonState threadStartAS =  new AutomatonState(threadAutomaton, threadAutomaton.getStart(), null, threadNum);
//				automStates.add(threadStartAS);
//				varStatesMap.setMappingsAllStatesAsCall(threadStartAS);
//				// Actively move from start state
//				moveAutomatonsForward();
//			}
//		}
//	}
//
}