package org.ow2.dsrg.fm.tbpjava.checker;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.checker.ThreadStateStack.ReturnValues;
import org.ow2.dsrg.fm.tbpjava.utils.HashMapWithHashCode;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.ltsa.State;
import org.ow2.dsrg.fm.tbplib.resolved.LastCallRef;

/**
 * Maps {@link VariablesEvaluation} with {@link AutomatonState}. Class {@link StatesMapperLayer} is a part of 
 *   stack in {@link StatesMapper}. {@link StatesMapper} process stack of {@link StatesMapperLayer} and forwards
 *   business method invocation into {@link StatesMapperLayer} methods with same name.
 * <br>
 * @see StatesMapper
 * <br>
 * <h2>{@link StatesMapperLayer} details.</h2>
 * <br>
 * One variables evaluation is mapped with one automaton state from each thread.
 * With one automaton state can be associated many variables evaluation. 
 * Changes of variables evaluations created by one event 
 *   (clone one evaluation into more different evaluations, removing evaluation)
 *   affect all other mappings (of this variables evaluation) with other automaton states. 
 * <br>
 * <h3>Using patterns</h3>
 * <br>Variables evaluation can be in three states -> NOT_MAPPED, MAPPED, PROCESSING
 * <br><ul>
 *  <li>NOT_MAPPED - in this state variable evaluation is not mapped to any automaton state 
 *     and not processed by {@link StatesMapper}. Such variables evaluation should not exists
 *     accessible to user.</li>
 *  <li>MAPPED - in this state variable evaluation is mapped to one or many automaton states. 
 *    One state for one thread. {@link VariablesEvaluation} can be moved into MAPPED state
 *    from PROCESSING state by calling {@link #setMapping(AutomatonState, VariablesEvaluation)}
 *     and other setMappings* routine</li>
 *  <li>PROCESSING - during automatons moving is variable evaluation in this state.
 *     Variable can move from MAPPED into PROCESSING state by calling 
 *     {@link StatesMapper#getMappings(AutomatonState)}. In this state you can derive
 *     new {@link VariablesEvaluation} by calling {@link VariablesEvaluation#setVariableValue(String, String)}
 *     Return value can be set by calling {@link #setReturnValue(int, VariablesEvaluation, String)} or
 *     obtained by calling {@link #getReturnValue(int, VariablesEvaluation)}.
 *     
 *   Creating copy of layer {@link StatesMapperLayer#StatesMapperLayer(StatesMapperLayer)} can take
 *     place only if no state is in PROCESSING state (this states aren't cloned).
 *   Same conditions holds for {@link #setMappingsAllStatesAsCall(AutomatonState)} and
 *    {@link #setMappingsAllStatesAsReturn(int)} 
 *   
 */
final public class StatesMapperLayer {
	/* 
	 * Mappings internals implementation characteristic
	 * 
	 * If variables evaluation is in MAPPED state than
	 *  variables evaluation is save in reverse evaluations2automatons map 
	 *  and has forward mapping in automatons2evaluations for each 
	 *  thread number that was set (only top level of ThreadStateStack).
	 *  And each automaton state has for all associated variables 
	 *  evaluation reverse mapping in evaluations2automaton under its 
	 *  thread number on the top of ThreadStateStack.
	 *  
	 * If variable is in PROSESSING state than
	 *  it's only in processingStates map. Variable evaluation is not
	 *  present in evaluations2automatons nor in any values subsets in automatos2evaluations.
	 *  
	 *  Any automatons2evaluations entry doesn't have assigned null as variable evaluation set.
	 *  Any evaluations2automatons entry doesn't have assigned null as subMap value entry. 
	 *
	 * Note: Two different instances of variables evaluation are considered to be different even if they maps same values for all variables.
	 *   This means that we can proceed same variables values more times. But this is needed for easier variable evaluation handling. 
	 *   (if we join them together we could have problem on return -> it could have problem with return values, on return makes problems)
	 *   
	 * Note: Error handling. Errors are printed into System.err stream, prepared error string saved into error constant error flag is set,and then throw as RuntimeException  
	 *
	 */

	private static final boolean DEBUG = false;
	private static final boolean DEBUG_INFO = false; // More talky debug informations
	private static final boolean INTERNAL_CHECKS = false; // Enables/disables internal consistency checking
	
	private static final double CONST_HASHMAP_GROWING = 1.5;

	private Set<HashMapWithHashCode<Integer, ThreadStateStack>> mappedState; /// All mappings in the MAPPED state. Note: Special handling is necessary. See HashMapWithHashCode
	
	private Map<AutomatonState, Set<VariablesEvaluation>> automatons2evaluations; /// Maps automatons states into variables evaluation that are associated with automaton state.
	private Map<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>> evaluations2automatons; /// Reverse mapping. For each variable evaluation in MAPPED state holds by thread separated stacks with automaton states that are mapped to variables evaluation.
	
	private Map<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>> processingState; // Holds mappings for {@link VariableEvaluation} at PROCESSING state.
	
	private boolean errorFlag; /// Marks if any error during processing take place.
	private String errorString; /// Standardized error string (for first error)
	private String errorStringDetails; /// Holds more details about error (for first error)
	
	public static final String ERROR_MESSAGE_NO_ERROR = "no error"; /// Marks that no error take place
	public static final String ERROR_MESSAGE_ILLEGAL_RETURN = "return without associated call"; /// {@link #setMappingAsReturn(int, VariablesEvaluation)} call without previous Call in given thread. More return events than call events.
	public static final String ERROR_MESSAGE_NOT_IN_PROCESSING_STATE = "variables evaluation not in PROCESSING state"; /// {@link VariablesEvaluation} not in excepted PROCESSIN state as written in documentation.
	public static final String ERROR_MESSAGE_NOT_IN_MAPPED_STATE = "variables evaluation not in MAPPED state"; /// {@link VariablesEvaluation} not in excepted MAPPED state as written in documentation.
	public static final String ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD = "variables evaluation not mapped for given thread"; /// {@link VariablesEvaluation} 
	
	/**
	 * Makes initial stack layer. For details about initial layer see {@link StatesMapper}.
	 *
	 * @param comp Component for which map {@link AutomatonState} with {@link VariablesEvaluation}. Cann't be null.
	 * @param statesMapper States mapper that creates layer belongs to. 
	 */
	public StatesMapperLayer(LTSAComponentSpecification comp, StatesMapper statesMapper) {
		assert comp != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + "." + StatesMapperLayer.class.getSimpleName() + "()"); }
		
		automatons2evaluations = new HashMap<AutomatonState, Set<VariablesEvaluation>>();
		evaluations2automatons = new HashMap<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>>();
		
		processingState = new HashMap<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>>();
		mappedState = new HashSet<HashMapWithHashCode<Integer, ThreadStateStack>>();

		errorFlag = false;
		errorString = ERROR_MESSAGE_NO_ERROR;
		errorStringDetails = null;
		
		setInitialContent(comp, statesMapper);
		
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}

	/**
	 * Makes deep copy of source layer.
	 *
	 * @param source Source layer to copy. Cann't be null.
	 */
	public StatesMapperLayer(StatesMapperLayer source) {
		assert source != null;
		
		if (DEBUG) { System.out.println( StatesMapperLayer.class.getSimpleName() + "." + StatesMapperLayer.class.getSimpleName() + "(source=" + source + ")"); System.out.println("  Source -> "); source.DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { source.DEBUG_checkConsistency(); }
		
		automatons2evaluations = new HashMap<AutomatonState, Set<VariablesEvaluation>>((int)(source.automatons2evaluations.size() * CONST_HASHMAP_GROWING));
	
		mappedState = new HashSet<HashMapWithHashCode<Integer,ThreadStateStack>>((int)(source.mappedState.size() * CONST_HASHMAP_GROWING));
		
		evaluations2automatons = new HashMap<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>>((int)(source.evaluations2automatons.size() * CONST_HASHMAP_GROWING));
		processingState = new HashMap<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>>();
		
		errorFlag = source.errorFlag;
		errorString = source.errorString;
		errorStringDetails = source.errorStringDetails;

		// Copy all evaluation in MAPPED state
		for(VariablesEvaluation values : source.evaluations2automatons.keySet()) {

			HashMapWithHashCode<Integer, ThreadStateStack> valuesAssignedAutomatonStates = new HashMapWithHashCode<Integer, ThreadStateStack>(source.evaluations2automatons.get(values));
			evaluations2automatons.put(values, valuesAssignedAutomatonStates);
			mappedState.add(valuesAssignedAutomatonStates);
			
			// Update mappings in automatons2states map
			for(ThreadStateStack tss : valuesAssignedAutomatonStates.values()) {
				AutomatonState as = tss.getAutomatonState();

				Set<VariablesEvaluation> asSet = automatons2evaluations.get(as);
				if (asSet == null) {
					asSet = new HashSet<VariablesEvaluation>();
				}
				asSet.add(values);

				automatons2evaluations.put(as, asSet);
			}
		}
		
		if (DEBUG) { System.out.println( StatesMapperLayer.class.getSimpleName() + "." + StatesMapperLayer.class.getSimpleName() + "(source=" + source + ")"); System.out.println("  This after call"); this.DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}
	
	/**
	 * Create mapping for initial variable evaluation. 
	 *
	 * @param statesMapper States mapper this layer belongs to.  
	 * @param comp Component for which map {@link AutomatonState} with {@link VariablesEvaluation}. Cann't be null.
	 */
	private void setInitialContent(LTSAComponentSpecification comp, StatesMapper statesMapper) {
		// initial variables evaluation
		VariablesEvaluation initialValues = new VariablesEvaluation(comp, statesMapper);

		// Create unique anonymous state
		AutomatonState initialAS = new AutomatonState(null, new State(), -1); 
		ThreadStateStack initialTss = new ThreadStateStack(initialAS, null);
		// Virtual never used state ... its used as owner of initial state
		  // This state is never visible outside StateMapper 
		  // This virtual AS maps variable evaluations if no event is processed by component 
		
		HashMapWithHashCode<Integer, ThreadStateStack> tssMap = new HashMapWithHashCode<Integer, ThreadStateStack>(initialValues);

		tssMap.put(Integer.valueOf(-1), initialTss);
		evaluations2automatons.put(initialValues, tssMap);
		mappedState.add(tssMap);
		
		Set<VariablesEvaluation> varSet = new HashSet<VariablesEvaluation>();
		varSet.add(initialValues);
		automatons2evaluations.put(initialAS, varSet);
	}

	/**
	 * Adds NEW automaton state into mappings table and associates them with all current MAPPED variables evaluations.
	 * <br>
	 * This method can be called only if no variables are in PROCESSING states.
	 *   (Current implementation ignores states that are in PROCESSING mode)
	 * <br>
	 * Is useful for provided call.
	 *
	 * @param as Automatons state to map. Cann't be null.
	 */
	public void setMappingsAllStatesAsCall(AutomatonState as) {
		assert as != null;
		
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setMappingsAllStatesAsCall(as= " + as + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		
		//Note: If all HashMapWithHashCode in the evaluations2automatons has been different before call
		//        then after call newly updated HashMapWithHashCode has to be different too. 
		//        There is no way how to create equal HashMapWithHashCode instances 
		mappedState.clear();
		
		// Set forward mapping
		automatons2evaluations.put(as, new HashSet<VariablesEvaluation>(evaluations2automatons.keySet()));
		
		// Set reverse mapping (affect all VariablesEvaluations)
		for(Entry<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>> entry : evaluations2automatons.entrySet()) {
			if (DEBUG_INFO) { System.out.println("DEBUG -   adding mapping for " + entry.getValue()); }

			final HashMapWithHashCode<Integer, ThreadStateStack> mappings = entry.getValue();
			// Update reverse mappings informations thread specific
			ThreadStateStack tss = mappings.remove(Integer.valueOf(as.getThreadNum()));
			ThreadStateStack newTss = new ThreadStateStack(as, tss);
			mappings.put(Integer.valueOf(as.getThreadNum()), newTss);
			mappedState.add(mappings);
			
			// Remove previous forward mapping in automatons2evaluations
			if (tss != null) {
				// TSS can be null if state wasn't mapped to any state in current thread (first call in new thread)
				removeEvaluationFromForwardMapping(tss.getAutomatonState(), entry.getKey());
			}
		}
		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}
	
	/**
	 * For all MAPPED {@link VariablesEvaluation}s removes mapping for given thread and maps 
	 * with {@link AutomatonState} that was associated with variables before call. 
	 * If no automaton state was mapped before last call, mapping for given thread is forgotten.  
	 * After call {@link VariablesEvaluation} remain in MAPPED state. 
	 * <br>
	 * Is useful for provided return. (Return without processing return values).
	 * <br>
	 * This method can be called only if no variables are in PROCESSING states. 
	 *   (Current implementation ignores states that are in PROCESSING mode)
	 * <br>
	 * Note: Each {@link VariablesEvaluation} needs to have mapping set for given threadNum, otherwise error occur
	 * 
	 * @param threadNum Thread that makes return
	 */
	public void setMappingsAllStatesAsReturn(int threadNum) {
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setMappingsAllStatesAsReturn(threadNum= " + threadNum + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		mappedState.clear();

		// Process saved states
		// For each variables evaluation, find TSS-> obtain current AS, and previous TSS(AS) -> remove mapping from current AS and set with previous AS
		for(Entry<VariablesEvaluation, HashMapWithHashCode<Integer, ThreadStateStack>> entry : evaluations2automatons.entrySet()) {
			HashMapWithHashCode<Integer, ThreadStateStack> mappings = entry.getValue();
			ThreadStateStack tss = mappings.remove(Integer.valueOf(threadNum));
			assert tss != null;
			if (tss == null) {
				// tss == null -> No mappings to remove. State is not mapped with any automaton state from given thread.
				// This shouldn't happen ... illegal threadNum, return without call?
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".setMappingsAllStatesAsReturn(threadNum= " + threadNum + ") values=" + entry.getKey() + " - Non existing mapping for give thread");
				continue; 
			}
			
			// update reverse mapping
			ThreadStateStack prevTss = tss.getPrevious();
			
			if (prevTss != null) {
				// Handle return values (move return value to lastCallReturnValue)
				prevTss = prevTss.processReturnEvent(tss); 
				// We have to respect rule that no TSS will point to null value
				mappings.put(Integer.valueOf(threadNum), prevTss);
			}
			// Test whether there is not same (equal) mappings
			if (!mappedState.contains(mappings)) {
				// remove from forward mapping
				removeEvaluationFromForwardMapping(tss.getAutomatonState(), entry.getKey());
				// Add forward mapping for automaton state before call.
				if (prevTss != null) {
					// prevTss == null --> no previous event for this thread, only remove mappings 
					addEvaluationToForwardMapping(prevTss.getAutomatonState(), entry.getKey());
				}
				mappedState.add(mappings);
			} else {
				// The same mappings exists ... completely remove this mappings (no reason to process completely same mappings)
				removeEvaluationFromForwardMappings(entry.getKey());
				evaluations2automatons.remove(entry.getKey());
			}
		}

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}

	/**
	 * Gets all MAPPED {@link VariablesEvaluation}s that are associated with given {@link AutomatonState} and
	 *   moves them into PROCESSING state.
	 *   
	 * @param as {@link AutomatonState}
	 * @return {@link VariablesEvaluation}s mapped to given {@link AutomatonState}
	 */
	public Set<VariablesEvaluation> getMappedStates(AutomatonState as) {
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".getMappedStates(as= " + as + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }

		Set<VariablesEvaluation> result = new HashSet<VariablesEvaluation>(automatons2evaluations.get(as));
		
		if (result != null) {
			// result == Null -> means that there are no VariablesEvaluations associated to AS (can happen if others threads fail moving with all associated variables states) 

			// result != null -> We have some variables states assigned to automaton state -> move theirs mapping into processing table
			for(VariablesEvaluation values : result) {
				changeEvaluationStateMapped2Processing(values);
			}
		}

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		return result;
	}

	/** 
	 * Associate given {@link VariablesEvaluation} to {@link AutomatonState}. 
	 *   Previously associated automaton state from same thread is replaced by given new {@link AutomatonState}.
	 *   {@link VariablesEvaluation} has to be in state PROCESSING and after call new mapping is saved into state MAPPED.
	 *   Old mapping remains in PROCESSING state (to be mapped again to different state).
     * <br>
     * Note: Returns error if no {@link AutomatonState} was previously associated for given {@link VariablesEvaluation} and thread.  
     * 
	 * @param as Specify to which {@link AutomatonState} associate {@link VariablesEvaluation}. Cann't be null.
	 * @param values {@link VariablesEvaluation} to use. Cann't be null.
	 */
	public void setMapping(AutomatonState as, VariablesEvaluation values) {
		assert as != null;
		assert values != null;

		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setMapping(as= " + as + ", values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// find associated informations about state
		VariablesEvaluation valuesCopy = new VariablesEvaluation(values);
		cloneMappings(values, valuesCopy);
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(valuesCopy);
		
		// Update associated states
		ThreadStateStack oldASrepresentant = mappings.get(as.getThreadNum());
		ThreadStateStack newASrepresentant = null;

		if (oldASrepresentant != null) {
			newASrepresentant = oldASrepresentant.changeAutomatonState(as);
		} else {
			// Not previously associated for any state in current thread
			processError(ERROR_MESSAGE_ILLEGAL_RETURN, StatesMapperLayer.class.getSimpleName() + ".setMapping - Given VariablesEvaluation not found previous state in current thread");
			
			newASrepresentant = new ThreadStateStack(as, null);
		}

		mappings.put(Integer.valueOf(as.getThreadNum()), newASrepresentant);
		
		// process changes
		changeEvaluationStateProcessing2Mapped(valuesCopy);
		
		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}

	/** 
	 * Associate given {@link VariablesEvaluation} with given {@link AutomatonState}. 
	 *   Previously associated {@link AutomatonState} from same thread 
	 *   is saved into {@link ThreadStateStack} under given new {@link AutomatonState}. (To be able handle return.)
	 * {@link VariablesEvaluation} has to be in state PROCESSING and after call it's moved into state MAPPED.
	 *
	 * @param as Automaton state to associate with {@link VariablesEvaluation}. Cann't be null.
	 * @param values {@link VariablesEvaluation} to use. Cann't be null.
	 */
	public void setMappingAsCall(AutomatonState as, VariablesEvaluation values) {
		assert as != null;
		assert values != null;
		
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setMappingAsCall(as= " + as + ", values=" + values.toString() + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// find associated informations about state
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);

		// Update ThreadStateStack (and new entry for given AutomatonState)
		ThreadStateStack oldASrepresentant = mappings.get(as.getThreadNum());
		ThreadStateStack newASrepresentant = new ThreadStateStack(as, oldASrepresentant);
		mappings.put(Integer.valueOf(as.getThreadNum()), newASrepresentant);

		// Change evaluation state PROCESSING -> MAPPED
		changeEvaluationStateProcessing2Mapped(values);

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}

	/** 
	 * Modify mapping for given {@link VariablesEvaluation} under given thread. 
	 *   Currently mapped {@link AutomatonState} for given thread is removed and
	 *   {@link VariablesEvaluation} is remapped to state, that it had been mapped before last 
	 *   {@link #setMappingAsCall(AutomatonState, VariablesEvaluation)} or 
	 *   {@link StatesMapperLayer#setMappingsAllStatesAsCall(AutomatonState)} call in given thread.
	 * <br>
	 * If no such mappings exists (no previous call in given thread). 
	 *      Mapping for given thread is removed and error reported.      
	 * {@link VariablesEvaluation} has to be in PROCESSING state and after call it's moved into MAPPED state.
	 *
	 * @param threadNum Specify thread. We change mapping for this thread. Another mapping are untouched.
	 * @param values {@link VariablesEvaluation} to use. Cann't be null.
	 * @return {@link AutomatonState} that was associated with given {@link VariablesEvaluation} 
	 *   (or its source evaluation) before last call. It means {@link AutomatonState} variables are 
	 *   mapped after call.
	 */
	public AutomatonState setMappingAsReturn(int threadNum, VariablesEvaluation values) {
		assert values != null;
		
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setMappingAsReturn(threadNum= " + threadNum + ", values=" + values.toString() + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// find associated informations about state
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);

		// update associated states
		ThreadStateStack oldRepresentant = mappings.get(threadNum);
		ThreadStateStack newRepresentant = null; // Stack entry to be mapped with evaluation after call. (predecessor of oldRepresentant in stack if exists)
		
		assert oldRepresentant != null; // Return without any call (pairing call return can be broken before ... where we catch stack bottom
		if (oldRepresentant != null) {
			ThreadStateStack prevRepresentant = oldRepresentant.getPrevious();
			
			// Handle return values (move return value to lastCallReturnValue)
			newRepresentant = prevRepresentant.processReturnEvent(oldRepresentant);

			mappings.put(Integer.valueOf(threadNum), newRepresentant);
			
			//save state (both mappings)
			changeEvaluationStateProcessing2Mapped(values);
		} else {
			// Nothing to do. No mapping to remove. Leave untouched
			processError(ERROR_MESSAGE_ILLEGAL_RETURN, StatesMapperLayer.class.getSimpleName() + ".setMappingAsReturn - Given VariablesEvaluation not found previous state in current thread");
		}
		

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		
		if (newRepresentant != null) {
			return newRepresentant.getAutomatonState();
		}

		return null;
	}
	
	/**
	 * Move mappings that is in MAPPED state into PROCESSING state.
	 * If given mapping not in MAPPED state prints error.
	 * <br>
	 * Note: Used for processing of return values. 
	 * 
	 * @param values {@link VariablesEvaluation} to change state. Cann't be null.
	 */
	public void moveMappingToProcessing(VariablesEvaluation values) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".moveMappingToProcessing(values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		
		changeEvaluationStateMapped2Processing(values);

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
	}
	
	/**
	 * Automaton states associated with source {@link VariablesEvaluation} are associated also with
	 * target {@link VariablesEvaluation}. 
	 * <br>
	 * Source {@link VariablesEvaluation} has to be in PROCESSING state. 
	 * Target {@link VariablesEvaluation} has to be in NOT_MAPPED state.
	 * After call both states are in PROCESSING state.
	 * <br>
	 * Note: Only for internal purposes {@link VariablesEvaluation#setVariableValue(String, String)} method.
	 * Note:  Used for cloning in {@link #setMapping(AutomatonState, VariablesEvaluation)}
	 * Note: Current implementations removes targets mappings and replaces them with source's one.
	 * Note: Mappings mean call history (not only top layer) - whole {@link ThreadStateStack} for each thread.
	 * 
	 * @param source {@link VariablesEvaluation} which associated {@link AutomatonState}s have to be copied. Cann't be null.
	 * @param target {@link VariablesEvaluation} which should get new mappings. Cann't be null.
	 */
	public void cloneMappings(VariablesEvaluation source, VariablesEvaluation target) {
		assert source != null;
		assert target != null;
		
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".cloneMappings(source=" + source + ", target=" + target + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		if (source == null || target == null) {
			return;
		}
		
		// Make copy of mappings
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = processingState.get(source);	
		HashMapWithHashCode<Integer, ThreadStateStack> newMappings = new HashMapWithHashCode<Integer, ThreadStateStack>(target, mappings);

		// Save mappings
		processingState.put(target, newMappings);
		
		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		
	}

	/**
	 * Sets return value for given {@link VariablesEvaluation} in given thread.
	 * Rewrites previously saved value.
	 * Given {@link VariablesEvaluation} has to be in PROCESSING state.
	 * <br>
	 * Note: State has to have saved mappings for given thread, otherwise error is reported.
	 * Note: Return value can be retrieved by calling {@link #getReturnValue(int, VariablesEvaluation)} before 
	 *   calling {@link #setMappingAsReturn(int, VariablesEvaluation)} or {@link #setMappingsAllStatesAsReturn(int)}
	 * 
	 * @param threadNum Specify thread to set return value
	 * @param values {@link VariablesEvaluation} to set return value
	 * @param retVal Return value to set
	 */
	public void setReturnValue(int threadNum, VariablesEvaluation values, String retVal) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setReturnValue(threadNum=" + threadNum + ", values=" + values + ", retVal=" + retVal + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// Find where state take place
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		
		if (mappings != null) {
			ThreadStateStack tss = mappings.get(threadNum);
	
			if (tss == null) {
				// tss == null -> No mappings to set return value. State is not mapped with any automaton state from given thread.
				// This shouldn't happen
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".setReturnValue(threadNum= " + threadNum + ", values=" + values + ", retVal=" + retVal + ") - Non existing mapping for give thread");
				return;
			}
			// Setting return value
			ThreadStateStack newTss = new ThreadStateStack(tss, retVal); 
			// Saving return value
			mappings.put(Integer.valueOf(threadNum), newTss);
		}

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
			
	}
	
	/**
	 * Gets return value associated with given {@link VariablesEvaluation} in given thread.
	 * Given {@link VariablesEvaluation} has to be in PROCESSING state.
	 * <br>
	 * Note: State has to have saved mappings for given thread, otherwise error is reported.
	 * Note: Return value set by last call {@link #setReturnValue(int, VariablesEvaluation, String)}
	 * 
	 * @param threadNum Specify thread to get return value
	 * @param values {@link VariablesEvaluation} to get return value. Cann't be null.
	 * @return Return value that have been set to pair threadNum and {@link VariablesEvaluation}, or null if havn't been set.
	 */
	public String getReturnValue(int threadNum, VariablesEvaluation values) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".getReturnValue(threadNum=" + threadNum + ", values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// Find where state take place

		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		if (mappings != null) {
			ThreadStateStack tss = mappings.get(threadNum);
			assert tss != null; // Cann't get return value for nonexisting threadNum
			if (tss == null) {
				// tss == null -> No mappings to set return value. State is not mapped with any automaton state from given thread.
				// 	This shouldn't happen
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".getReturnValue(threadNum= " + threadNum + ", values=" + values + ") - Non existing mapping for give thread");
				return null;
			}

			if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

			return tss.getReturnValue();
		}

		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
		throw new RuntimeException("Unknown variables evaluation ... not in processing state " + values);
		// return null;
	}
	
	/**
	 * Gets return value of last call associated with given {@link VariablesEvaluation} in given thread.
	 * Given {@link VariablesEvaluation} has to be in PROCESSING state.
	 * 
	 * <br>Note: State has to have saved mappings for given thread, otherwise error is reported.
	 * <br>Note: Value is copied from {@link #getReturnValue(int, VariablesEvaluation)} during return.
	 * <br>Note: Used to support {@link LastCallRef} edge.
	 * 
	 * @param threadNum Specifies the thread to get the return value
	 * @param values {@link VariablesEvaluation} to get return value. Cann't be null.
	 * @return Return value that have been set to pair threadNum and {@link VariablesEvaluation}, or null if havn't been set.
	 */
	public String getLastCallReturnValue(int threadNum, VariablesEvaluation values) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".getLastCallReturnValue(threadNum=" + threadNum + ", values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// Find where state take place
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		if (mappings != null) {
			ThreadStateStack tss = mappings.get(threadNum);
			assert tss != null; // Cann't get return value for nonexisting threadNum
			if (tss == null) {
				// tss == null -> No mappings to get return value. State is not mapped with any automaton state from given thread.
				// 	This shouldn't happen
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".getLastCallReturnValue(threadNum= " + threadNum + ", values=" + values + ") - Non existing mapping for the given thread");
				return null;
			}
			if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
			return tss.getLastCallReturnValue();
		}

		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		throw new RuntimeException("Unknown variables evaluation ... not in processing state " + values);
		// return null;
	}

	/**
	 * Gets type of the return value of the last call in given thread that takes place over given 
	 *  {@link VariablesEvaluation} (in automaton trace)
	 *  
	 *  <br>Note: See {@link StatesMapperLayer#getLastCallReturnValue(int, VariablesEvaluation)}
	 *  <br>Note: Used to support {@link LastCallRef} edge and {@link TBPResolvedUndefinedEmit}.
	 *  
	 * @param threadNum Thread number where the return takes place.
	 * @param values Variable (evaluation) which is associated with call.
	 * @return Type of last call in automaton trace (with given variable).
	 */
	public ReturnValues getLastCallReturnValueType(int threadNum, VariablesEvaluation values) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".getLastCallReturnValueType(threadNum=" + threadNum + ", values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// Find where state take place

		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		if (mappings != null) {
			ThreadStateStack tss = mappings.get(threadNum);
			assert tss != null; // Cann't get return value for nonexisting threadNum
			if (tss == null) {
				// tss == null -> No mappings to get return value. State is not mapped with any automaton state from given thread.
				// 	This shouldn't happen
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".getLastCallReturnValueType(threadNum= " + threadNum + ", values=" + values + ") - Non existing mapping for the given thread");
				return null;
			}
			if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
			return tss.getLastCallReturnValueType();
		}

		throw new RuntimeException("Unknown variables evaluation ... not in processing state " + values);
		// return null;
	}

	/**
	 * Sets type of the return value of the last call in given thread to {@link ReturnValues#RET_VAL_UNDEFINED_EMIT}. 
	 *  for given {@link VariablesEvaluation} and thread.
	 * Given  {@link VariablesEvaluation} has to be in the PROCESSING state (and stay in the PROCESSING state)
	 *  
	 *  <br>Note: See {@link StatesMapperLayer#getLastCallReturnValueType(int, VariablesEvaluation)}
	 *  <br>Note: Used to support {@link LastCallRef} edge and {@link TBPResolvedUndefinedEmit}.
	 *  
	 * @param threadNum Thread number where the return takes place.
	 * @param values Variable (evaluation) which is associated with call.
	 */
	public void setLastCallReturnValueTypeUndefinedEmit(int threadNum, VariablesEvaluation values) {
		assert values != null;
		if (DEBUG) { System.out.println("DEBUG - " + StatesMapperLayer.class.getSimpleName() + ".setLastCallReturnValueTypeUndefinedEmit(threadNum=" + threadNum + ", values=" + values + ")"); }
		if (DEBUG_INFO) { System.out.println("  Before call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		// Find where state take place
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		if (mappings != null) {
			ThreadStateStack tss = mappings.get(threadNum);
			assert tss != null; // Cann't get return value for nonexisting threadNum
			if (tss == null) {
				// tss == null -> No mappings to get return value. State is not mapped with any automaton state from given thread.
				// This shouldn't happen
				processError(ERROR_MESSAGE_NOT_MAPPED_FOR_GIVEN_THREAD, StatesMapperLayer.class.getSimpleName() + ".setLastCallReturnValueTypeUndefinedEmit(threadNum= " + threadNum + ", values=" + values + ") - Non existing mapping for the given thread");
				return;
			}
			
			ThreadStateStack newTSS = tss.changeLastCallRetValTypeUndefEmit();
			mappings.put(Integer.valueOf(threadNum), newTSS);

			if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
			if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }
			return;
		}

		if (DEBUG_INFO) { System.out.println("  After call"); DEBUG_toString(4); }
		if (INTERNAL_CHECKS) { DEBUG_checkConsistency(); }

		throw new RuntimeException("Unknown variables evaluation ... not in processing state " + values);
		// return null;
	}

	/**
	 * Changes state of given {@link VariablesEvaluation}. 
	 *   Moves evaluation from PROCESSING into MAPPED state. 
	 * <br>
	 * Note: It {@link VariablesEvaluation} isn't in PROCESSING state does nothing.
	 * 
	 * @param values {@link VariablesEvaluation} to move.
	 * 
	 */
	private void changeEvaluationStateProcessing2Mapped(VariablesEvaluation values) {
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = getProcessingVariableEvaluationMapping(values);
		processingState.remove(values);
		
		// Check whether there is not same (equal mappings)
		if (mappedState.contains(mappings)) {
			// Same mappings exists. No reason to add them twice
			return;
		}
		
		mappedState.add(mappings);
		// set reverse mapping
		evaluations2automatons.put(values, mappings);

		// set forward mapping AS->Evaluations
		for(ThreadStateStack tss : mappings.values()) {
			AutomatonState updatedAS = tss.getAutomatonState();
			
			addEvaluationToForwardMapping(updatedAS, values);
		}
	}
	
	/**
	 * Change state of given {@link VariablesEvaluation}. 
	 *   Moves evaluation from MAPPED into PROCESSING state. 
	 * <br>
	 * Note: It {@link VariablesEvaluation} isn't in MAPPED state print error.
	 * 
	 * @param values {@link VariablesEvaluation} to move. Cann't be null.
	 */
	private void changeEvaluationStateMapped2Processing(VariablesEvaluation values) {
		assert values != null;
		
		// Remove from reverse mappings
		HashMapWithHashCode<Integer, ThreadStateStack> mappings = evaluations2automatons.remove(values);
		mappedState.remove(mappings);
		
		//  if mapping == null ... not mapped state
		if (mappings == null) {
			processError(ERROR_MESSAGE_NOT_IN_MAPPED_STATE, StatesMapperLayer.class.getSimpleName() + ".changeEvaluationStateMapped2Processing(values=" + values + ") - Given VariablesEvaluation not found in MAPPED state");
			return;
		}

		// Move to PROCESSING state
		processingState.put(values, mappings);
			
		// Remove from forward mappings
		for(Entry<Integer, ThreadStateStack> entry : mappings.entrySet()) {
			final AutomatonState as = entry.getValue().getAutomatonState();

			removeEvaluationFromForwardMapping(as, values);
		}
	}
	
	/**
	 * Gets mapping associated with given {@link VariablesEvaluation}. 
	 * {@link VariablesEvaluation} has to be in PROCESSING state.
	 * <br>
	 * Note: If not {@link VariablesEvaluation} sets error. 
	 * Note: {@link VariablesEvaluation} remains in PROCESSING state.
	 *
	 * @param values {@link VariablesEvaluation} with mappings.
	 * @return Mappings associated with gives {@link VariablesEvaluation}
	 */
	private HashMapWithHashCode<Integer, ThreadStateStack> getProcessingVariableEvaluationMapping(VariablesEvaluation values) {
		HashMapWithHashCode<Integer, ThreadStateStack> result = processingState.get(values);
		if (result == null) {
			processError(ERROR_MESSAGE_NOT_IN_PROCESSING_STATE, StatesMapperLayer.class.getSimpleName() + ".getProcessingVariableEvaluationMapping(values=" + values + ") - Given VariablesEvaluation not found in PROCESSING state");
		}
		return result;
	}
	
	/**
	 * Gets mapping associated with given {@link VariablesEvaluation}. 
	 * {@link VariablesEvaluation} has to be in MAPPED state.
	 * <br>
	 * Note: If not {@link VariablesEvaluation} sets error. 
	 * Note: {@link VariablesEvaluation} remains in MAPPED state.
	 *
	 * @param values {@link VariablesEvaluation} with mappings.
	 * @return Mappings associated with gives {@link VariablesEvaluation}
	 */
	private HashMapWithHashCode<Integer, ThreadStateStack> getMappedVariableEvaluationMapping(VariablesEvaluation values) {
		HashMapWithHashCode<Integer, ThreadStateStack> result = evaluations2automatons.get(values);
		if (result == null) {
			processError(ERROR_MESSAGE_NOT_IN_MAPPED_STATE, StatesMapperLayer.class.getSimpleName() + ".getMappedVariableEvaluationMapping(values=" + values + ") - Given VariablesEvaluation not found in MAPPED state");
		}
		return result;
	}
	
	/**
	 * Updates forward mappings. Add given {@link VariablesEvaluation} 
	 *   into set of associated to given {@link AutomatonState}
	 *  <br>
	 *  Note: Updates only {@link #automatons2evaluations} map.
	 *  
	 * @param as {@link AutomatonState} which associations to update. Cann't be null.
	 * @param values {@link VariablesEvaluation} to add into mapping. Cann't be null.
	 */
	private void addEvaluationToForwardMapping(AutomatonState as, VariablesEvaluation values) {
		assert as != null;
		assert values != null;
		
		// Add new mappings
		Set<VariablesEvaluation> newMapping = automatons2evaluations.get(as);
		if (newMapping == null) {
			newMapping = new HashSet<VariablesEvaluation>();
			automatons2evaluations.put(as, newMapping);
		}
		newMapping.add(values);
	}

	 /**
	 * Updates forward mapping. Removes given {@link VariablesEvaluation} 
	 *   from set of associated to given {@link AutomatonState}
	 *  <br>
	 *  Note: Updates only {@link #automatons2evaluations} map.
	 *  
	 * @param as {@link AutomatonState} which associations to update.
	 * @param values {@link VariablesEvaluation} to remove from mapping.
	 * 
	 */
	private void removeEvaluationFromForwardMapping(AutomatonState as, VariablesEvaluation values) {
		Set<VariablesEvaluation> asMapping = automatons2evaluations.get(as);
		assert asMapping != null; // If null ... either invalid AS or Interal error
		asMapping.remove(values);
		if (asMapping.isEmpty()) {
			automatons2evaluations.remove(as);
		}
	}
	 /**
	 * Updates forward mappings. Removes given {@link VariablesEvaluation} 
	 *   from set of all {@link AutomatonState}
	 *  <br>
	 *  Note: Updates only {@link #automatons2evaluations} map.
	 *  
	 * @param values {@link VariablesEvaluation} to remove from mapping.
	 * 
	 */
	private void removeEvaluationFromForwardMappings(VariablesEvaluation values) {
		HashMapWithHashCode<Integer, ThreadStateStack> backwardMapping = evaluations2automatons.get(values);
		assert backwardMapping != null;
		for(ThreadStateStack tss : backwardMapping.values()) {
			AutomatonState as = tss.getAutomatonState();
			
			Set<VariablesEvaluation> asMapping = automatons2evaluations.get(as);
			assert asMapping != null; // If null ... either invalid AS or Interal error
			asMapping.remove(backwardMapping.getAssociatedVariablesEvaluation());
			if (asMapping.isEmpty()) {
				automatons2evaluations.remove(as);
			}
		}
	}
	/**
	 * Process error that occur. Sets error messages and flags.
	 * <br>
	 * Prints out errorStringDetails and throw RuntimeException with errorStringDetails.
	 * 
	 * @param errorString Basic error message. One from ERROR_MESSAGE_* constants.
	 * @param errorStringDetails Detailed error message description.
	 */
	private void processError(String errorString, String errorStringDetails) {
		if (!errorFlag) {
			this.errorFlag = true;
			this.errorString = errorString;
			this.errorStringDetails = errorStringDetails;
		}
		
		System.err.println("INTERNAL ERROR - " + errorStringDetails);
		throw new RuntimeException(errorStringDetails);
	}
	
	
	public int getHashCode() {
		return hashCode();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		StatesMapperLayer other = (StatesMapperLayer) obj;

		if (getHashCode() != other.getHashCode()) {
			return false;
		}
		
		if (mappedState.size() != other.mappedState.size()) {
			return false;
		}

		for(HashMapWithHashCode<Integer, ThreadStateStack> val : mappedState) {
			if (!other.mappedState.contains(val)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Get "memento" where current mappings in the MAPPED state are caught.
	 * Note: Memento represents current position in the TBP Automaton.
	 * Note: Is used by JPFSearch to extend JPF state by positions in the TBP.
	 * 
	 * @return Get "memento" where current mappings in the MAPPED state are caught.
	 */
	public MappedStateHashableMemento getTBPPosition() {
		return new MappedStateHashableMemento(mappedState);
	}
	
	public static class MappedStateHashableMemento {
		private final Set<HashMapWithHashCode<Integer, ThreadStateStack>> mappedState; /// All mappings in the MAPPED state. Note: Special handling is necessary. See HashMapWithHashCode
		private final int hashCode;
		
		private final static int HASH_CODE_PRIME = 43;

		public static final int INVALID_MEMENTO_INDEX = -1;
		private int mementoIndex = INVALID_MEMENTO_INDEX; /// Unique index of this memento. This is shadow variable. Not included into hash and equalOperator(). {@link JPFProgramStateMapping}  maintains this field. 

		public MappedStateHashableMemento(Set<HashMapWithHashCode<Integer, ThreadStateStack>> source) {
			assert source != null;
			SortedSet<HashMapWithHashCode<Integer, ThreadStateStack>> sortedSource = new TreeSet<HashMapWithHashCode<Integer,ThreadStateStack>>(source);
			mappedState = new HashSet<HashMapWithHashCode<Integer,ThreadStateStack>>((int)(source.size()*CONST_HASHMAP_GROWING));
			
			int tmpHashCode = 0;
			for(HashMapWithHashCode<Integer, ThreadStateStack> mappings : sortedSource) {
				HashMapWithHashCode<Integer, ThreadStateStack> mappingsCopy = new HashMapWithHashCode<Integer, ThreadStateStack>(mappings);
				mappedState.add(mappingsCopy);
				tmpHashCode = (tmpHashCode * HASH_CODE_PRIME) + mappingsCopy.getHashCode();
			}
			hashCode = tmpHashCode;
		}
		
		@Override
		public int hashCode() {
			return hashCode;
		}
		
		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof MappedStateHashableMemento))
				return false;
			MappedStateHashableMemento other = (MappedStateHashableMemento) obj;
			
			if (hashCode() != other.hashCode()) {
				return false;
			}
			
			if (mappedState.size() != other.mappedState.size()) {
				return false;
			}
			
			for(HashMapWithHashCode<Integer, ThreadStateStack> mappings : mappedState) {
				if (!other.mappedState.contains(mappings)) {
					return false;
				}
			}
			return true;
		}

		@Override
		public String toString() {
			return toString(2);
		}
		
		public String toString(int indent) {
			StringBuffer sb = new StringBuffer();
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("Debugging " + MappedStateHashableMemento.class.getSimpleName() + "\n");
			indent+=2;
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("hashCode="); sb.append(hashCode); sb.append('\n');
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("mementoIndex="); sb.append(mementoIndex); sb.append('\n');
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("MappedStates=Set:\n");
			DEBUG_printMappedStates(sb, indent+2);
			return sb.toString();
		}
		
		public void DEBUG_printMappedStates(StringBuffer sb, int indent) {
			for(HashMapWithHashCode<Integer, ThreadStateStack> mappings : mappedState) {
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
				sb.append("ValEval ID=" + mappings.getAssociatedVariablesEvaluation().DEBUG_get_variableEvaluationID() + " " + mappings.getAssociatedVariablesEvaluation());
				sb.append('\n');
				
				for(Entry<Integer, ThreadStateStack> entry : mappings.entrySet()) {
					// threadNum, tss
					ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
					sb.append(entry.getKey());
					sb.append("-->");
					sb.append(entry.getValue());
					sb.append('\n');
				}
			}
		}

		/**
		 * @return Gets index of this instance. Default value is ({@link #INVALID_MEMENTO_INDEX})
		 */
		public int getMementoIndex() {
			return mementoIndex;
		}
		
		/**
		 * Set index of this memento instance. This method can be called only once.
		 * @param newIndex Set index of this memento instance.
		 */
		public void setMementoIndex(int newIndex) {
			if (mementoIndex != INVALID_MEMENTO_INDEX) {
				throw new RuntimeException("Multiple setMementoIndex calls");
			}
			mementoIndex = newIndex;
		}
	}
	// ************************************************************************
	// * DEBUG Facility
	// ************************************************************************
	
	/** 
	 * Prints into std. out information about all managed mappings.
	 */
	public void DEBUG_toString(int indent) {
		StringBuffer sb = new StringBuffer();
		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append(StatesMapper.class.getSimpleName());
		sb.append(".. Debugging all mappings");
		sb.append(", hash=" + hashCode());
		sb.append("\n");

		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("MappedStates\n");
		sb.append(DEBUG_printMappedStates(indent+2));

		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("Automatons-->Variables\n");
		sb.append(DEBUG_printAutomatons2evaluationd(indent+2));

		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("Variables-->Automatons\n");
		sb.append(DEBUG_printEvaluations2automatons(indent+2));

		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("ProcessingStates\n");
		sb.append(DEBUG_printProcessingStates(indent+2));
		
		System.out.println(sb);
	}
	
	/**
	 * Transforms forward mapping from {@link AutomatonState} 
	 *  to {@link VariablesEvaluation} into human readable form. 
	 * 
	 * @param indent Indentation on line start
	 * @return Dump of {@link #automatons2evaluations} table. 
	 */
	public StringBuffer DEBUG_printAutomatons2evaluationd(int indent) {
		StringBuffer sb = new StringBuffer();
		for(AutomatonState as : automatons2evaluations.keySet()) {
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append(as);
			sb.append('\n');
			
			Set<VariablesEvaluation> vars = automatons2evaluations.get(as);
			if (vars == null) {
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
				sb.append("[null]\n");
			} else {
				for(VariablesEvaluation var : vars) {
					ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
					sb.append(var.toString());
					sb.append('\n');
				}
			}
		}
		return sb;
	}
	
	/**
	 * Transforms reverse mapping from {@link VariablesEvaluation} 
	 *   to {@link AutomatonState} into human readable form. 
	 * 
	 * @param indent Indentation level (Space count on start of each new line) 
	 * @return Dump of {@link #evaluations2automatons} table.
	 */
	public StringBuffer DEBUG_printEvaluations2automatons(int indent) {
		StringBuffer sb = new StringBuffer();
		for(VariablesEvaluation var : evaluations2automatons.keySet()) {
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append(var.toString());
			sb.append('\n');
			
			HashMapWithHashCode<Integer, ThreadStateStack> automatonStatesMap = evaluations2automatons.get(var);
			if (automatonStatesMap == null) {
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
				sb.append("[null]\n");
			} else {
				for(Entry<Integer, ThreadStateStack> entry : automatonStatesMap.entrySet()) {
					// threadNum, tss
					ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
					sb.append(entry.getKey());
					sb.append("-->");
					sb.append(entry.getValue());
					sb.append('\n');
				}
			}
		}
		return sb;
	}

	/**
	 * Transforms mappings for MAPPED states from {@link VariablesEvaluation} 
	 *   to {@link AutomatonState} into human readable form. 
	 * Dumps {@link #mappedState} set.
	 * 
	 * @param indent Indentation level (Space count on start of each new line) 
	 * @return Dump of {@link #evaluations2automatons} table.
	 */

	public StringBuffer DEBUG_printMappedStates(int indent) {
		StringBuffer sb = new StringBuffer();
		for(HashMapWithHashCode<Integer, ThreadStateStack> mappings : mappedState) {
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("ValEval ID=" + mappings.getAssociatedVariablesEvaluation().DEBUG_get_variableEvaluationID() + " " + mappings.getAssociatedVariablesEvaluation());
			sb.append('\n');
			
			for(Entry<Integer, ThreadStateStack> entry : mappings.entrySet()) {
				// threadNum, tss
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
				sb.append(entry.getKey());
				sb.append("-->");
				sb.append(entry.getValue());
				sb.append('\n');
			}
		}
		return sb;
	}
	
	/**
	 * Transforms mapping for PROCESSING states from {@link VariablesEvaluation} 
	 *   to {@link AutomatonState} into human readable form. 
	 * 
	 * @param indent Indentation level (Space count on start of each new line) 
	 * @return Dump of {@link #evaluations2automatons} table.
	 */

	public StringBuffer DEBUG_printProcessingStates(int indent) {
		StringBuffer sb = new StringBuffer();
		for(VariablesEvaluation values : processingState.keySet()) {
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append("ValEval ID=" + values.DEBUG_get_variableEvaluationID()  + " " + values);
			sb.append('\n');
			
			HashMapWithHashCode<Integer, ThreadStateStack> automatonStatesMap = processingState.get(values);
			if (automatonStatesMap == null) {
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
				sb.append("[null]\n");
			} else {
				for(Entry<Integer, ThreadStateStack> entry : automatonStatesMap.entrySet()) {
					// threadNum, tss
					ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
					sb.append(entry.getKey());
					sb.append("-->");
					sb.append(entry.getValue());
					sb.append('\n');
				}
			}
		}
		return sb;
	}

	/**
	 * Get {@link VariablesEvaluation}s in MAPPED state associated with given {@link AutomatonState}.
	 * It's similar to {@link #getMappedStates(AutomatonState)}, but it does'n modify
	 * called instance so don't remove returned mappings.
	 * <br>
	 * Note: Only for debugging instances
	 * Note: Returned Set in read only.
	 * 
	 * @param as AutomatonState you interested in.
	 * @return {@link VariablesEvaluation}s associated with given automaton state.
	 */
	
	public Set<VariablesEvaluation> DEBUG_getAssociatedMappedVariables(AutomatonState as) {
		Set<VariablesEvaluation> result = automatons2evaluations.get(as);
		if (result != null) {
			return Collections.unmodifiableSet(result);
		}
		return null;
	}
	
	/** 
	 * Check if internal structures are consistent.
	 * 
	 * @return true if no error found otherwise returns false
	 */
	public boolean DEBUG_checkConsistency() {
		String checkResult = DEBUG_checkConsistency1();
		
		if (checkResult != DEBUG_OK) {
			// Breakpoint place
			System.err.println("CHECK consistency fail !!!!");
			throw new RuntimeException("CHECK consistency fail !!!! \n"  + checkResult);
		}
		return checkResult == DEBUG_OK;
	}
	
	public static final String DEBUG_OK = "OK"; /// Internal result if no error found

	/** 
	 * Check if internal structures are consistent.
	 * 
	 * @return String {@link #DEBUG_OK} if no error in structures found, 
	 *   else error message precisely describing error.
	 */
	public String DEBUG_checkConsistency1() {
		String result = DEBUG_checkConsistencty_mappedStates();
		if (result != DEBUG_OK) {
			return result;
		}
		
		result = DEBUG_checkConsistencty_automatons2evaluations();
		if (result != DEBUG_OK) {
			return result;
		}
		
		result = DEBUG_checkConsistencty_evaluations2automatons();
		if (result != DEBUG_OK) {
			return result;
		}
		
		result = DEBUG_checkConsistencty_processingStates();
		if (result != DEBUG_OK) {
			return result;
		}

		return result;
	}
	
	
	/**
	 * Checks that mapped states stored in the {@link #mappedState} property are stored in the "cache" {@link #evaluations2automatons} too.
	 * 
	 * @return String {@link #DEBUG_OK} if no error in structures found, else error message precisely describing error.
	 */
	private String DEBUG_checkConsistencty_mappedStates() {
		// Note: return string different than DEBUG_OK means error
		if (mappedState == null) {
			return "property mappedState is null";
		}
	
		for(HashMapWithHashCode<Integer, ThreadStateStack> mappings : mappedState) {
			if (mappings == null) {
				return "property mappings contains null entry";
			}
			
			VariablesEvaluation var = mappings.getAssociatedVariablesEvaluation();
			if (var == null) {
				return "property mappings, entry=" + mappings + " contains null as associted VariablesEvaluation";
			}
			
			HashMapWithHashCode<Integer, ThreadStateStack> other = evaluations2automatons.get(var);
			if (other == null) {
				return "property mappings, entry=" + mappings + " associated varEval=" + var + " has no asociated mappings in the evaluations2automatons";
			}
			if (!mappings.equals(other)) {
				return "property mappings, entry=" + mappings + " associated varEval=" + var + ". Evaluations2automatons has associated different mappings=" + other;
			}
			
			//TODO remove
			mappings.DEBUG_checkConsistency();
		}
		return DEBUG_OK;
	}
	/**
	 * Checks if forward {@link #automatons2evaluations} mapping is set correctly 
	 *   and reversed mapping are consistent with {@link #automatons2evaluations}
	 * 
	 * @return String {@link #DEBUG_OK} if no error in structures found, else error message precisely describing error.
	 */
	private String DEBUG_checkConsistencty_automatons2evaluations() {
		// Note: return string different than DEBUG_OK means error
		if (automatons2evaluations == null) {
			return "property automatons2evaluations is null";
		}


		for(AutomatonState as : automatons2evaluations.keySet()) {
			if (as == null) {
				return "property automatons2evaluations contains null entry";
			}

			Set<VariablesEvaluation> vars = automatons2evaluations.get(as);
			if (vars == null) {
				return "property automatons2evaluations, entry=" + as + " has null value as associated states"; 
			}

			for(VariablesEvaluation var : vars) {
				if (var == null) {
					return "property automatons2evaluations, entry=" + as + " has null as one from associated variable evaluation";
				}

				if (evaluations2automatons.containsKey(var) == false) {
					return "property automatons2evaluations, entry=" + as + ", var=" + var + " " + var.toString() + " has not any counterpart variable mappings";
				}
				
				if (processingState.containsKey(var) == true) {
					return "property automatons2evaluations, entry=" + as + ", var=" + var + " " + var.toString() + " is in PROCESSING state";
				}
				
				HashMapWithHashCode<Integer, ThreadStateStack> asMap = evaluations2automatons.get(var);
				if (asMap == null) {
					return "property automatons2evaluations, entry=" + as + ", var=" + var + " " + var.toString() + " has null value set in reverse states2automaton map";
				}

				ThreadStateStack tss = asMap.get(as.getThreadNum());
				if (tss == null) {
					return "property automatons2evaluations, entry=" + as + ", var=" + var + " " + var.toString() + " has not thread specific counterpart in reverse evaluations2automaton map";
				}
				
				if (!tss.getAutomatonState().equals(as)) {
					return "property automatons2evaluations, entry=" + as + ", var=" + var + " " + var.toString() + " has set invalid counterpart state (tss.as=" + tss.getAutomatonState() + ")in reverse states2automaton map";
				}
			}
		} // end for each state
		return DEBUG_OK;
	}

	/**
	 * Checks if reverse {@link #evaluations2automatons} mapping is set correctly 
	 *   and forward mapping {@link #automatons2evaluations} are consistent 
	 *   with {@link #evaluations2automatons}
	 * 
	 * @return String {@link #DEBUG_OK} if no error in structures found, else error message precisely describing error.
	 */
	private String DEBUG_checkConsistencty_evaluations2automatons() {
		if (evaluations2automatons == null) {
			return "property evaluations2automatons is null";
		}

		for(VariablesEvaluation var : evaluations2automatons.keySet()) {
			// Note: return string different than DEBUG_OK means error
			if (var == null) {
				return "property evaluations2automatons contains null entry";
			}
			
			// Check if not in processing state
			if (processingState.containsKey(var) == true) {
				return "property evaluations2automatons, contains var=" + var + ", but processingStates contains this var too. Evaluation is in both MAPPED and PROCESSING states";
			}

			HashMapWithHashCode<Integer, ThreadStateStack> varAsocMap = evaluations2automatons.get(var);
			if (varAsocMap == null) {
				return "property evaluations2automatons, var=" + var + " has null as mapped value";
			}
			if (varAsocMap.getAssociatedVariablesEvaluation() != var) {
				return "property evaluations2automatons, var=" + var + " mapped HashMap is internaly associted with different evaluation (" + varAsocMap.getAssociatedVariablesEvaluation() + ").";
			}
			
			if (!mappedState.contains(varAsocMap)) {
				return "property evaluations2automatons, var=" + var + " no counterpart stored in the mappedState set";
			}
			
			for( Entry<Integer, ThreadStateStack> entry : varAsocMap.entrySet()) {
				final Integer threadNum = entry.getKey();
				if (threadNum == null) {
					return "property evaluations2automatons, var=" + var + " has null as one mapped thread number";
				}

				final ThreadStateStack tss = entry.getValue();
				if (tss == null) {
					return "property evaluations2automatons, var=" + var + " has under thread=" + threadNum + " associated null as thread state stack value";
				}
				
				AutomatonState as = tss.getAutomatonState();
				if (as == null) {
					return "property evaluations2automatons, var=" + var + " has under thread=" + threadNum + " associated tss=" + tss + " with null automaton state";
				}

				// Check forward mapping ... if variables are associated under this AS
				if (automatons2evaluations.containsKey(as)==false) {
					return "property evaluations2automatons, var=" + var + " has under thread=" + threadNum + " associated tss=" + tss + " with as=" + as + " that isn't mapped in forward automatons2evaluations mapping";
				}
				Set<VariablesEvaluation> asVars = automatons2evaluations.get(as);
				if (asVars == null) {
					return "property evaluations2automatons, var=" + var + " has under thread=" + threadNum + " associated tss=" + tss + " with as=" + as + " that has in forward automatons2evaluations mapping null as set with variables";
				}
				
				if (asVars.contains(var) == false) {
					return "property evaluations2automatons, var=" + var + " has under thread=" + threadNum + " associated tss=" + tss + " with as=" + as + " that hasn't in forward automatons2evaluations mapping variables this variables mapped";
				} 
			} // end for each thread
		} // end for each variables evaluation

		return DEBUG_OK;
	}
	
	/**
	 * Checks if {@link #processingState} mapping is set correctly.
	 *   Checks {@link VariablesEvaluation}s in PROCESSING state. 
	 * 
	 * @return String {@link #DEBUG_OK} if no error in structures found, else error message precisely describing error.
	 */
	private String DEBUG_checkConsistencty_processingStates() {
		if (processingState == null) {
			return "property processingState is null";
		}

		for(VariablesEvaluation var : processingState.keySet()) {
			// Note: return string different than DEBUG_OK means error
			if (var == null) {
				return "property processingState contains null entry";
			}
			
			// Check if not in processing state
			if (evaluations2automatons.containsKey(var) == true) {
				return "property processingState, contains var=" + var + ", but evaluations2automatons contains this var too. Evaluation is in both MAPPED and PROCESSING states";
			}

			HashMapWithHashCode<Integer, ThreadStateStack> varAsocMap = processingState.get(var);
			if (varAsocMap == null) {
				return "property processingState, var=" + var + " has null as mapped value";
			}
			
			if (varAsocMap.getAssociatedVariablesEvaluation() != var) {
				return "property processingState, var=" + var + " mapped HashMap is internaly associted with different evaluation (" + varAsocMap.getAssociatedVariablesEvaluation() + ").";
			}
			//TODO remove
			varAsocMap.DEBUG_checkConsistency();

			for(Entry<Integer, ThreadStateStack> entry : varAsocMap.entrySet()) {
				final Integer threadNum = entry.getKey();
				if (threadNum == null) {
					return "property processingState, var=" + var + " has null as one mapped thread number";
				}

				final ThreadStateStack tss = entry.getValue();
				if (tss == null) {
					return "property processingState, var=" + var + " has under thread=" + threadNum + " associated null as thread state stack value";
				}
				
				AutomatonState as = tss.getAutomatonState();
				if (as == null) {
					return "property processingState, var=" + var + " " + var.toString() + " has under thread=" + threadNum + " associated tss=" + tss + " with null automaton state";
				}

				// Check non existence of forward mapping ... if variables aren't associated under this AS
				if (automatons2evaluations.containsKey(as)==true) {
					// Potentially can be mapped
					Set<VariablesEvaluation> asVars = automatons2evaluations.get(as);
					if (asVars == null) {
						return "property processingState, var=" + var + " " + var.toString() + " has under thread=" + threadNum + " associated tss=" + tss + " with as=" + as + " that has in forward automatons2evaluations mapping null as set with variables";
					}
					
					if (asVars.contains(var) == true) {
						return "property processingState, var=" + var + " " + var.toString() + " has under thread=" + threadNum + " associated tss=" + tss + " with as=" + as + " that has mapping in forward automatons2evaluations variables this variables mapped";
					} 
				}
			} // end for each thread
		} // end for each variables evaluation

		return DEBUG_OK;
	}

}
