package org.ow2.dsrg.fm.tbpjava.checker;
//TODO if new thread come validate Maps with threadNum->ThreadStateStack

import java.util.Set;
import java.util.Stack;

import org.ow2.dsrg.fm.tbpjava.checker.StatesMapperLayer.MappedStateHashableMemento;
import org.ow2.dsrg.fm.tbpjava.checker.ThreadStateStack.ReturnValues;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.LastCallRef;

/**
 * {@link StatesMapper} initial settings
 * <br>If created {@link StatesMapper} has only one internal layer, 
 *     where there is one variable evaluation in mapped state (with initial variables values)
 *     that is mapped in thread 0 into one virtual unique state. So first method to call probably 
 *     will be any from setMapping function {@link #setMappingsAllStatesAsCall(AutomatonState)}  
 *
 *   <br>
 *   {@link StatesMapper} consists only from stack of {@link StatesMapperLayer} 
 *    where described logics is implemented. {@link StatesMapper}'s stack simulates
 *    stack in {@link EventParser} and has same reason, to support JPF backtracking.
 */
//TODO It's good idea move to StatesMapper's and EventParser's stack into one separate stack, to remove stack duplicate
public final class StatesMapper {
	
	private Stack<StatesMapperLayer> stateLayerStack; /// Stack of states2variables mapper. To support JPF backtracking.
		
	/**
	 * Create new {@link StatesMapper} and set it into initial state.
	 *  It mean that maps one anonymous automaton state 
	 *    with {@link VariablesEvaluation} with initial values in mapped state.
	 *
	 * @param comp Component for which map {@link AutomatonState} with {@link VariablesEvaluation}. Cann't be null.
	 */
	public StatesMapper(LTSAComponentSpecification comp) {
		assert comp != null;

		stateLayerStack = new Stack<StatesMapperLayer>();
		StatesMapperLayer initialLayer = new StatesMapperLayer(comp, this);
		stateLayerStack.add(initialLayer);
	}
	
	/**
	 * Adds new layer into stack with mappings.
	 * Makes snapshot of current mappings.
	 *  
	 */
	public void newLayer() {
		StatesMapperLayer newTopLayerStackCopy = new StatesMapperLayer(stateLayerStack.peek());
		stateLayerStack.push(newTopLayerStackCopy);
	}
	
	/**
	 * Removes top layer form stack with mappings.
	 * Removes current mappings and uses mappings from last {@link #newLayer()} call (before current layer creation)
	 * Revert mapping to last snapshot create before current layer.  
	 * 
	 * @return true if removing layer was successful, false if no layer to remove.
	 */
	public boolean removeStackLayer() {
		if (stateLayerStack.isEmpty() == true) {
			return false; // No entry to remove
		}
		stateLayerStack.pop();
		return true;
	}

	/** 
	 * Is intended for storing return value of automaton during processing of give variables.
	 * 
	 * @param threadNum Thread which set return value
	 * @param values Variables (array / evaluation) to which associate return value
	 * @param retVal Return value to set
	 */
	public void setReturnValue(int threadNum, VariablesEvaluation values, String retVal) {
		stateLayerStack.peek().setReturnValue(threadNum, values, retVal);
	}

	/** 
	 * Is intended for obtaining return value of automaton associated 
	 *   with given variables evaluation and thread number.
	 * 
	 * @param threadNum Thread number to which is associated return value
	 * @param values Variables (array / evaluation) to which is associated return value
	 * @return Associated return value.
	 */
	public String getReturnValue(int threadNum, VariablesEvaluation values) {
		return stateLayerStack.peek().getReturnValue(threadNum, values);
	}
	
	/**
	 * Return value of last call in given thread that happen with given 
	 *  {@link VariablesEvaluation} (in automaton trace)
	 *  
	 *  <br>Note: See {@link StatesMapperLayer#getLastCallReturnValue(int, VariablesEvaluation)}
	 *  <br>Note: Used to support {@link LastCallRef} edge.
	 *  
	 * @param threadNum Thread number where the return takes place.
	 * @param values Variable (evaluation) which was associated with call.
	 * @return Value of last call in automaton trace (with given variable) or null, if last called method was procedure. 
	 */
	public String getLastCallReturnValue(int threadNum, VariablesEvaluation values) {
		return stateLayerStack.peek().getLastCallReturnValue(threadNum, values); 
	}
	
	/**
	 * Gets type of the return value of the last call in given thread that takes place over given 
	 *  {@link VariablesEvaluation} (in automaton trace)
	 *  
	 *  <br>Note: See {@link StatesMapperLayer#getLastCallReturnValueType(int, VariablesEvaluation)}
	 *  <br>Note: Used to support {@link LastCallRef} edge and {@link TBPResolvedUndefinedEmit}.
	 *  
	 * @param threadNum Thread number where the return takes place.
	 * @param values Variable (evaluation) which is associated with call.
	 * @return Type of last call in automaton trace (with given variable).
	 */
	public ReturnValues getLastCallReturnValueType(int threadNum, VariablesEvaluation values) {
		return stateLayerStack.peek().getLastCallReturnValueType(threadNum, values); 
	}

	/**
	 * Sets type of the return value of the last call in given thread to {@link ReturnValues#RET_VAL_UNDEFINED_EMIT}. 
	 *  for given {@link VariablesEvaluation} and thread.
	 * Given  {@link VariablesEvaluation} has to be in the PROCESSING state (and stay in the PROCESSING state)
	 *  
	 *  <br>Note: See {@link StatesMapperLayer#getLastCallReturnValueType(int, VariablesEvaluation)}
	 *  <br>Note: Used to support {@link LastCallRef} edge and {@link TBPResolvedUndefinedEmit}.
	 *  
	 * @param threadNum Thread number where the return takes place.
	 * @param values Variable (evaluation) which is associated with call.
	 */
	public void setLastCallReturnValueTypeUndefinedEmit(int threadNum, VariablesEvaluation values) {
		stateLayerStack.peek().setLastCallReturnValueTypeUndefinedEmit(threadNum, values); 
	}

	/**
	 * Given automaton state maps (as call) with all current variables (evaluations/states).
	 * This method is intended for processing call where there is known target automaton - 
	 *  it means all providedCalls and some requiredCall (Emit).
	 * 
	 * @param as Automaton state to map.
	 */
	public void setMappingsAllStatesAsCall(AutomatonState as) {
		stateLayerStack.peek().setMappingsAllStatesAsCall(as);
	}
	
	/**
	 * All variables associated with any automaton state from given thread are remapped 
	 *   to {@link AutomatonState} they was associated before last call 
	 *   ({@link #setMappingsAllStatesAsCall(AutomatonState)}) on given thread.
	 *    
	 * @param threadNum Thread number to process
	 */
	public void setMappingsAllStatesAsReturn(int threadNum) {
		stateLayerStack.peek().setMappingsAllStatesAsReturn(threadNum);
	}
	
	 
	/**
	 * Associate given {@link VariablesEvaluation} with given {@link AutomatonState}. 
	 *   Previously associated {@link AutomatonState} from same thread is saved into {@link ThreadStateStack}.
	 * <br>
	 * For detail see {@link StatesMapperLayer#setMappingAsCall(AutomatonState, VariablesEvaluation)}
	 */
	public void setMappingsAsCall(AutomatonState as, VariablesEvaluation values) {
		stateLayerStack.peek().setMappingAsCall(as, values);
	}
	
	/**
	 * Given variables state remap to automaton state that was 
	 *  associated with (source) variables before last call. 
	 *  Modify mapping only for given thread. 
	 *    
	 * @param threadNum Thread number to process
	 * @param values Variables (evaluation/state) to process
	 * @return AutomatonState that was associated with variables state before last call from given tread.
	 */
	public AutomatonState setMappingsAsReturn(int threadNum, VariablesEvaluation values) {
		return stateLayerStack.peek().setMappingAsReturn(threadNum, values);
	}
	
	/**
	 * Gets all MAPPED {@link VariablesEvaluation}s that are associated with given {@link AutomatonState} and
	 *   moves them into PROCESSING state.
	 * <br>
	 * For details see {@link StatesMapperLayer#getMappedStates(AutomatonState)}
	 */
	public Set<VariablesEvaluation> getMappings(AutomatonState as) {
		assert as != null;
		
		return stateLayerStack.peek().getMappedStates(as);
	}
	
	/** 
	 * Associate given {@link VariablesEvaluation} to {@link AutomatonState}. 
	 *   Previously associated automaton state from same thread is replaced by given new {@link AutomatonState}.
	 *  <br>
	 *  For details see {@link StatesMapperLayer#setMapping(AutomatonState, VariablesEvaluation)}
	 *  
	 * @param as Specify to which {@link AutomatonState} associate {@link VariablesEvaluation}. Cann't be null.
	 * @param values {@link VariablesEvaluation} to use. Cann't be null.
	 */ 
	public void setMappings(AutomatonState as, VariablesEvaluation values) {
		stateLayerStack.peek().setMapping(as, values);
	}

	/**
	 * Move mappings that is in MAPPED state into PROCESSING state.
	 * <br>
	 * For details see {@link StatesMapperLayer#moveMappingToProcessing(VariablesEvaluation)} 
	 * 
	 * @param values {@link VariablesEvaluation} to change state. Cann't be null.
	 */
	public void moveMappingToProcessing(VariablesEvaluation values) {
		stateLayerStack.peek().moveMappingToProcessing(values);
	}
	
	/**
	 * Automaton states associated with source {@link VariablesEvaluation} are associated also with
	 * target {@link VariablesEvaluation}. 
	 * <br>
	 * Source {@link VariablesEvaluation} has to be in PROCESSING state. 
	 * Target {@link VariablesEvaluation} has to be in NOT_MAPPED state.
	 * After call both states are in PROCESSING state.
	 * <br>
	 * Note: Only for internal purposes {@link VariablesEvaluation#setVariableValue(String, String)}
	 * Note: Current implementations removes targets mappings and replaces them with source's one.
	 * Note: Mappings mean call history (not only top layer) - whole {@link ThreadStateStack} for each thread.
	 * 
	 * @param source {@link VariablesEvaluation} which associated {@link AutomatonState}s have to be copied. Cann't be null.
	 * @param target {@link VariablesEvaluation} which should get new mappings. Cann't be null.
	 */
	public void cloneMappings(VariablesEvaluation source, VariablesEvaluation target) {
		stateLayerStack.peek().cloneMappings(source, target);
	}

	/**
	 * Get "memento" where current mappings in the MAPPED state are caught.
	 * Note: Memento represents current position in the TBP Automaton.
	 * Note: Is used by JPFSearch to extend JPF state by positions in the TBP.
	 * 
	 * @return Get "memento" where current mappings in the MAPPED state are caught.
	 */
	public MappedStateHashableMemento getTBPPosition() {
		return stateLayerStack.peek().getTBPPosition();
	}

	/**
	 * Get variable evaluations associated with given {@link AutomatonState}.
	 * <br>
	 * Note: Only for debug purposes.
	 * 
	 * For details see {@link StatesMapperLayer#DEBUG_getAssociatedMappedVariables(AutomatonState)}
	 */
	public Set<VariablesEvaluation> DEBUG_getAssociatedMappings(AutomatonState as) {
		assert as != null;
		
		return stateLayerStack.peek().DEBUG_getAssociatedMappedVariables(as);
	}

}
