package org.ow2.dsrg.fm.tbpjava.checker;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import org.ow2.dsrg.fm.tbpjava.checker.StatesMapperLayer.MappedStateHashableMemento;

/**
 * Filters {@link StatesMapperLayer.MappedStateHashableMemento} instances.
 *   It stores only the one instance of all equals {@link StatesMapperLayer.MappedStateHashableMemento} (to conserve the memory)
 *   and manages {@link StatesMapperLayer.MappedStateHashableMemento} indexes.
 *
 */
final public class StateMementoToIndexTransformer {

	private static final boolean DEBUG = false;
	private static final boolean DEBUG_INFO = false;

	private static final boolean DEBUG_COLLISIONS = false; // Print examples of found (MappedStateHashableMemento.hashCode() collisions into final checking results.
	private static final boolean INTERNAL_CHECKS = false; // Enables/disables internal consistency checking

	private final PrintStream out = System.out;

	private final Map<StatesMapperLayer.MappedStateHashableMemento, StatesMapperLayer.MappedStateHashableMemento> stateMementos = new HashMap<StatesMapperLayer.MappedStateHashableMemento, StatesMapperLayer.MappedStateHashableMemento>(); /// Storage of unique StateMemento instances. One to one mapping.
	
	private int lastAssignedMementoIndex = StatesMapperLayer.MappedStateHashableMemento.INVALID_MEMENTO_INDEX;  /// State memento index
	private int cntGetMementoIndex = 0; // Counts calls of the getMementoIndex.
	private int cntEqualMementoInstances = 0; // Number of seen equal instances of the state memento. 
	
	public StateMementoToIndexTransformer() {};
	
	/**
	 * For given instance of memento, get index of equal state memento.
	 *  
	 * @param memento State memento to transform into the index. Cannot be null.
	 * @return Index of the memento (or of the equal memento).
	 */
	public int getMementoIndex(StatesMapperLayer.MappedStateHashableMemento memento) {
		assert memento != null;
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".getMementoIndex()\t" + toString());} 
		if (DEBUG_INFO) { out.println("Dumping content before the call\n"); DEBUG_printMapping(); }
		if (INTERNAL_CHECKS) { DEBUG_checkStructure(); }

		cntGetMementoIndex++;
		
		StatesMapperLayer.MappedStateHashableMemento equalMemento = stateMementos.get(memento);
		if (equalMemento != null) {
			cntEqualMementoInstances++; // Multiple representation of the same state
			assert equalMemento.equals(memento);
		} else {
			stateMementos.put(memento, memento);
			
			
			memento.setMementoIndex(++lastAssignedMementoIndex);
			equalMemento = memento;
		}
		
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".getMementoIndex() - after call \t" + toString());} 
		if (DEBUG_INFO) { out.println("Dumping content after the call\n"); DEBUG_printMapping(); }
		if (INTERNAL_CHECKS) { DEBUG_checkStructure(); }
		return equalMemento.getMementoIndex();
	}
	
	public String toString() {
		return this.getClass().getSimpleName() + " - Statictics: uniqueStateMementos=" + stateMementos.size() + ", seen equalsMementos=" +cntEqualMementoInstances + ", getMementoIndex calls=" + cntGetMementoIndex; 
	}
	
	/**
	 * Checks mapped {@link StatesMapperLayer.MappedStateHashableMemento} for hash code uniqueness.
	 * Used in the final output summary.
	 * @return String with short report of found hash collisions.
	 */
	public String reportHashCollisions() {
		StringBuffer result = new StringBuffer();
		// Found hash collisions
		HashMap<Integer, MappedStateHashableMemento> positionsByHashCode = new HashMap<Integer, MappedStateHashableMemento>();
		int collisions = 0;
		for(StatesMapperLayer.MappedStateHashableMemento memento : stateMementos.keySet()) {
			assert memento != null;
			
			int memHash = memento.hashCode();
			
			if (positionsByHashCode.containsKey(memHash)) {
				// Collision found
				collisions++;
				if (DEBUG_COLLISIONS && collisions < 10) {
					// Only in debug prints
					result.append("Colision detected (" + collisions + ") State1=" + memento.toString() + " State2=" + positionsByHashCode.get(memHash).toString() + "\n");
				}
			} else {
				positionsByHashCode.put(memHash, memento);
			}
		}
		if (collisions > 0) {
			result.append("Notice: Detected" + collisions + " hash collisions. Colisions do not influence the result.\n");
		} else {
			result.append("Fine: No hashCode() collisions in the stored states detected.\n");
		}
		return result.toString();
	}
	
	public void DEBUG_checkStructure() {
		if (stateMementos.size() != lastAssignedMementoIndex+1) {
			throw new RuntimeException(this.getClass().getSimpleName() + " - memento index counter and number of stored mementos not match");
		}
		
		// No invalid memento index
		for(StatesMapperLayer.MappedStateHashableMemento memento : stateMementos.values()) {
			assert memento != null;
			if (memento.getMementoIndex() == MappedStateHashableMemento.INVALID_MEMENTO_INDEX) {
				throw new RuntimeException(this.getClass().getSimpleName() + " - memento with invalid index.");
			}
			if (!(memento.getMementoIndex() <= lastAssignedMementoIndex)) {
				throw new RuntimeException(this.getClass().getSimpleName() + " - memento index to big.");
			}
		}
		
		StatesMapperLayer.MappedStateHashableMemento[] storedData = stateMementos.values().toArray(new StatesMapperLayer.MappedStateHashableMemento[0]);
		
		// Uniqueness
		for(int i = 0; i < storedData.length; i++) {
			final StatesMapperLayer.MappedStateHashableMemento iMem = storedData[i];
			for(int j = i+1; j < storedData.length; j++) {
				final StatesMapperLayer.MappedStateHashableMemento jMem = storedData[j];
				// Check uniqueness of stored instances
				if (iMem.equals(jMem)) {
					throw new RuntimeException(this.getClass().getSimpleName() + " more equal memtos in list.");
				}
				// Unique indexes
				if (iMem.getMementoIndex() == jMem.getMementoIndex()) {
					throw new RuntimeException(this.getClass().getSimpleName() + " more mementos with same index.");
				}
			}
		}
	}
	
	/**
	 *  Prints stored mapping into the output.
	 */
	public void DEBUG_printMapping() {
		out.println(this.getClass().getSimpleName() + " - printing stored mapping (sorting)");
		StatesMapperLayer.MappedStateHashableMemento[] storedData = stateMementos.values().toArray(new StatesMapperLayer.MappedStateHashableMemento[0]);
		
		Comparator<StatesMapperLayer.MappedStateHashableMemento> comp = 
			new Comparator<StatesMapperLayer.MappedStateHashableMemento>() {

				@Override
				public int compare(MappedStateHashableMemento left,
						MappedStateHashableMemento right) {
					
					return left.getMementoIndex() - right.getMementoIndex();
				}
			};
		
		Arrays.sort(storedData, comp);
		
		for(int i = 0; i < storedData.length; i++) {
			out.println("  " + i + " --> " + storedData[i].toString(4));
		}
		
		
	}
}
