package org.ow2.dsrg.fm.tbpjava.checker;

import java.io.PrintStream;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.Checker;
import org.ow2.dsrg.fm.tbpjava.EnvGenerator;
import org.ow2.dsrg.fm.tbpjava.EnvGenerator.EnvironmentDescription;
import org.ow2.dsrg.fm.tbpjava.checker.EventItem.EventTypes;
import org.ow2.dsrg.fm.tbpjava.checker.ThreadAutomatons.ThreadType;
import org.ow2.dsrg.fm.tbpjava.utils.Configuration;

import gov.nasa.jpf.PropertyListenerAdapter;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.search.Search;

import gov.nasa.jpf.jvm.MethodInfo;
import gov.nasa.jpf.jvm.StackFrame;
import gov.nasa.jpf.jvm.ThreadInfo;
import gov.nasa.jpf.jvm.ThreadList;
import gov.nasa.jpf.jvm.bytecode.INVOKESTATIC;
import gov.nasa.jpf.jvm.bytecode.Instruction;
import gov.nasa.jpf.jvm.bytecode.InvokeInstruction;
import gov.nasa.jpf.jvm.bytecode.ReturnInstruction;

//TODO error ... update error reasons
//TODO test for instances, to be able recognize different instances of provided and required interfaces (of component)
//TODO ... testing for provided interface call made base on properties
// 			ne podle pripraveneho listu, ale instance environment class (first test) bude mit seznam properties environment class a jeji mapovani na provided interface ... 
//			pri callu a returnu bych se dival, jestli neni na instanci ulozene v nektere z techto properties.

/**
 * Standard JPF listener, that observes executed instruction 
 *   (calls, returns, exception), threads (creation, ends) and 
 *   walking through state space.
 * <br>
 * Forwards (filtered) important events into {@link EventParser}.
 *   Important events are calls, returns, exception on component interfaces, 
 *   thread creation and ends.
 * <br>
 * Takes care of error reporting.
 */
public class ProtocolListener extends PropertyListenerAdapter {
	final static private boolean DEBUG = false;
	final static private boolean DEBUG_INFO = false;

	private static final String METHOD_NAME_THREAD_RUN = "run"; /// Thread main method name
	private static final String METHOD_NAME_CONSTRUCTOR = "<init>"; /// Method name of constructor in stack  

	private EventParser eventParser; /// Event parser, where pass collected events for further processing.
	final private StatesMapper varStateMap; /// Maps automaton states into sets of associates variables arrays. Used only for initialization


	private Map<String, String> mapFullName2ItfsName; /// Map string in format "implementing class name"."methodName" into interface name (from specification). (That implements given method if more interface implemented by impl. class)
	private Stack<StackEntryStruct> stackEvents = new Stack<StackEntryStruct>(); /// Stack with events from each step of JPF. Note: Stack entry could be null if no event occurs.

	private PrintStream out = System.out; /// Stream used to print outputs.

	// statistics
	private int uniqueStates = 0; /// Counts number of visited unique states. (Each state is added maximally once)
	private int visitedStates = 0; /// Counts all visited states. Each visit of same state (due to backtracking) is added. 

	// This classes are good for debugging and needed for creating reverted map
	private Map<String, String> provItfs2implClass; /// Maps names of provided interface names (from specification) into class names that implements this interfaces.
	private Map<String, String> reqItfs2implClass;  /// Maps names of required interface names (from specification) into class names that implements this interfaces.

	private Map<Integer, ThreadAutomatons.ThreadType> startedThreads = new HashMap<Integer, ThreadAutomatons.ThreadType>(); /// Holds thread numbers that has been started in last JPF step and thread type. 
	private List<Integer> endedThreads = new ArrayList<Integer>(); /// Holds thread numbers that has ended during last JPF step)

	// End and error processing
	private boolean checkFailed = false; /// Hold true if error was found.
	private boolean endStateOccured = false; /// Hold true if we found  end state. So exists execution where program ends.
	private boolean errorReported = false; /// Holds true if we report error to user.
	private String errorString = ""; /// Standardized error string (for first error)
	
	private final String environmentClassName; /// Name of class generated by environment generator that contains main and generated environment.
	private final Configuration config; /// Configuration of Checker
	
	private final List<EventItem> eventList = new ArrayList<EventItem>(); /// List where {@link #instructionExecuted(JVM)} adds event during JPF step (transition)
	
	private JPFProgramStateMapping programStateMapper = null; /// JPF state mapper to which we has to notify about forward and backward steps
	
	/**
	 * Creates protocol listener.
	 * 
	 * @param config Parsed Checker configuration. Cann't be null.
	 * @param eventParser Event parser, where pass collected events for further processing. Cann't be null.
	 * @param envDesc Description of the processed environment. Cann't be null.  
	 * @param reqItfs2implClass  Maps interface name (specification name scope) to classes that implements them and are used in tested environment. Holds information for required interface. Typically points to generated stub classes. Cann't be null.
	 * @param environmentClassName Name of that is tested by JPF - class with environment.
	 * @param out Stream used for printing output.
	 */
	public ProtocolListener(Configuration config, EventParser eventParser, StatesMapper varStateMap, EnvironmentDescription envDesc, Map<String, String> reqItfs2implClass, String environmentClassName, PrintStream out) {
		if (DEBUG) { out.println("DEBUG - " + ProtocolListener.class.getSimpleName() + "." + ProtocolListener.class.getSimpleName() + "(config=" + config + ", eventParser=" + eventParser + ", provItfs2implClass=" + provItfs2implClass + ", reqItfs2implClass=" + reqItfs2implClass + ",environmentClassName=" + environmentClassName + ", out=" + out + ")"); }
		assert environmentClassName != null;

		this.config = config;
		this.provItfs2implClass = generateProvItfs2implClass(envDesc);
		this.reqItfs2implClass = reqItfs2implClass;
		this.eventParser = eventParser;
		this.environmentClassName = environmentClassName;
		this.varStateMap = varStateMap;
		
		if (out != null) {
			this.out = out;
		}
		mapFullName2ItfsName = createFullNameMap(config, provItfs2implClass, reqItfs2implClass, out);
		
		
	}

	/**
	 * JPF starts state traversal
	 * 
	 * @param search {@link Search} descendant used for generating next states to explore. 
	 */
	public void searchStarted (Search search) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".searchStarted(...)"); }
		stackEvents.clear();
		endedThreads.clear();
		startedThreads.clear();

		// Set eventPraser
		if (search instanceof JPFSearch) {
			JPFSearch jpfSearch = (JPFSearch)search;
			jpfSearch.setStatesMapper(varStateMap);
			programStateMapper = jpfSearch.getJPFProgramStateMapping();
			eventParser.initializeEventParser();
		} else {
			throw new RuntimeException("Invalid search used. Expected " + JPFSearch.class.getName() + " but used " + search.getClass().getName());

			// Failsafe possibility ... program runs but don't get expected results, can miss some errors
			//JPFProgramStateMapping programStateMapper = new JPFProgramStateMapping();
			//eventParser.setJPFProgramStateMapper(programStateMapper);
		}

		// Move in automatons
		super.searchStarted(search);
	}

	/**
	 * JPF stop searching. Used to print usage statistics and reporting problems.
	 * 
	 * @param search {@link Search} descendant used for state traversing.
	 */
	public void searchFinished(Search search) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".searchFinished(...)"); }

		if ((!checkFailed) && (!endStateOccured)) {
			out.println("Warning: no end state occured - there is probably an infinite loop or a deadlock in the code");
		}

		if (checkFailed) {
			out.println("Interface calls event trace");
			printEventStack(true);
		}
		

		out.println("Unique states: " + uniqueStates);
		out.println("Visited states: " + visitedStates);
		out.println(programStateMapper.getUsageStatistics());
	}


	/**
	 * JPF calls if during state traversal (forward step) move into another (new/next) state.
	 * <br>
	 * Our task is to extract occurred events(started and finished threads) and 
	 *  pass them to further processing to {@link EventParser}.
	 *  
	 * @param search {@link Search} descendant that holds current state of executed program and JPF.
	 */
	public void stateAdvanced(Search search) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".stateAdvanced(...), stackDepth=" + stackEvents.size()); }
		
		if (search.isNewState()) uniqueStates++;
		visitedStates++;

		int jpfStateID = search.getVM().getSystemState().getId();
		if (eventList.size() > 0 || startedThreads.size() > 0 || endedThreads.size() > 0) {
			
			stackEvents.add(new StackEntryStruct(eventList, jpfStateID, startedThreads, endedThreads));
			checkFailed |= !eventParser.processEvents(startedThreads, eventList, endedThreads);
		} else {
			// No interesting event takes place
			stackEvents.push(null);
		}
		programStateMapper.advancedEvents(eventList, jpfStateID); // Actualize mapping of reached protocol states to program state

		if (search.isEndState()) {
			endStateOccured = true;
		}
		startedThreads.clear();
		endedThreads.clear();
		eventList.clear();

	}

	/**
	 * JPF makes step back to previous state. 
	 *  Updates informations about called and returned events. 
	 *  Notify {@link EventParser} about backtracking if needed.
	 *  
	 * @param search {@link Search} descendant that holds current state of executed program and JPF.
	 */
	public void stateBacktracked(Search search) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".stateBacktracked(...), stackDepth=" + stackEvents.size()); }
		//{ out.println(this.getClass().getSimpleName() + ".stateBacktracked(...), stackDepth=" + stackEvents.size()); }

		StackEntryStruct event = stackEvents.pop();
		if (event != null) {
			eventParser.undoEvents();
		}
		programStateMapper.undoEvents();
	}

	/**
	 * State is fully processed (all its descendants). And ready to backtrack.
	 * 
	 * @param search {@link Search} descendant that holds current state of executed program and JPF.
	 */
	public void stateProcessed(Search search) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".stateProcessed(...), stackDepth=" + stackEvents.size()); }

		// Save mapping between current JPF program state and TBP protocol state
		int currentJPFStateID = search.getVM().getStateId();
		programStateMapper.addMapping(currentJPFStateID);
	}

	/**
	 * JPF calls after instruction is executed.
	 * <br>
	 * Breaks JPF transition if important call occur. 
	 * (Not to miss different events sequence due to POR).
	 * Interface method call is scheduling relevant event. 
	 *
	 * @param vm Holds state of JVM with executed program
	 */
	public void instructionExecuted(JVM vm) {
		// if (DEBUG) { out.println(this.getClass().getSimpleName() + ".instructionExecuted(...)"); }

		Instruction instr = vm.getLastInstruction();
		ThreadInfo ti = vm.getLastThreadInfo();
		
		if (DEBUG_INFO) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".instructionExecuted(...)- instr: type = "+instr.getClass().getName()+", location = " + instr.getFileLocation()); }

		if (instr instanceof InvokeInstruction) {

			InvokeInstruction invokeInstr = (InvokeInstruction) instr;
			MethodInfo mi = invokeInstr.getInvokedMethod();

			String className = getClassName(mi);
			String methodName = getMethodName(mi);

			String fullMethodName = className + "." + methodName;

			String ifaceName = mapFullName2ItfsName.get(fullMethodName);
			// Check if method call is on known (provided, required) interface
			if (ifaceName != null) {
				if (DEBUG) { out.println(this.getClass().getSimpleName() + ".instructionExecuted() - Component method call " + fullMethodName + ", thread=" + ti.getIndex()); }
				EventItem event = new EventItem(ti.getIndex(), EventItem.EventTypes.EVENT_CALL, methodName, ifaceName, instr.getLineNumber());
				//System.out.println("New call event detected - " + event);
				eventList.add(event);

				Instruction nextInst = vm.getNextInstruction();
				if ( !isVerifyCall(nextInst)) {
					// Verify calls  (ex getNextBoolean())... breaks Transition on its own
					//   the getNextBoolean behaves badly if now instruction is executed after we break the transition
					ti.reschedule(true);
				}
			}
		} else if (instr instanceof ReturnInstruction) {
			MethodInfo mi = instr.getMethodInfo();
			
			String className = getClassName(mi);
			String methodName = getMethodName(mi);

			String fullMethodName = className + "." + methodName;

			String ifaceName = mapFullName2ItfsName.get(fullMethodName);
			// Check if method call is on known (provided, required) interface
			if (ifaceName != null) {
				if (DEBUG) { out.println(this.getClass().getSimpleName() + ".instructionExecuted() - Component method return " + fullMethodName + ", thread=" + ti.getIndex()); }
				// Note return events have invalid lines (lines where return instruction takes place)
				EventItem event = new EventItem(ti.getIndex(), EventItem.EventTypes.EVENT_RETURN, methodName, ifaceName, ti.getLine());
				//System.out.println("New return event deTetected - " + event);
				eventList.add(event);

				Instruction nextInst = vm.getNextInstruction();
				if ( !isVerifyCall(nextInst)) {
					// Verify calls  (ex getNextBoolean())... breaks Transition on its own
					//   the getNextBoolean behaves badly if now instruction is executed after we break the transition
					ti.reschedule(true);
				}
			}
		}
	}

	/**
	 * JPF calls on thread end.
	 * <br>
	 * We collect terminated threads to know if thread ends are correct 
	 * (no return or call event in such thread are expected). 
	 *
	 * @param vm Holds state of JVM with terminated thread.
	 */
	public void threadTerminated (JVM vm) {
		ThreadInfo ti = vm.getLastThreadInfo();
		endedThreads.add(ti.getIndex());

		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".threadTerminated(...) + threadNum=" + ti.getIndex()); }
		//{ out.println(this.getClass().getSimpleName() + ".threadTerminated(...) + threadNum=" + ti.getIndex()); }
	}

	/**
	 * JPF calls on thread start.
	 * <br>
	 * We collect started threads to keep track of threads and its origin. 
	 *
	 * @param vm Holds state of JVM with started thread.
	 */
	public void threadStarted (JVM vm) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".threadStarted(...) + threadNum=" + vm.getLastThreadInfo().getIndex()); }
		//out.println(this.getClass().getSimpleName() + ".threadStarted(...) + NewThreadNum=" + vm.getLastThreadInfo().getIndex() + " StardedThreadNum=" + vm.getCurrentThread().getIndex()); 
		//DEBUG_printJVMThreadsWithStack(vm, out);

		ThreadInfo threadStarted = vm.getCurrentThread(); /// Thread that calls start() method on thread variable.
		startedThreads.put(vm.getLastThreadInfo().getIndex(), analyzeStackThreadCreation(threadStarted));
	}

	public void exceptionThrown(JVM vm) {
		if (DEBUG) { 
			ThreadInfo ti = vm.getLastThreadInfo();
			out.println(this.getClass().getSimpleName() + ".exceptionThrown(...) threadNum=" + ti.getIndex() + "pendingException=" + vm.getPendingException()); }
	}
	
	public void exceptionBailout(JVM vm) {
		ThreadInfo ti = vm.getLastThreadInfo();
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".exceptionBailout(...) threadNum=" + ti.getIndex() + "pendingException=" + vm.getPendingException()); }
		StackFrame topSF = ti.getTopFrame();
		MethodInfo mi = topSF.getMethodInfo(); /// Method that we leave by throwing the exception
		String className = getClassName(mi);
		String methodName = getMethodName(mi);
		String fullMethodName = className + "." + methodName;
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".exceptionBailout(...) \tLeavingMethod=" + mi.getCompleteName()); }
		String ifaceName = mapFullName2ItfsName.get(fullMethodName);
		// Check if method call is on known (provided, required) interface
		if (ifaceName != null) {
			// Note return events have invalid lines (lines where return instruction takes place)
			EventItem event = new EventItem(ti.getIndex(), EventItem.EventTypes.EVENT_RETURN, methodName, ifaceName, ti.getLine());
			if (DEBUG) { out.println(this.getClass().getSimpleName() + ".exceptionBailout(...) \tGeneratingReturnEvent=" + event); }
			eventList.add(event);
			ti.reschedule(true);
		}
	}
	public void exceptionHandled(JVM vm) {
		if (DEBUG) { 
			ThreadInfo ti = vm.getLastThreadInfo();
			out.println(this.getClass().getSimpleName() + ".exceptionHandled(...) threadNum=" + ti.getIndex() + "pendingException=" + vm.getPendingException()); }
	}

	/**
	 * JPF calls after each step to check if no property is violated.
	 * <br>
	 * Checking take place during event processing in {@link #stateAdvanced(Search)} 
	 *  so we only process saved flags.
	 * 
	 * @return true if property is NOT violated 
	 */
	public boolean check(Search search, JVM vm) {
		if (DEBUG) { out.println("DEBUG - " + this.getClass().getSimpleName() + ".check(...)"); }

		if (errorReported) return true;

		if (checkFailed) errorReported = true;

		return !checkFailed;
	}

	/**
	 * @return Gets (generic) error message with problem description.
	 */
	public final String getErrorMessage() {
		return "Tested component is not compliant with the protocol\n" + errorString ;
	}

	/**
	 * Getter for {@link #provItfs2implClass} map. For each 
	 * @return Map that for all provided interface holds name of class that implement interface. 
	 *   (and acts as source of events)  
	 */
	public final Map<String, String> getProvItfs2implClass() {
		return provItfs2implClass;
	}

	/**
	 * Getter for {@link #reqItfs2implClass} map. For each 
	 * @return Map that for all required interface holds name of class that implement interface. 
	 *   (and acts as source of events)  
	 */
	public final Map<String, String> getReqItfs2implClass() {
		return reqItfs2implClass;
	}




	/* utility methods */
	/**
	 * Prints into {@link ProtocolListener#out} stream {@link EventItem} that 
	 *   occur from start of tested application into current state. 
	 * (Note backtracked event are removed.)
	 *  Prints events only from steps currently on JPF stack.
	 * @param filterSubsequentProvidedCalls If false print all events, if true (multiple(subsequent)) provided calls are filtered out (if provided method calls other provided method)
	 */
	public void printEventStack(boolean filterSubsequentProvidedCalls) {
		// Compute number of threads
		int maxThreadNum = 0;
		for(StackEntryStruct stackStepEntry : stackEvents) {
			if (stackStepEntry == null) {
				continue;
			}
			List<EventItem> eventList = stackStepEntry.getEvents();
			if (eventList != null) {
				for(EventItem event : eventList) {
					if (event.thread > maxThreadNum) {
						maxThreadNum = event.thread;
					}
				}
			}
		}
		
		int[] providedCallDepth = new int[maxThreadNum+1];
	
		out.println(EventItem.eventTraceHeader());
		out.println(EventItem.eventTraceSeparator());
		for(StackEntryStruct stackStepEntry : stackEvents) {
			//boolean printSeparator = false; // Whether print separating line into the output .. only if any thread created or ended
			if (stackStepEntry != null) {
				if (stackStepEntry.getStartedThreads() != null && stackStepEntry.getStartedThreads().isEmpty() == false) {
					out.print("Started threads :");
					boolean first = true;
					for(Entry<Integer, ThreadAutomatons.ThreadType> startedThread : stackStepEntry.getStartedThreads().entrySet()) {
						if (first) { 
							first = false;
						} else {
							out.print(", ");
						}
						out.print(startedThread.getKey() + "(" + startedThread.getValue() + ")");
					}
					out.println();
					//printSeparator = true;
				}

				List<EventItem> eventList = stackStepEntry.getEvents();
				if (eventList != null) {
					int eventToPrint = eventList.size();
					// If last stack item (last processed events that caused error) .. so not all events have to be processed
					if (eventList.equals(stackEvents.lastElement())) {
						int processedEvents = eventParser.getProcessEventIndexErrorEvent() + 1; // One for problematic error creating event
						if (processedEvents != EventParser.NO_ERROR && processedEvents >= 0 && processedEvents < eventList.size()) {
							eventToPrint = processedEvents;
						}
					}
					for(int i = 0; i < eventToPrint; i++) {
						EventItem event = eventList.get(i);
						boolean isProvided = getProvItfs2implClass().containsKey(event.interfaceName);
						if (isProvided) {
							// Provided call
							if (event.type == EventTypes.EVENT_CALL) {
								providedCallDepth[event.thread]++;
							} else if (event.type == EventTypes.EVENT_RETURN) {
								// Nothing to do
							} else {
								throw new RuntimeException(Checker.INTERNAL_ERROR_PREFIX + "Unknown EventTypes entry=" + event.type);
							}
							if (!(filterSubsequentProvidedCalls && providedCallDepth[event.thread] > 1)) {
								out.println(event.toStringEventTrace(this));
							}
							if (event.type == EventTypes.EVENT_CALL) {
								// Nothing to do
							} else if (event.type == EventTypes.EVENT_RETURN) {
								providedCallDepth[event.thread]--;
							} else {
								throw new RuntimeException(Checker.INTERNAL_ERROR_PREFIX + "Unknown EventTypes entry=" + event.type);
							}
						} else {
							// Required calls are printed always
							out.println(event.toStringEventTrace(this));
						}
					}
					
					if (stackStepEntry.getEndedThreads() != null && stackStepEntry.getEndedThreads().isEmpty() == false) {
						out.print("Ended threads :");
						boolean first = true;
						for(Integer endedThread : stackStepEntry.getEndedThreads()) {
							if (first) { 
								first = false;
							} else {
								out.print(", ");
							}
							out.print(endedThread);
						}
						out.println();
						//printSeparator = true;
					}

				}
				//if (printSeparator) {
				//	out.println(EventItem.eventTraceSeparator());
				//}
			} // if stackStepEntry not null
		} // for each stackEvents
	}

	/** 
	 * Lists all threads in JVM, prints their stack and state.
	 * Shows active thread and currently running thread.
	 * 
	 * @param jvm JVM to inspect.
	 */
	@SuppressWarnings("unused")
	private void DEBUG_printJVMThreadsWithStack(JVM jvm, PrintStream out) {
		out.println("Debuging JVM state - threads");

		ThreadList tl = jvm.getThreadList();
		ThreadInfo[] tis = tl.getThreads();
		for(ThreadInfo ti : tis) {
			out.print("   Thread:" + ti + ", ThreadNum: " + ti.getIndex() + ", Status=" + ti.getStateName() + ", NewlyStartedThread=" + (jvm.getLastThreadInfo() == ti) + ", ThreadCalledStart=" + (jvm.getCurrentThread() == ti) + ", ThreadType=");
			try {
				ThreadAutomatons.ThreadType tt = getThreadType(ti.getIndex()); /// Can throw Runtime exception if thread was not started until now ... it's legal, non DEBUG program shouldn't be interested in such information
				out.println(tt);
			} catch (RuntimeException e) {
				// Note resolved thread type
				out.println("???");
			}
			for(StackFrame sf : ti.getStack()) {
				out.println("       Class:" + sf.getClassName() + ", Method: " + sf.getMethodName() + ", line=" + sf.getLine());
			}
		}

	}
	
	
	/**
	 * Derive Class name from {@link MethodInfo}.
	 * 
	 * @param mi Specification of method its class name is interesting.
	 * @return Gets name of class where given method is defined.
	 */
	static private String getClassName(MethodInfo mi) {
		if (mi == null) return "";
		if (mi.getClassInfo() == null) return "";

		return mi.getClassInfo().getName();
	}

	/**
	 * Derive Method name from {@link MethodInfo}
	 * 
	 * @param mi Specification of method its name is interesting.
	 * @return Gets name of given method.
	 */
	static private String getMethodName(MethodInfo mi) {
		if (mi == null) return "";

		return mi.getName();
	}
	
	static final private boolean isVerifyCall(Instruction instr) {
		return (instr instanceof INVOKESTATIC) && (((INVOKESTATIC)instr).getInvokedMethod().getClassName().equals("gov.nasa.jpf.jvm.Verify"));
	}

	/**
	 * Create map for faster interesting events identification.
	 * 
	 * @param config Parsed Checker configuration. Cann't be null. Holds specification interfaces names.
	 * @param provItfs2implClass Maps interface name (specification name scope) to classes that implements them and are used in tested environment. Holds information for provided interface. Typically map interfaces to tested component class name. Cann't be null.  
	 * @param reqItfs2implClass  Maps interface name (specification name scope) to classes that implements them and are used in tested environment. Holds information for required interface. Typically points to generated stub classes. Cann't be null.
	 * @param out Stream where print errors. Cann't be null.
	 * @return Gets map that string in format "implementing class name"."methodName" maps into interface name (from specification). See {@link #mapFullName2ItfsName} 
	 */
	static private Map<String, String> createFullNameMap(Configuration config,
			Map<String, String> provItfs2implClass, Map<String, String> reqItfs2implClass, PrintStream out) {
		assert config != null;

		ClassLoader cl = Thread.currentThread().getContextClassLoader();

		Map<String, String> result = new HashMap<String, String>();

		createFullNameMap1(result, config.getComponentProvidedInterfaces(), provItfs2implClass, out, cl);
		createFullNameMap1(result, config.getComponentRequiredInterfaces(), reqItfs2implClass, out, cl);

		return result;
	}

	/**
	 * Helper method for {@link #createFullNameMap(Configuration, Map, Map, PrintStream)}.
	 * Adds into given result map mappings......
	 * 
	 * @param result Map where add new mappings. For newly added mapping format see (@link {@link ProtocolListener#mapFullName2ItfsName})
	 * @param interfaces Interface names from specification. Cann't be null.
	 * @param implClasses Maps interface name (specification name scope) to classes that implements them. Cann't be null.
	 * @param out Stream where print errors. Cann't be null.
	 * @param cl Class loader that should be used for generated classes loading
	 * @return True if error found and reported, false otherwise.
	 */
	static private boolean createFullNameMap1(Map<String, String> result, Map<String, String> interfaces, Map<String, String> implClasses, PrintStream out, ClassLoader cl) {
		assert result != null;
		assert interfaces != null;
		assert implClasses != null;
		assert out != null;

		boolean error = false;

		for(Entry<String, String> entry : interfaces.entrySet()) {
			final String ifaceName = entry.getKey();
			final String ifaceClassName = entry.getValue();

			if (DEBUG) { out.println("DEBUG - Processing iface " + ifaceName); }

			// Found interface Class
			Class<?> ifaceClass = null;
			try {
				ifaceClass = Thread.currentThread().getContextClassLoader().loadClass(ifaceClassName);
				//ifaceClass = Class.forName(ifaceClassName, true, cl);
			} catch (ClassNotFoundException e) {
				out.println("Error - interface " + ifaceName +" definition (Class " + ifaceClassName + ") not found.");
				error = true;
			}


			String ifaceImplClassName = implClasses.get(ifaceName);
			if (ifaceImplClassName == null) {
				out.println("Error - not specified class implementing interface " + ifaceName);
				error = true;
			}

			// Check if implementing class exists
			Class<?> ifaceImplClass = null;
			try {
				ifaceImplClass = Thread.currentThread().getContextClassLoader().loadClass(ifaceImplClassName);
				//ifaceImplClass = Class.forName(ifaceImplClassName, true, cl);
			} catch (ClassNotFoundException e) {
				out.println("Error - implementing class not found. " + ifaceImplClassName);
				error = true;
			}

			// Check if implements processed interface
			if (ifaceImplClass != null && ifaceClass != null) {

				if (ifaceClass.isAssignableFrom(ifaceImplClass) == false) {
					out.println("Error - class (" + ifaceImplClassName + ") not implement specified interface (" + ifaceClassName + ")");
					error = true;
				}
			}
			if (!error) {
				// Testing done -> add mappings for all public method for interface
				for(Method ifaceMethod : ifaceClass.getMethods()) {
					String ifaceMethodName = ifaceMethod.getName();
					String addedName = ifaceImplClassName + "." + ifaceMethodName;

					if (DEBUG) { out.println("DEBUG -   Adding mapping " + addedName + "-->" + ifaceName); }
					result.put(addedName, ifaceName);
				}
			}
		}
		return error;
	}

	/**
	 * Analyze stack with thread position where new thread is created and 
	 *   according position decide type of newly created thread - 
	 *   which behavior is expected from thread.
	 *     
	 * @param ti Thread that started new thread. (Thread that calls start())
	 * @return Type of newly created thread.
	 * 
	 * Note: Depends on generated environment.
	 * Note: Currently recognize 3 types of threads.
	 *   a) Threads from environment - they simulates enabled behavior from provision
	 *      They directly call only provided methods.
	 *      Stack is in form of EnvironmentGeneratedClass.main ; EnvironmentGeneratedClass.runEnvironment
	 *          of in form EnvironmentGeneratedClass.runEnvironment$num.run() in thread that is with a) type
	 * 
	 *   b) Threads from tested component - they are checked against Thread section from specification
	 *       Stack contains call of the component constructor or provided method.
	 *       EnvironmentGeneratedClass.main ; EnvironmentGeneratedClass.<init>; TestedComponent.<init>
	 *       EnvironmentGeneratedClass.runEnvironment$num.run()
	 * 
	 *   c) Other threads  (Threads created be object passed as parameters - objects from EnvValueSets)
	 *      No events are permitted.
	 *      
	 *      Stack is in form EnvironmentGeneratedClass.main ; EnvironmentGeneratedClass.<init> ; EnvValueSets.<init> ; ...
	 *      All other stacks are considered as no event threads too. 
	 */
	private ThreadAutomatons.ThreadType analyzeStackThreadCreation(ThreadInfo ti) {
		if (ti == null) {
			return ThreadType.NONE_EVENT_THREAD;
		}
		
		List<StackFrame> sfs = ti.getStack();
		
		// Check for a) type
		if (sfs.size() == 3 && isMethodCall(sfs.get(0), "main") && isMethodCall(sfs.get(1), EnvGenerator.METHOD_GENERATED_PROTOCOL_NAME)) {
			return ThreadType.ENVIRONMENT_THREAD;
		}
		
		if (sfs.size() == 2 && isRunMethodCall(sfs.get(0)) && getThreadType(ti.getIndex()) == ThreadType.ENVIRONMENT_THREAD) {
			return ThreadType.ENVIRONMENT_THREAD;
		}
		
		// Check for c) type
		if (sfs.size() >= 3 && isMethodCall(sfs.get(0), "main") && isMethodCall(sfs.get(1), METHOD_NAME_CONSTRUCTOR) && isMethodCall(sfs.get(2), config.getEnvValueSets(), METHOD_NAME_CONSTRUCTOR)) {
			return ThreadType.NONE_EVENT_THREAD;
		}
		
		// Check for b) type
		int i = 0;
		// Goes down the stack and try to find interesting events
		for(i = 0; i < sfs.size(); i++) {
			StackFrame sf = sfs.get(i);
			// Checks whether provided interface call takes places
			
			
			String className = getClassName(sf.getMethodInfo());
			String methodName = getMethodName(sf.getMethodInfo());

			String fullMethodName = className + "." + methodName;
			String itfsName = mapFullName2ItfsName.get(fullMethodName);
			if (itfsName != null) {
				// Interesting call - provided or required call
				if (config.getComponentProvidedInterfaces().containsKey(itfsName)) {
					// Last call was provided ... component itself is responsible for created thread
					return ThreadType.THREAD_SECTION;
				}
				if (config.getComponentRequiredInterfaces().containsKey(itfsName)) {
					// Last call was required ... component is not responsible for created thread
					//  created thread mustn't call any interface method
					return ThreadType.NONE_EVENT_THREAD;
				}
			
			}
			
			if (isConstructorCall(sf, config.getComponentImplementationClasses())) {
				// Thread created in constructor of components
				return ThreadType.THREAD_SECTION;
			}
		} // End - for each stack frame
		
		// Maybe to restrictive ... what about other methods - setters, on provided, but also necessary?	
		// Unknown thread type
		return ThreadType.NONE_EVENT_THREAD;
		//throw new RuntimeException("Cannot determine type of newly created thread");
	}

	/**
	 * Checks whether given StackFrame represents constructor call of one from specified classes. 
	 * @param sf Stackframe to analyze.
	 * @param componentImplementationClasses Classes to consider
	 * @return true if Stack frame represents constructor of one from specifies classes.
	 */
	private boolean isConstructorCall(StackFrame sf, List<String> componentImplementationClasses) {
		for(String compClass : componentImplementationClasses) {
			if (isMethodCall(sf, compClass, METHOD_NAME_CONSTRUCTOR)) {
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Gets type of thread with given number.
	 *  Checks if not in started threads in current step, and then in older already running threads.
	 * 
	 * @param threadNum Number of thread
	 * @return Gets type of given thread. Throws exception when no such thread exists.
	 */
	private final ThreadAutomatons.ThreadType getThreadType(int threadNum) {
		ThreadAutomatons.ThreadType result =  startedThreads.get(threadNum);
		
		if (result == null) {
			result = eventParser.getThreadType(threadNum);
		}
		
		if (result == null) {
			//result = ThreadType.NONE_EVENT_THREAD;
			throw new RuntimeException("Internal error - Cann't determine thread type");
		}
		return result;
	}

	/**
	 * Checks whether given {@link StackFrame} represents call of given method on tested environment class. 
	 * 
	 * @param sf Stack frame to test.
	 * @param methodName Method which should by called in {@link StackFrame}
	 * @return True is given {@link StackFrame} represents call of given method on tested environment class. False otherwise.
	 */
	private final boolean isMethodCall(StackFrame sf, String methodName) {
		return isMethodCall(sf, environmentClassName, methodName);  
	}

	/**
	 * Checks whether given {@link StackFrame} represents call of given method on given class. 
	 * 
	 * @param sf Stack frame to test.
	 * @param className Class which method should by called in {@link StackFrame}
	 * @param methodName Method which should by called in {@link StackFrame}
	 * @return True is given {@link StackFrame} represents call of given method of given class. False otherwise.
	 */
	private final boolean isMethodCall(StackFrame sf, String className, String methodName) {
		return (sf != null) && ( className.equals(sf.getClassName()) && methodName.equals(sf.getMethodName()));  
	}

	/**
	 * Checks whether given {@link StackFrame} corresponds to run method of anonymous thread that is in environment class.
	 * Tests whether given stack frame corresponds to one of the anonymous threads from environment.
	 * 
	 * @param sf Stack frame to test.
	 * @return True if given {@link StackFrame} was created by call of "run" method of anonymous thread in tested generated environment class.
	 */
	private final boolean isRunMethodCall(StackFrame sf) {
		return (sf != null) && ( sf.getClassName().startsWith(environmentClassName + "$") && METHOD_NAME_THREAD_RUN.equals(sf.getMethodName()));  
	}

	/**
	 * Create map that maps provided interfaces to implementation component classes (which implements given interface) 
	 * @param envDesc Description of generated environment.
	 * @return Tha of the provided interfaces to class names (from component) which implements given interface.
	 */
	private static Map<String, String> generateProvItfs2implClass(EnvironmentDescription envDesc) {
		Map<String, String> provItfs2implClass = new HashMap<String, String>();

		for(Entry<String, String> provItfsEntry : envDesc.provItfs2envFieldName.entrySet()) {
			//We will use the field in the environment to found the class that which implements given interface
			String fieldName = provItfsEntry.getValue();
			
			for(Entry<String, String> compImpl2fieldEntry : envDesc.compImpl2envFieldName.entrySet()) {
				if (compImpl2fieldEntry.getValue().equals(fieldName)) {
					provItfs2implClass.put(provItfsEntry.getKey(), compImpl2fieldEntry.getKey());
					break;
				}
			}
		}
		return provItfs2implClass;
	}
	
	/**
	 * Structure that holds all types of events that occur in one JPF step.
	 *
	 */
	private static class StackEntryStruct {
		private List<EventItem> events; /// Event that take place during last step
		private int jpfStateID; /// Identification of the JPF state after transition end (in which all mapped events take place)
		private List<Integer> endedThreads; /// Number of threads that was ended during last JPF step
		private  Map<Integer, ThreadAutomatons.ThreadType> startedThreads; /// Threads that was create during last JPF step. To each step there is associated stack where creation of new thread takes place (to be able analyze thread type).
		
		/**
		 * Sets entries of structure.
		 * @param events Sets given list with entries. (Doesn't make copy)
		 * @param jpfStateID Identification of the JPF state after transition end (in which all mapped events take place)
		 * @param startedThreads Makes new copy of map with started threads.
		 * @param endedThreads Makes new copy of list with ended thread. 
		 */
		public StackEntryStruct(List<EventItem> events, int jpfStateID, Map<Integer, ThreadAutomatons.ThreadType> startedThreads, List<Integer> endedThreads) {
			this.events = new ArrayList<EventItem>(events);
			this.endedThreads = new ArrayList<Integer>(endedThreads);
			this.startedThreads = new HashMap<Integer, ThreadAutomatons.ThreadType>(startedThreads);
			this.jpfStateID = jpfStateID;
		}

		public final List<EventItem> getEvents() {
			return events;
		}

		public final int getJPFStateID() {
			return jpfStateID;
		}

		public final List<Integer> getEndedThreads() {
			return endedThreads;
		}

		public final Map<Integer, ThreadAutomatons.ThreadType> getStartedThreads() {
			return startedThreads;
		}
	}
}
