package org.ow2.dsrg.fm.tbpjava.checker;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.jvm.ElementInfo;
import gov.nasa.jpf.jvm.ExceptionInfo;
import gov.nasa.jpf.jvm.Heap;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.jvm.MJIEnv;
import gov.nasa.jpf.jvm.MethodInfo;
import gov.nasa.jpf.jvm.NoUncaughtExceptionsProperty;
import gov.nasa.jpf.jvm.ThreadInfo;
import gov.nasa.jpf.search.Search;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;

import org.ow2.dsrg.fm.tbpjava.utils.Configuration;

/**
 * Checks whether our generated environment doesn't end on uncaught exception.
 * 
 * It updates original JPF property checker error string to be more meaningful in
 * our context.
 * 
 * Another most complicated task is mask environment exception wrapping into RuntimeException.
 * 
 * @author Alfifi
 *
 */
public class PropertyUncaughtExceptions extends NoUncaughtExceptionsProperty {
	final static private boolean DEBUG = false;

	private static final String RUNTIME_EXCEPTION_NAME = "java.lang.RuntimeException";
	
    private PrintStream out = System.out; /// Stream used to print outputs.
    private Configuration checkerConfig;
    private String checkedEnviromentClassName; /// Name of generated environment class that is tested.

    /**
     * 
     * @param config JPF configuration
     */
	public PropertyUncaughtExceptions(Config config) {
		super(config);
	}

	/**
	 * Initialize class to be able work correctly.
	 * 
	 * @param checkerConfig Parsed configuration of checker. Cann't be null.
	 * @param enviromentClassName Name of generated environment class that is tested Cann't be null. 
	 * @param out Stream where print debug prints and other output. If null uses standard output.
	 */
	public void initialize(Configuration checkerConfig, String enviromentClassName, PrintStream out) {
		assert checkerConfig != null;
		assert enviromentClassName != null;
		
		if (out != null) {
			this.out = out;
		}
		this.checkerConfig = checkerConfig;
		this.checkedEnviromentClassName = enviromentClassName;
	}
	
	@Override
	public String getExplanation () {
			return "Tested method throw exception. Typicaly it is bad method call (illegal parameter or invalid call time). Check provision definition (int TPB protocol) or parameter definition in " + checkerConfig.getEnvValueSets() + " class" ;
	}

	@Override
	public String getErrorMessage() {
		ExceptionInfo expI = getUncaughtExceptionInfo();
		if (expI != null) {			
			// Check if runtime exception
			if (checkIfCheckerException(expI)) {
				// Found true reason of exception -> cause wrapped in given RuntimeException
				Heap da = JVM.getVM().getHeap();

				int origExcRef = getOriginalExceptionHeapRef();
				ElementInfo origExcElemI = da.get(origExcRef);
				if (origExcElemI == null) {
					// It looks reference field points to invalid "address"
					//   it's strange ... not runtime exception from generated environment of change of JPF behavior  
					// Should not be null. If null occur its unexpected JPF behavior  
					throw new RuntimeException("Unexpected JPF behavior");
				}
				
				return exceptionInfoPrintOn(expI.getThread(), origExcRef);
			}
		}
		return super.getErrorMessage();
	}
	
	/**
	 * Check if exception was thrown in catch block in generated environment. (As concatenation of any previous exception)
	 * <br>
	 * Check if exception is RuntimeException thrown from generated environment file.
	 * 
	 * @param ei Exception description. Cann't be null.
	 * @return True if exception as describe before, false otherwise.
	 */
	private boolean checkIfCheckerException(ExceptionInfo ei) {
		assert ei != null;
		
		// Check exception type
		if (!RUNTIME_EXCEPTION_NAME.equals(ei.getExceptionClassname())) {
			return false;
		}
		
		// Version 2 ... process stack trace
		// Idea taken from ThreadInfo.printStackTrace ... stay update with them
		// Depends on JPF internal structure ... works with rev1258
		Heap da = JVM.getVM().getHeap();
		assert da != null; // For sure check ... hope that JPF behavior in this way
		
		int excRef = ei.getExceptionReference();

		ElementInfo expElemI = da.get(excRef);// RunTime representation of RuntimeException
		if (expElemI == null) {
			// Should not be null. If null occur its unexpected JPF behavior  
			throw new RuntimeException("Problem with JPF exception representation");
		}
		
		// Get stack trace element info
		int stackTraceRef = expElemI.getReferenceField("snapshot");
		
		if (stackTraceRef == MJIEnv.NULL) {
			// It looks NULL constant means reference field not set
			// It look that exception hasn't saved stackTrace   
			// Should not be null. If null occur its unexpected JPF behavior (some change in behavior)
			throw new RuntimeException("Unexpected JPF behavior");
		}	

		MJIEnv env = ei.getThread().getEnv(); /// Used to resolve rest references

		int[] snapshot = env.getIntArrayObject(stackTraceRef);
		if (snapshot == null) {
			// Exception stack trace with no entry .. report error ... unexpected
	    	// It look that exception hasn't saved stackTrace   
			// Should not be null. If null occur its unexpected JPF behavior (some change in behavior)
			throw new RuntimeException("Unexpected JPF behavior");
		}

		int len = snapshot.length/2;
	    if (len == 0) {
			// Exception stack trace with no entry .. report error ... unexpected
	    	// It look that exception hasn't saved stackTrace   
			// Should not be null. If null occur its unexpected JPF behavior (some change in behavior)
			throw new RuntimeException("Unexpected JPF behavior");
	    }
	    
	    // Shit ... StackTraceElement is accessible only in ThreadInfo
	    // Simulate work of StackTrace element ... stay update with them ...     
	    // 		StackTraceElement.StackTraceElement(MethodInfo mi, int pcOffset){
	    MethodInfo mi = MethodInfo.getMethodInfo(snapshot[0]);
	    if (mi == null) {
			// Unknown place where exception occur (where throw) .. report error ... unexpected
	    	// It look that exception hasn't source ... may be return false (can be internal JPF results)   
			// Should not be null. If null occur its unexpected JPF behavior (some change in behavior)
			throw new RuntimeException("Unexpected JPF behavior");
	    }

	    String className = "unknownMethod";
	    
	    if (mi.isDirectCallStub()){
	        // Shit again isReflectionCallStub not public only default :-( (in JPF) ... strange coding level  
	    	// if (mi.isReflectionCallStub()) { 
	            className = "java.lang.reflect.Method";
	        //}
	        } else {
	        	// This should be standard branch ... here obtain class name where throw instruction take place 
	        	className = mi.getClassName();
	            //mthName = mi.getName(); // Inspiration where take method, ... 
	        }
	     
		if (className == null) {
			// Unknown class name where exception occur (where throw) .. report error ... unexpected behavior
	    	// It look that exception hasn't source ... may be return false (can be internal JPF results)   
			// Should not be null. If null occur its unexpected JPF behavior (some change in behavior)
			throw new RuntimeException("Unexpected JPF behavior");
		}
		return className.startsWith(checkedEnviromentClassName);
	}
	
	
	public boolean check (Search search, JVM vm) {
		if (DEBUG) { out.println(PropertyUncaughtExceptions.class.getSimpleName() + ".check( search=" + search + ", JVM=" + vm + ") .. getUncaughtExceptionInfo()=" + getUncaughtExceptionInfo()); }
		return super.check(search, vm);
	}
	
	public void reset() {
		if (DEBUG) { out.println(PropertyUncaughtExceptions.class.getSimpleName() + ".reset()"); }
		super.reset();
	}

	/**
	 * Emulates behavior ExceptionInfo.printOn method.
	 *  
	 * <br>Note: This code should be same as {@link ExceptionInfo#printOn(PrintWriter) method}
	 * <br>Note: This method gets same result as {@link NoUncaughtExceptionsProperty#getErrorMessage()}.
	 * 
	 * <br>Note: We cannot use getErrorMessage() directly because we don't have ExceptionInfo for used previous (cause) exception.
	 *
	 * @param ti ThreadInfo where exception take place
	 * @param excReference Reference to exception you wanted to print stack trace.
	 * @return
	 */
	private static String exceptionInfoPrintOn(ThreadInfo ti, int excReference) { 
	      // Emulation of getUncaughtExceptionInfo.getErrorMessage
		  StringWriter sw = new StringWriter();
	      // emulation of ExceptionInfo.printOn
	      ti.printStackTrace( new PrintWriter(sw), excReference);

	      return sw.toString();
	}
	
	@Override
	public StackTraceElement[] getStackTrace() {
		ExceptionInfo expI = getUncaughtExceptionInfo();
		if (expI != null) {			
			// Check if runtime exception
			if (checkIfCheckerException(expI)) {
				ExceptionInfo ei = getUncaughtExceptionInfo();
				ThreadInfo.StackTraceElement[] internalStackTrace = ei.getThread().getExeptionStackTrace( getOriginalExceptionHeapRef());
				return NoUncaughtExceptionsProperty.convertStackTrace(internalStackTrace);
			}
		}
		return super.getStackTrace();
	}
	
	/**
	 * 
	 * @return Gets reference to heap where Original exception is stored or {@link MJIEnv#NULL} if (not checker exception)
	 */
	protected int getOriginalExceptionHeapRef() {
		ExceptionInfo ei = getUncaughtExceptionInfo();
		if ( !checkIfCheckerException(ei)) {
			return MJIEnv.NULL;
		}
		
		int checkerExpRef = ei.getExceptionReference();
		
		Heap da = JVM.getVM().getHeap();
		assert da != null; // For sure check ... hope that JPF behavior in this way
		ElementInfo expElemI = da.get(checkerExpRef);// RunTime representation of RuntimeException
		if (expElemI == null) {
			// Should not be null. If null occur its unexpected JPF behavior  
			throw new RuntimeException("Problem with JPF exception representation");
		}
		
		int origExcRef = expElemI.getReferenceField("cause");
		if (origExcRef == MJIEnv.NULL) {
			// It looks NULL constant means reference field not set
			//   it's strange ... not runtime exception from generated environment of change of JPF behavior  
			// Should not be null. If null occur its unexpected JPF behavior  
			throw new RuntimeException("Unexpected JPF behavior");
		}
		
		return origExcRef;
	}

}

