//
// Copyright (C) 2006 United States Government as represented by the
// Administrator of the National Aeronautics and Space Administration
// (NASA).  All Rights Reserved.
//
// This software is distributed under the NASA Open Source Agreement
// (NOSA), version 1.3.  The NOSA has been approved by the Open Source
// Initiative.  See the file NOSA-1.3-JPF at the top of the distribution
// directory tree for the complete NOSA document.
//
// THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY OF ANY
// KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY, INCLUDING, BUT NOT
// LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL CONFORM TO
// SPECIFICATIONS, ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR
// A PARTICULAR PURPOSE, OR FREEDOM FROM INFRINGEMENT, ANY WARRANTY THAT
// THE SUBJECT SOFTWARE WILL BE ERROR FREE, OR ANY WARRANTY THAT
// DOCUMENTATION, IF PROVIDED, WILL CONFORM TO THE SUBJECT SOFTWARE.
//
package org.ow2.dsrg.fm.tbpjava.checker;


import java.util.Collections;
import java.util.List;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.Property;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.search.DFSearch;
import gov.nasa.jpf.search.Search;


/**
 * Searcher used for state traversing in checker. Based on {@link DFSearch}.
 * 
 * Extension is made to add state in automatons as a part of program "state". 
 */
public class JPFSearch extends Search {
	private static final boolean DEBUG = false;
	private static final JPFProgramStateMappers psmType = JPFProgramStateMappers.JPF_PSM_TBPProtocolPositions;

	private JPFProgramStateMapping programStateMapping = null; /// Class that maps "JPF program states" with "TBP protocol state" 


	public JPFSearch (Config config, JVM vm) {
		super(config,vm);
	}

	public List<Property>getJPFProperties() {
		return Collections.unmodifiableList(properties);
	}
	/**
	 * Sets event parser, that holds another part of "program state" - position in protocol (automatons)
	 * 
	 * <br>Note: Automaton state can be set only once. Subsequent changes will be ignored. 
	 * 
	 * @param eventParser {@link EventParser} that will be used to resolve if "program state" was fully explored. Cann't be null.
	 */
	public JPFProgramStateMapping getJPFProgramStateMapping() {
		return programStateMapping;
	}

	/**
	 * Has to be called before first usage to finish initialization.
	 * @param varMapper Variable mapper used by the checker
	 */
	public void setStatesMapper(StatesMapper varMapper) {
		if (programStateMapping == null) {
			if (psmType == JPFProgramStateMappers.JPF_PSM_EventList) {
				programStateMapping = new JPFProgramStateMappingEventList();
			} else if (psmType == JPFProgramStateMappers.JPF_PSM_TBPProtocolPositions) {
				programStateMapping = new JPFProgramStateMappingPrecise(varMapper);
			} else if (psmType == JPFProgramStateMappers.JPF_PSM_TBPProtocolPositionsHash) {
				programStateMapping = new JPFProgramStateMappingTBPProtocolPositionsHash(varMapper);
			} else {
				throw new RuntimeException("Unsupported enum " + (psmType!=null?psmType.getClass().getSimpleName():"!null!") + " entry " + psmType);
			}
		} else {
			throw new RuntimeException("Internal error - multiple method call. We expect only the one call before the search class is used.");
		}
	}
	
	public boolean requestBacktrack () {
		doBacktrack = true;

		return true;
	}


	public void search () {
		int maxDepth = getMaxSearchDepth();
		boolean depthLimitReached = false;

		depth = 0;

		notifySearchStarted();

		while (!done) {
			
			if (DEBUG) { System.out.println( JPFSearch.class.getSimpleName() + ".search() ... stateId = " + vm.getSystemState().getId() + ", isNewState = " + isNewState() + ", programStateMapping.wasProcessed(" +  vm.getSystemState().getId() + ") = " + programStateMapping.wasProcessed(vm.getSystemState().getId()) + ", isEndState = " + isEndState()); }

			int programStateID = vm.getSystemState().getId(); /// Represents run-time state of program (heap, stack, threads)
			
			// programStateMapping.wasProcessed(programStateID)) checks 
			//   if current state in "TBP protocol" was previously processed from same programStateID  
			if ( (!isNewState() && programStateMapping.wasProcessed(programStateID)) || isEndState() || isIgnoredState() || depthLimitReached) {
				if (!backtrack()) { // backtrack not possible, done
					break;
				}

				depthLimitReached = false;
				depth--;
				notifyStateBacktracked();
			}

			if (forward()) {
				depth++;
				notifyStateAdvanced();

				if (currentError != null) {
					notifyPropertyViolated();
				}

				if (hasPropertyTermination()) {
					break;
				}

				if (depth >= maxDepth) {
					depthLimitReached = true;
					notifySearchConstraintHit(DEPTH_CONSTRAINT + ": " + maxDepth);
					continue;
				}

				if (!checkStateSpaceLimit()) {
					notifySearchConstraintHit(FREE_MEMORY_CONSTRAINT + ": " + minFreeMemory);
					// can't go on, we exhausted our memory
					break;
				}
			} else { // state was processed
				notifyStateProcessed();
			}
		}

		notifySearchFinished();
	}


	public boolean supportsBacktrack () {
		return true;
	}
	
	private enum JPFProgramStateMappers {
		JPF_PSM_TBPProtocolPositions,
		JPF_PSM_TBPProtocolPositionsHash,
		JPF_PSM_EventList
	}
	
	static public String getStaticConfiguration() {
		if (psmType == JPFProgramStateMappers.JPF_PSM_EventList) {
			return "Used program state mapper " + JPFProgramStateMappingEventList.class.getSimpleName();
		} else if (psmType == JPFProgramStateMappers.JPF_PSM_TBPProtocolPositions) {
			return "Used program state mapper " + JPFProgramStateMappingPrecise.class.getSimpleName();
		} else if (psmType == JPFProgramStateMappers.JPF_PSM_TBPProtocolPositionsHash) {
			return "Used program state mapper " + JPFProgramStateMappingTBPProtocolPositionsHash.class.getSimpleName();
		} else {
			throw new RuntimeException("Unsupported enum " + (psmType!=null?psmType.getClass().getSimpleName():"!null!") + " entry " + psmType);
		}
	}
}
