package org.ow2.dsrg.fm.tbpjava.checker;

import gov.nasa.jpf.util.Pair;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import org.ow2.dsrg.fm.tbpjava.checker.StatesMapperLayer.MappedStateHashableMemento;


/**
 * Maps JPF states to TBP protocol positions. This mapping is used to extend the JPF state.
 * 
 * @author Alfifi
 *
 */
final public class JPFProgramStateMappingPrecise implements JPFProgramStateMapping {
	private static final boolean DEBUG = false;
	private static final boolean DEBUG_INFO = DEBUG;

	private static final boolean INTERNLAL_TESTS = false;
	
	private final PrintStream out = System.out;
	
	private final StatesMapper varMapper;
	private final StateMementoToIndexTransformer indexer;
	
	private static final int INVALID_INDEX = -1;

	private static final int INITIAL_SIZE = 16 * 1024; /// Initial size of arrays
	
	/**
	 * For each JPF state id (array index) contains index 
	 *   in the {@link #tbpStateList} where starts linked list of 
	 *   the TBP state id or {@link #INVALID_INDEX} if no TBP state
	 *   is not mapped to given JPF state.
	 * 
	 * <br>Note: the index has to be multiplied be 2 to obtain index into
	 *  {@link #tbpStateList}.
	 */
	private int[] jpfState2listIndex = new int[INITIAL_SIZE]; 
	
	/**
	 * Logical linked list of pairs 
	 *     (index of the TBP state [even indexes],
	 *      index of next pair in list or {@link #INVALID_INDEX} 
	 *       in the end of the list [odd indexes]).
	 */
	private int[] tbpStateList = new int[2 * INITIAL_SIZE];
	
	/** Last used index in the array */
	private int tbpStateListEnd = -1;
	
	
	private final Stack< Pair<Integer,Integer>> stack = new Stack<Pair<Integer,Integer>>();

	public JPFProgramStateMappingPrecise(StatesMapper varMapper) {
		assert varMapper != null;
		if (DEBUG) { out.println(this.getClass().getSimpleName() + "." + this.getClass().getSimpleName() + "()"); }
		
		this.varMapper = varMapper;
		this.indexer = new StateMementoToIndexTransformer();
		// Initialize array
		Arrays.fill(jpfState2listIndex, 0, jpfState2listIndex.length, INVALID_INDEX);
	}

	@Override
	public void addMapping(int stateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ")"); } 
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ") ... MappedStates before call\n"); DEBUG_printMapping(); }

		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }

		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition();
		int currentTBPPositionIndex = indexer.getMementoIndex(currentTBPPosition);

		// Found index  in "JPF -> start index in linked list" mapping
		// Ensure that jpfState2listIndex is long enough
		if (stateID >= jpfState2listIndex.length) {
			// New state, we has to enlarge the array
			int newSize = jpfState2listIndex.length * 2;
			while (newSize <= stateID) {
				newSize *= 2;
			}
			int origLen = jpfState2listIndex.length;
			jpfState2listIndex = Arrays.copyOf(jpfState2listIndex, newSize);
			Arrays.fill(jpfState2listIndex, origLen, newSize-1, INVALID_INDEX);
		}
		
		if (jpfState2listIndex[stateID] == INVALID_INDEX) {
			// We need to add first item into the linked list 

			// Check if there is empty position in linked list
			if (tbpStateList.length <= tbpStateListEnd + 2) {
				tbpStateList = Arrays.copyOf(tbpStateList, tbpStateList.length * 2);
			}
			jpfState2listIndex[stateID] = ++tbpStateListEnd;
			tbpStateList[tbpStateListEnd] = stateID;
			tbpStateListEnd++;
			tbpStateList[tbpStateListEnd] = INVALID_INDEX;
			
		} else {
			// Check if not in stored linked list (add into the end of the list)
			int currIndex = jpfState2listIndex[stateID];
			int prevIndex = INVALID_INDEX;
			while (currIndex != INVALID_INDEX) {
				if (tbpStateList[currIndex] == currentTBPPositionIndex) {
					break;
				}
				prevIndex = currIndex;
				currIndex = tbpStateList[currIndex+1];
			}
			
			if (currIndex == INVALID_INDEX) {
				// Append into the end of the list
				assert prevIndex != INVALID_INDEX;
				
				// Check if there is empty position in linked list
				if (tbpStateList.length <= tbpStateListEnd + 2) {
					tbpStateList = Arrays.copyOf(tbpStateList, tbpStateList.length * 2);
				}
				tbpStateListEnd++;
				tbpStateList[tbpStateListEnd] = stateID;
				tbpStateListEnd++;
				tbpStateList[tbpStateListEnd] = INVALID_INDEX;
			
				tbpStateList[prevIndex+1] = tbpStateListEnd-1;
			}
		}
		
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ")"); } 
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ") ... MappedStates after call\n"); DEBUG_printMapping(); }

		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }

		
	}

	@Override
	public boolean wasProcessed(int stateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ")"); } 
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... MappedStates after call\n"); DEBUG_printMapping(); }

		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }

		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition();
		int currentTBPPositionIndex = indexer.getMementoIndex(currentTBPPosition);
		
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... currentTBPPositionIndex=" + currentTBPPositionIndex + ", currentcurrentTBPPosition=" + currentTBPPosition); }
		
		// Logical "or operation" of the findMap and findStack with short evaluation
		boolean findMap = isStoredInTBPStateList(stateID, currentTBPPositionIndex);
		if (findMap == true) {
			if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... return true after findMap"); }
			return true;
		}
		boolean findStack = isStoredInTheStack(stateID, currentTBPPositionIndex);
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... findMap=" + findMap + ", findStack=" + findStack); }
		 
		return findStack; 
	}

	private boolean isStoredInTheStack(int stateID, int currentTBPPositionIndex) {
		Iterator<Pair<Integer, Integer>> it = stack.iterator();
		for(int i = 0; i < stack.size() - 1; i++) {
			Pair<Integer, Integer> entry = it.next();
			if (entry.a == stateID && entry.b == currentTBPPositionIndex) {
				return true;
			}
		}
		return false;
	}
	private boolean isStoredInTBPStateList(int stateID, int currentTBPPositionIndex) {
		assert stateID >= 0;
		assert currentTBPPositionIndex >= 0;
		
		if (stateID >= jpfState2listIndex.length) {
			return false;
		}
		
		int listIndex = jpfState2listIndex[stateID];
		if (listIndex == INVALID_INDEX) {
			return false;
		}
		
		do {
			// Is correct mapping?
			if (tbpStateList[listIndex] == stateID) {
				return true;
			}
			// Move to next element
			listIndex = tbpStateList[listIndex+1];
		} while(listIndex != INVALID_INDEX);
		
		return false;
	}

	@Override
	public void advancedEvents(List<EventItem> events, int jpfStateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".advancedEvents(events=" + events + ",jpfStateID=" + jpfStateID + ")"); }
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".advanceEvents() ... MappedStates before call\n"); DEBUG_printMapping(); }
		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }

		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition();
		int currentTBPPositionIndex = indexer.getMementoIndex(currentTBPPosition);
		
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".advancedEvents() ... currentTBPPositionIndex=" + currentTBPPositionIndex + ", currentTBPPosition=" + currentTBPPosition); }
		stack.push( new Pair<Integer, Integer>(jpfStateID, currentTBPPositionIndex));

		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".advancedEvents()  ... MappedStates after call\n"); DEBUG_printMapping(); }
		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }
	}

	@Override
	public void undoEvents() {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".undoEvents()"); }
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".undoEvents()  ... MappedStates before call\n"); DEBUG_printMapping(); }
		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }
		stack.pop();
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".undoEvents()  ... MappedStates after call\n"); DEBUG_printMapping(); }
		if (INTERNLAL_TESTS) { DEBUG_checkStructure(); }
	}



	@Override
	public String getUsageStatistics() {
		StringBuffer result = new StringBuffer();
		result.append(indexer.reportHashCollisions());

		int[] lengthListCounts = new int[1024];
		Arrays.fill(lengthListCounts, 0, lengthListCounts.length, 0);
		
		for(int i = 0; i < jpfState2listIndex.length; i++) {
			if (jpfState2listIndex[i] == INVALID_INDEX) {
				continue; // No list to process
			}

			int listLen = 0;
			int listIndex = jpfState2listIndex[i];
			
			do {
				listLen++;
				listIndex = tbpStateList[listIndex+1];
			} while (listIndex != INVALID_INDEX);
			
			// Enlarge the length counter
			if (listLen > lengthListCounts.length) {
				int[] newLengthListCounts = Arrays.copyOf(lengthListCounts, listLen * 2);
				Arrays.fill(newLengthListCounts, lengthListCounts.length, newLengthListCounts.length-1, 0);
				lengthListCounts = newLengthListCounts;
			}
			lengthListCounts[listLen]++;
		}
		
		result.append("Dumping histogram of Number of the TBPStatem mapped to JPFState -> Number of occurences\n");
		for(int i = 0; i < lengthListCounts.length; i++) {
			if (lengthListCounts[i] == 0) {
				continue;
			}
			result.append("\t" + i + " ->" + lengthListCounts[i] + '\n');
		}

		result.append("Cross test Extended JPF states : " + (tbpStateListEnd + 1) / 2);
		return result.toString();
	}

	public void DEBUG_checkStructure() {
		
		// Last used index is valid
		if (tbpStateListEnd >= tbpStateList.length) {
			throw new RuntimeException("tbpStateListEnd is too BIG");
		}
		
		//JPF protocol indexes are valid
		for(int i = 0; i < jpfState2listIndex.length; i++) {
			if (jpfState2listIndex[i] == INVALID_INDEX) {
				continue; // Valid value
			}
			if (! (jpfState2listIndex[i] >= 0 && jpfState2listIndex[i] < tbpStateListEnd)) {
				throw new RuntimeException("jpfState2listIndex invalid value ...jpfState2listIndex[" + i + "]=" + jpfState2listIndex[i]);
			}
			
			// Even index
			if (!((jpfState2listIndex[i] & 1) == jpfState2listIndex[i])) {
				throw new RuntimeException("Index points into the second pard of the pair - odd asress  ...jpfState2listIndex[" + i + "]=" + jpfState2listIndex[i]);
			}
		}
		
		// Check if all entries are in tbpStateList are reachable 
		Set<Integer> visitedIndexes = new HashSet<Integer>(tbpStateListEnd);
		Set<Integer> mementoIndexesInList = new HashSet<Integer>();

		for(int i = 0; i < jpfState2listIndex.length; i++) {
			if (jpfState2listIndex[i] == INVALID_INDEX) {
				continue; // No list to process
			}

			int listIndex = jpfState2listIndex[i];
			
			do {
				if (visitedIndexes.contains(listIndex)) {
					throw new RuntimeException("multiple entries in jpfState2listIndex points into same linked list jpfState2listIndex[" + i + "]=" + jpfState2listIndex[i]);
				}
				visitedIndexes.add(listIndex);

				if (mementoIndexesInList.contains( tbpStateList[listIndex])) {
					throw new RuntimeException("same memento ID entries in one linked list jpfState2listIndex[" + i + "]=" + jpfState2listIndex[i] + " ... tbpStateList[" + listIndex + "]=" + tbpStateList[listIndex]);
				}
				mementoIndexesInList.add(tbpStateList[listIndex]);

				// Move to next element
				listIndex = tbpStateList[listIndex+1];
				
			} while(listIndex != INVALID_INDEX);
			
			mementoIndexesInList.clear();
		}
		
		if ((2*visitedIndexes.size()-1) != tbpStateListEnd) {
			throw new RuntimeException("Unexpected total size of linked lists."); // Probably there are unreachable nodes in e
		}
	}

	public void DEBUG_printMapping() {
		out.println(this.getClass().getSimpleName() + "DEBUG_printMapping - stored mapping");
		// Output format JPF state id -> list of mapped TBP protocol position IDa
		for(int i = 0; i < jpfState2listIndex.length; i++) {
			out.print("\t - " + i + " --> ");
			int listIdx = jpfState2listIndex[i];
			if (listIdx == INVALID_INDEX) {
				out.println("no mapped state");
			} else {
				out.print(tbpStateList[listIdx]);
				listIdx = tbpStateList[listIdx + 1];
				while (listIdx != INVALID_INDEX) {
					out.print(", " + tbpStateList[listIdx]);
					listIdx = tbpStateList[listIdx + 1];
				}
				out.println();
			}
		}
		out.println(this.getClass().getSimpleName() + "DEBUG_printMapping - dumping actual stack [Depth -> jpfStateID,TBPProtocolStateID]");
		for(int i = 0; i < stack.size(); i++) {
			Pair<Integer, Integer> pair = stack.get(i);
			out.print('[' + i + " --> " + pair.a + ", " + pair.b + ']');
		}
		out.println();
	}
}

