package org.ow2.dsrg.fm.tbpjava.checker;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.checker.EventItem.EventTypes;
import org.ow2.dsrg.fm.tbpjava.checker.JPFProgramStateMappingEventList.SetTBPProtocolEvent.TBPProtocolEventList;
import org.ow2.dsrg.fm.tbpjava.checker.StatesMapperLayer.MappedStateHashableMemento;
import org.ow2.dsrg.fm.tbpjava.utils.Pair;


/** 
 * Maps "program state" in JPF to states in TBP protocol.
 * Makes illusion that "program state" is extended by "TBP protocol state" for {@link JPFSearch}. 
 * ({@link JPFSearch} behaves in that way).
 * Is closely connected with {@link JPFSearch}.
 * 
 * @author Alfifi
 */
public interface JPFProgramStateMapping {
	/** Associate to the given JPF state current checker state (TBP positions)
	 * @param stateID The JPF state 
	 */
	public void addMapping(int stateID);

	/**
	 * Tests whether current checker state (TBPPositions) is associated to given JPF state.
	 * @param stateID The JPF state identification
	 * @return True if current checker state (TBPPositions) is associated to given JPF state, false otherwise.
	 */
	public boolean wasProcessed(int stateID);

	/**
	 * Notify the mapper about JPF "advancedState" where some event take place.
	 * It's important for the mapper to maintain stack with current events and JPF state to detect cycles.
	 * Method is called AFTER the checker has processed given events, but BEFORE the SEARCH decide next (forward/backward) step.
	 *
	 * @param events Events that take place in last transition 
	 * @param jpfStateID JPF state after transition end.
	 */
	public void advancedEvents(List<EventItem> events, int jpfStateID);

	public void undoEvents();
	
	public String getUsageStatistics();
}


final class JPFProgramStateMappingTBPProtocolPositionsHash implements JPFProgramStateMapping {
	private static final boolean DEBUG = false;
	private static final boolean DEBUG_INFO = DEBUG;
	
	private static final boolean INTERNLAL_TESTS = false;
	
	private final PrintStream out = System.out;
	
	private final StatesMapper varMapper;
	private Map<Integer, Set<Integer>> mapJPFState2TBPProtocolStateHash; /// Contains only states fully processed by the JPF. ()
	
	private Stack< Pair<Integer,MappedStateHashableMemento>> stack = new Stack<Pair<Integer,MappedStateHashableMemento>>();
	
	public JPFProgramStateMappingTBPProtocolPositionsHash(StatesMapper varMapper) {
		assert varMapper != null;
		if (DEBUG) { out.println(this.getClass().getSimpleName() + "." + this.getClass().getSimpleName() + "()"); }

		this.varMapper = varMapper;
		this.mapJPFState2TBPProtocolStateHash = new HashMap<Integer, Set<Integer>>();
	}
	@Override
	public void addMapping(int stateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ")"); } 
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ") ... MappedStates before call\n"); out.append(toString_DEBUG_INFO(2, stateID)); }

		Set<Integer> tbpPositionsSet = mapJPFState2TBPProtocolStateHash.get(stateID);
		if (tbpPositionsSet == null) {
			tbpPositionsSet = new HashSet<Integer>();
			mapJPFState2TBPProtocolStateHash.put(stateID, tbpPositionsSet);
		}
		
		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition(); // Current position from obtained form the stack
		if (INTERNLAL_TESTS) {
			// Generated new actual (not cached in the stack) TBP position and compare with the used one
			if (!currentTBPPosition.equals(varMapper.getTBPPosition())) {
				throw new RuntimeException("Internal error - unsynchronized stack");
			}
		}
		
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ") ... currentTBPPosition=" + currentTBPPosition); }
		tbpPositionsSet.add(currentTBPPosition.hashCode());

		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".addMapping(stateID=" + stateID + ")  ... MappedStates after call\n"); out.append(toString_DEBUG_INFO(2, stateID)); }
	}

	@Override
	public void advancedEvents(List<EventItem> events, int jpfStateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".advancedEvents(events=" + events + ",jpfStateID=" + jpfStateID + ")"); }
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".advancedEvents()  ... MappedStates before call\n"); out.append(toString_DEBUG_INFO(2, -1)); }

		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition(); 
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".advancedEvents() ... currentTBPPosition=" + currentTBPPosition); }
		stack.push( new Pair<Integer, MappedStateHashableMemento>(jpfStateID, currentTBPPosition));

		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".advancedEvents()  ... MappedStates after call\n"); out.append(toString_DEBUG_INFO(2, -1)); }
	}

	@Override
	public void undoEvents() {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".undoEvents()"); }
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".undoEvents()  ... MappedStates before call\n"); out.append(toString_DEBUG_INFO(2, -1)); }
		stack.pop();
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".undoEvents()  ... MappedStates after call\n"); out.append(toString_DEBUG_INFO(2, -1)); }
	}

	@Override
	public boolean wasProcessed(int stateID) {
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ")"); }
		if (DEBUG_INFO) { out.print(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ")  ... MappedStates before call\n"); out.append(toString_DEBUG_INFO(2, stateID)); }

		MappedStateHashableMemento currentTBPPosition = varMapper.getTBPPosition(); 
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... currentTBPPosition=" + currentTBPPosition); }
		
		// Logical "or operation" of the findMap and FindStack with short evaluation
		boolean findMap = isStoredInProcessedStateMapHash(stateID, currentTBPPosition);
		if (findMap == true) {
			if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... return true after findMap"); }
			return true;
		}
		boolean findStack = isStoredInTheStack(stateID, currentTBPPosition);
		if (DEBUG) { out.println(this.getClass().getSimpleName() + ".wasProcessed(stateID=" + stateID + ") ... findMap=" + findMap + ", findStack=" + findStack); }
		 
		return findStack; 
	}
	
	private boolean isStoredInTheStack(int stateID, MappedStateHashableMemento currentTBPPosition) {
		Iterator<Pair<Integer, MappedStateHashableMemento>> it = stack.iterator();
		for(int i = 0; i < stack.size() - 1; i++) {
			Pair<Integer, MappedStateHashableMemento> entry = it.next();
			if (entry.getFirst() == stateID && entry.getSecond().equals(currentTBPPosition)) {
				return true;
			}
		}
		return false;
		//return stack.contains( new Pair<Integer, MappedStateHashableMemento>(stateID, currentTBPPosition));
	}
	/// Whether is mapped in the mapJPFState2TBPProtocolState
	private boolean isStoredInProcessedStateMapHash(int stateID, MappedStateHashableMemento currentTBPPosition) {
		Set<Integer> tbpPositionsSet = mapJPFState2TBPProtocolStateHash.get(stateID);
		if (tbpPositionsSet == null) {
			return false;
		}
		return tbpPositionsSet.contains(currentTBPPosition.hashCode()); 
	}
	
	private void DEBUG_TBPPositionSet(int indent, int stateID, StringBuffer sb) {
		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("stateID ->" + stateID);
		Set<Integer> tbpPostionsSet = mapJPFState2TBPProtocolStateHash.get(stateID);
		if (tbpPostionsSet == null) {
			sb.append("... no mappings ... null\n");
		} else {
			sb.append('\n');
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent + 2);
			boolean first = true;
			for(Integer tbpPosition : tbpPostionsSet) {
				if (!first) {
					sb.append(", ");
				}
				sb.append(tbpPosition);
				
			}
		}
	}
	
	private StringBuffer toString_DEBUG_INFO(int indent, int highlightedJPFState) {
		StringBuffer sb = new StringBuffer();
		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("Debugging Internal state of the " + this.getClass().getSimpleName() + "\n");
		indent += 2;
		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("mapJPFState2TBPProtocolState=\n");
		for(Integer jpfState : mapJPFState2TBPProtocolStateHash.keySet())
			DEBUG_TBPPositionSet((jpfState==highlightedJPFState?indent+2:indent+6), jpfState, sb);
		ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
		sb.append("stack=\n");
		for(int i = 0; i < stack.size(); i++) {
			ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
			sb.append(i);
			Pair<Integer, MappedStateHashableMemento> pair = stack.get(i);
			sb.append(" -> jpfStateID=");
			sb.append(pair.getFirst());
			sb.append(", TBPPosition=\n");
			sb.append(pair.getSecond().toString(indent+2));
		}
		return sb;
	}
	
	@Override
	public String getUsageStatistics() {
		String result = "";
		int extendedStated = 0;
		for(Entry<Integer, Set<Integer>> entryMappedPos : mapJPFState2TBPProtocolStateHash.entrySet()) {
			if (entryMappedPos.getValue() == null) {
				continue;
			}
			extendedStated += entryMappedPos.getValue().size();
		}
		extendedStated +=stack.size();
		
		result += "Extended JPF states : " + extendedStated;
		return result;
	}
}


class JPFProgramStateMappingEventList implements JPFProgramStateMapping {
	private static final boolean DEBUG = false;
	private static final boolean DEBUG_INFO = false;
	
	private TBPProtocolEventList eventListBottom; /// Special item that represents protocol state where no event occure.
	
	private Map<Integer, SetTBPProtocolEvent> mapJPFState2TBPProtocolState;
	

	private Stack<TBPProtocolEventList> eventStack; /// Has one additional bottom state {@link #eventListBottom}.
	
	
	public JPFProgramStateMappingEventList() {
		eventStack = new Stack<TBPProtocolEventList>();
		mapJPFState2TBPProtocolState = new HashMap<Integer, SetTBPProtocolEvent>();
		
		// Prepare eventListBottom
		EventItem ei = new EventItem(-1, EventTypes.EVENT_CALL, "noMethodCalled", "noInterface", -1);
		eventListBottom = new TBPProtocolEventList(null, ei);
		
		eventStack.push(eventListBottom);
	}
	
	//Adds mapping of JPF state with given stateID to current TBP protocol state
	public void addMapping(int stateID) {
		if (DEBUG) { System.out.println(JPFProgramStateMappingEventList.class.getSimpleName() + ".addMapping( stateID=" + stateID + ")"); }
		if (DEBUG_INFO) {System.out.println("CurrentState\n" + eventStack.peek().toString(4)); }
		SetTBPProtocolEvent holder;
		
		// Find holder that represents associated "TBP states"
		holder = mapJPFState2TBPProtocolState.get(stateID);

		// We need to add empty states to have correct run-time statistics
		if (holder == null) {
			holder = new SetTBPProtocolEvent();
			mapJPFState2TBPProtocolState.put(stateID, holder);
		}
		
		holder.addToProcessedList(eventStack.peek());
		
	}
	
	public void advancedEvents(List<EventItem> events, int jpfStateID) {
		if (DEBUG) { System.out.println(JPFProgramStateMappingEventList.class.getSimpleName() + ".processEvents( events=" + events + ")"); }
		if (DEBUG_INFO) {System.out.println("CurrentState before\n" + eventStack.peek().toString(4)); }
		TBPProtocolEventList currentEventList = eventStack.peek();	
	
		if (events != null) {
			// Update eventIndexList according newly explored events. (Add events into start of the list.)
			for(int eventIndex = 0; eventIndex < events.size(); eventIndex++) {
				currentEventList = new TBPProtocolEventList(currentEventList, events.get(eventIndex));
			}
		}
		
		eventStack.push(currentEventList);
		if (DEBUG_INFO) {System.out.println("CurrentState after\n" + eventStack.peek().toString(4)); }
		
	}
	
	public void undoEvents() {
		if (DEBUG) { System.out.println(JPFProgramStateMappingEventList.class.getSimpleName() + ".undoEvents()"); }
		if (DEBUG_INFO) {System.out.println("CurrentState before\n" + eventStack.peek().toString(4)); }
		if (eventStack.size() == 1) {
			throw new RuntimeException("Too many undo events - not complemented processEvent found");
		}
		
		eventStack.pop();
		if (DEBUG_INFO) {System.out.println("CurrentState after\n" + eventStack.peek().toString(4)); }
		
	}
	
	public boolean wasProcessed(int stateID) {
		if (DEBUG) { System.out.println(JPFProgramStateMappingEventList.class.getSimpleName() + ".wasProcessed(stateID=" + stateID + ")\n" + eventStack.peek().toString(4)); }
		SetTBPProtocolEvent holder;
		
		// Find holder that represents associated "TBP states"
		holder = mapJPFState2TBPProtocolState.get(stateID);

		// We need to add empty states to have correct run-time statistics
		if (holder == null) {
			holder = new SetTBPProtocolEvent();
			mapJPFState2TBPProtocolState.put(stateID, holder);
		}
	
		return holder.sameState(eventStack.peek());
	}
	
	@Override
	public String getUsageStatistics() {
		// No statistics are computed
		return "";
	}

	
	/**
	 * Each instance of this object is connected with one JPF state 
	 *   (see {@link JPFProgramStateMapping#mapJPFState2TBPProtocolState}).
	 * <br>  
	 * Holds set of {@link TBPProtocolEventList} (it means "TBP protocol states")  that was processed
	 *  from associated JPF state.
	 * <br>
	 * During JPF state space traversing JPF visits many states of program,
	 *    for each state exists some state in TBP Protocol (
	 *    here represented by list of events that occur).
	 * This objects holds such "TBP Protocol" states for one "JPF state".
	 * 
	 * <br>If JPF visits some state again, we need to check if we are again in same "TBP Protocol" state.
	 * You can imagine that we tries to extend JPF state by "TBP Protocol" state. 
	 * (It means JPF can backtrack if it's not only in same JPF state but in "TBP Protocol" state too.)
	 *  
	 *<br>Note: Calculates some run-time statistics.
	 *
	 * @author Alfifi
	 *
	 */
	static class SetTBPProtocolEvent {

		private Set<TBPProtocolEventList> eventSet = new HashSet<TBPProtocolEventList>();

		// Run time statistics 
		private int DEBUG_sameStateCalls = 0;
		private int DEBUG_sameStateSuccess = 0;
		private int DEBUG_sameStateFail = 0;
		private int DEBUG_List_sameStateCalls = 0; /// Number of calls TBPProtocolEventList#sameState on represented states 
		private int DEBUG_List_sameStateDepth = 0; // Sum of all TBPProtocolEventList#sameState test depths. sum(abs(DEBUG_List_sameStateCalls))
		
		/**
		 * Adds new {@link TBPProtocolEventList} into set of associated "TBP event list".
		 * @param newList List to add.
		 */
		public void addToProcessedList(TBPProtocolEventList newList) {
			eventSet.add(newList);
		}
		
		/**
		 * Checks if same states as given exists in processed states (in represented set).
		 * @param other {@link TBPProtocolEventList} to test.
		 * @return True if same (it means 
		 *   {@link TBPProtocolEventList#sameState(TBPProtocolEventList)} gets positive number)
		 *   state is stored in represented set, false otherwise. 
		 */
		public boolean sameState(TBPProtocolEventList other) {
			// Fast primary test
			DEBUG_sameStateCalls++;
			if (eventSet.contains(other)) {
				DEBUG_sameStateSuccess++;
				return true;
			}
			
			for(TBPProtocolEventList event :  eventSet) {
				int areSame = event.sameState(other);
				DEBUG_List_sameStateCalls++;
				
				if (areSame > 0) {
					DEBUG_List_sameStateDepth += areSame;
					DEBUG_sameStateSuccess++;
					return true;
				}
				
				DEBUG_List_sameStateDepth -= areSame; // Adding negative value
			}
			DEBUG_sameStateFail++;
			return false;
		}
		
		static class TBPProtocolEventList {
			private static final int STATES_NOT_SAME = -1; /// Represents return value 
			private static final int STATES_SAME = 1; /// Represents return value
		
			private final TBPProtocolEventList previous;
			private final EventItem eventItem;
			
			
			public TBPProtocolEventList(TBPProtocolEventList previous, EventItem eventItem) {
				assert eventItem != null;
				
				this.previous = previous;
				this.eventItem = eventItem;
			}
			
			
			/** 
			 * Check whether given {@link TBPProtocolEventList} represents same events
			 *  as current instance.
			 * @param other {@link TBPProtocolEventList} to check.
			 * @return Return positive number if (it's sure) that given {@link TBPProtocolEventList}
			 *    represents same TBP protocol state as current instance (exactly same events occur),
			 *    otherwise return false. {@link Math#abs(int)} from return value marks how deep it was 
			 *    necessary to go in linked list to find result. 
			 */
			public int sameState(TBPProtocolEventList other) {
				if (other == null) {
					return STATES_NOT_SAME;
				} else if (other == this) {
					return STATES_SAME;
				} else {
					// Different instances
					if (eventItem.equals(other.eventItem)) {
						// EventItems are same ... same call in code ... test previous calls if are same too
						if (previous == null) {
							if (other.previous == null) {
								// Both roots
								return STATES_SAME;
							} else {
								// previous != null. Different depth. Cann't represent same TBPPRocotol states, different event come
								return STATES_NOT_SAME;
							}
						}
						// previous != null;
						int previousSameState = previous.sameState(other.previous);
		
						// Change "abs" value by one. Sign remains same. ==> Test depth plus one. 
						if (previousSameState >= 0) {
							
							return previousSameState + 1;
						} else {
							return previousSameState -1;
						}
					} else {
						// Represents different calls in code
						return STATES_NOT_SAME;
					}
				}
			}
			
			/** 
			 * Converts whole list into string representation. Each list entry is on new line. 
			 * @param indent Indentation on each line start.
			 * @return Text representation of list.
			 */
			public String toString(int indent) {
				StringBuffer sb = new StringBuffer(); 
				ThreadAutomatons.DEBUG_indetStringBuffer(sb, indent);
				sb.append( eventItem.toString());
				sb.append('\n');
				if (previous != null) {
					sb.append( previous.toString(indent)); 
				}
				return sb.toString();
			}
			
			
			public String toString() {
				return this.toString(0);
			}
			
		}
	}
}