package org.ow2.dsrg.fm.tbpjava.checker;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.EnvGenerator.EnvironmentDescription;
import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;

/**
 * Process events given from {@link ProtocolListener}. 
 *   Main task is to separate events by thread numbers, type of interface. 
 *   It takes care about backtracking during JPF state traversal.
 *   And check it threads ends if protocol permit end.
 *   <br>
 * Processed events forward to specific {@link ThreadAutomatons} objects. 
 */
public class EventParser {
	private static final boolean DEBUG = false; // Set to true if you want to see debug prints from this object
	private static final boolean DEBUG_INFO = false; // Enable stack state debug prints. (Prints state saved in stack)
	
	private Configuration config; /// Configuration of Checker
	private LTSAComponentSpecification spec; /// Parsed specification of processed component
	private StatesMapper varStateMap; /// Maps automaton states into sets of associates variables arrays

	private Stack< Map<Integer, ThreadAutomatons>> state; /// Stack that store state after each JPF step to support backtracking. Stored map is used to separate events processing objects by thread number.

	public static final int NO_ERROR = -1; /// Special value in {@link #processEventsIndexErrorEvent} that marks that there was no error event processing in {@link #processEvents(List)}
	private int processEventsIndexErrorEvent = NO_ERROR; /// Specify index of event in {@link #processEvents(List)} parameter that caused error.
	
	private EnvironmentDescription envDesc; /// Description of current environment.

	public EventParser(Configuration config, LTSAComponentSpecification spec, StatesMapper varStateMap, EnvironmentDescription envDesc) {
		this.config = config;
		this.spec = spec;
		this.varStateMap = varStateMap;
		this.state = new Stack< Map<Integer, ThreadAutomatons>>();
		this.envDesc = envDesc;
	}
	
	/**
	 * Initialize event parser for a new checker run. It's necessary to call this method before event processing starts.
	 * It's expected to be called from {@link ProtocolListener#searchStarted(gov.nasa.jpf.search.Search)}
	 * 
	 *  @param jpfProgramState2TBPProtocolStateMapping New {@link JPFProgramStateMapping} that maps program states to TBP protocol states and where events (forward/backtrack) are passed on too.
	 */
	public void initializeEventParser() {
		this.state.clear();
		// Initialization - first 0 main thread
		Map<Integer, ThreadAutomatons.ThreadType> initialMainThread = new HashMap<Integer, ThreadAutomatons.ThreadType>(2);
		initialMainThread.put(0, ThreadAutomatons.ThreadType.ENVIRONMENT_THREAD);
		processEvents(initialMainThread, null, null); /// Bottom layer is an helper layer where initialization of main thread takes place 
		
	}
	
	// These properties are used during {@link #processEvents} call. We want to remove needless allocations. 
	private Set<Integer> threadNums = new HashSet<Integer>();  /// During {@link #processEvents} call holds thread that make any call or return.  
	private int processEventsCallCount = 0; /// Count {@link #processEvents} call to recognize if instance of ThreadAutomaton has been created during processing of this call

	
	/**
	 * Processes interesting actions that become during last step of JPF.
	 * It's sequentially calls {@link #processThreadStarts(Map)}, {@link #processEvents(List) and {@link #processThreadEnds(List)} 
	 *  if any event of given type exists. See referenced methods for more details about threir actions.
	 *  
	 * @param startedThreads Map with created threads and places where creation takes place.
	 * @param events Call and return events to process
	 * @param endedThreads List of thread that ended during last JPF step.
	 * @return True if no error occurred during processing, false if error found
	 */
	public boolean processEvents(Map<Integer, ThreadAutomatons.ThreadType> startedThreads, List<EventItem> events, List<Integer> endedThreads) {
		if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":processEvents( startedThreads=" + startedThreads + ", events=" + events + ", endedThreads=" + endedThreads + ")");  }

		// Initialization - preparation for new step - creating new clone of previous state 
		processEventsCallCount++;
		processEventsIndexErrorEvent = NO_ERROR;
		if (DEBUG) { System.out.println("  processEventsCallCount=" + processEventsCallCount); }

		boolean result = true; // error flag ... marks if error present
		
		// Make working copy of variables (evaluations)
		varStateMap.newLayer();

		// Pick all thread numbers
		threadNums.clear(); 
		if (events != null) {
			for(EventItem event : events) {
				threadNums.add(event.thread);
			}
		}

		// Add new level to state stack .. to support undo
		createNewStateLevel(threadNums);
		if (DEBUG_INFO) { DEBUG_printState(); }


		// Processing events from program
		if (startedThreads != null) {
			result &= processThreadStarts(startedThreads);
		}
		
		if (events != null) {
			result &= processEvents(events);
		}
		
		if (endedThreads != null) {
			result &= processThreadEnds(endedThreads);
		}

		return result;
	}
	
	/**
	 * Process given events one by one by redirecting them into proper 
	 *   {@link ThreadAutomatons} where movements in automatons take place.
	 * 
	 * @param events Call and return events to process
	 * @return True if no error occurred during processing, false if error found
	 */
	private boolean processEvents(List<EventItem> events) {
		if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":processEvents"); DEBUG_printEvents(events, 2); }
		if (DEBUG_INFO) { DEBUG_printState(); }

		boolean result = true; // error flag ... marks if error present
				
		// Gets current mapping of threads numbers into event processing classes 
		Map<Integer, ThreadAutomatons> currentState = state.peek(); 

		// Process all events (stop if error found)
		for(int eventIndex = 0; eventIndex < events.size() && result == true; eventIndex++) {
			EventItem event = events.get(eventIndex);
			ThreadAutomatons threadAutomatons = currentState.get(event.thread);

			if (DEBUG) { System.out.println("  PROCESSING event " + event + "\n    threadAutomatons:" + threadAutomatons); }
			
			if (threadAutomatons == null) {
				throw new RuntimeException("Internal Error - events from unknown (not started thread) (Event=" + event + ")");
			}

			// Check event type
			if (event.type == EventItem.EventTypes.EVENT_CALL) {
				// Check for interface type
				if (config.getComponentProvidedInterfaces().containsKey(event.interfaceName)) {
					// Call on provided interface ... create new automaton set
					ThreadAutomatons newState = threadAutomatons.processEventProvidedCall(event, processEventsCallCount);
					result &= !newState.getError();
					processEventsIndexErrorEvent = eventIndex; // Remember problematic event
 					currentState.put(event.thread, newState);
				} else {
					// Reaction on last provided iface call
					ThreadAutomatons newState = threadAutomatons.processEventRequiredCall(event, processEventsCallCount);
					result &= !threadAutomatons.getError();
					processEventsIndexErrorEvent = eventIndex; // Remember problematic event
					currentState.put(event.thread, newState);
				}
				
			} else if (event.type == EventItem.EventTypes.EVENT_RETURN) {
				// Check for interface type ... provided or required
				if (config.getComponentProvidedInterfaces().containsKey(event.interfaceName)) {
					// Return on provided interface
					if (threadAutomatons == null) {
						// No call on provided interface
						result = false;
						processEventsIndexErrorEvent = eventIndex; // Remember problematic event
						continue;
					}
					// process event
					ThreadAutomatons prevLevel = threadAutomatons.processEventProvidedReturn(event, processEventsCallCount);
					result &= !threadAutomatons.getError();
					processEventsIndexErrorEvent = eventIndex; // Remember problematic event
					currentState.put(event.thread, prevLevel);
				} else {
					// Return on required interface ... check if specification (automaton) permits this return
					ThreadAutomatons prevLevel = threadAutomatons.processEventRequiredReturn(event, processEventsCallCount);
					result &= !threadAutomatons.getError();
					processEventsIndexErrorEvent = eventIndex; // Remember problematic event
					currentState.put(event.thread, prevLevel);
				}
				
			} else {
				throw new RuntimeException("Unknown event type (Event=" + event + ")");
			}

		}	
		
		return result;
	}
	
	/**
	 * @return Gets index event in last {@link #processEvents(List)} call that caused error. Or value {@link EventParser#NO_ERROR} if no error occure. 
	 */
	public int getProcessEventIndexErrorEvent() {
		return processEventsIndexErrorEvent;
	}

	/**
	 * Process newly started threads. Analyze places where threads are started and
	 *  determine type of thread.  
	 * @param startedThreads Map with created threads and places where creation takes place.
	 * @return True if no error occurred during processing, false if any thread created in prohibited place of if thread already exists.
	 */
	private boolean processThreadStarts(Map<Integer, ThreadAutomatons.ThreadType> startedThreads) {
		assert startedThreads != null;
		if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":processThreadStarts( " + startedThreads + ")"); }
		
		Map<Integer, ThreadAutomatons> currentState = state.peek(); 
		
		for(Entry<Integer, ThreadAutomatons.ThreadType> threadRec : startedThreads.entrySet()) {
			if (currentState.containsKey(threadRec.getKey())) {
				//DEBUG_printState();
				throw new RuntimeException("Internal error - Existing thread newly started");
				// return false;
			}

			ThreadAutomatons newThreadTA = new ThreadAutomatons(spec, varStateMap, threadRec.getValue(), processEventsCallCount, threadRec.getKey(), envDesc);
			currentState.put(threadRec.getKey(), newThreadTA);
		}
		return true;
	}

	/**
	 * Process ended threads. Redirect action into proper {@link ThreadAutomatons} instance if possible.
	 * @param endedThreads List of thread that ended during last JPF step.
	 * @return True if no error occurred during processing, false if any thread cann't ended and excepted any call or return to be able stop.
	 */
	private boolean processThreadEnds(List<Integer> endedThreads) {
		if (DEBUG) { System.out.println(this.getClass().getSimpleName() + ":processThreadEnds( " + endedThreads + ")"); }
		if (DEBUG_INFO) { DEBUG_printState(); }
		boolean result = true; // error flag ... marks if error present
		
		
		if (state.isEmpty() == true) {
			// Some thread ends, but currently no thread that process events 
			//  -> this thread end isn't related to interesting thread

			return true;
		}
		Map<Integer, ThreadAutomatons> currentState = state.peek(); 
		
		for(Integer endedThreadNum : endedThreads) {
			ThreadAutomatons automatons = currentState.remove(endedThreadNum);
			if (automatons != null) { 
					// Null means no interesting action take place for this thread 
					//   returned from last provided interface call --> correct end
				
					// so this block catch errors from THREAD automatons and threads with some error
				result &= automatons.isEndState();
			}
		}
		
		return result;
		
	}

	/**
	 * Gets type of given thread. 
	 *   Thread has to be stored first in internal structures, 
	 *     {@link #processThreadStarts(Map)} must be called before this method.
	 * 
	 * @param threadNum Number of processed thread
	 * @return Gets type of given thread. Null if given thread not exists.
	 */
	public ThreadAutomatons.ThreadType getThreadType(int threadNum) {
		Map<Integer, ThreadAutomatons> currentState = state.peek(); 
		ThreadAutomatons threadAutomatons = currentState.get(threadNum);
		if (threadAutomatons == null) {
			// No such thread
			return null;
		}
		return threadAutomatons.getThreadType();
	}
	
	/**
	 * Take back actions made by last {@link #processEvents(List)} call. And return state as was before call. 
	 * @return true if there is {@link #processEvents(List)} that can be taken back, false otherwise. (No paired {@link #processEvents(List)})
	 */
	public boolean undoEvents() {
		if (DEBUG) { System.out.println(this.getClass().getName() + ":undoEvents()"); }
		if (DEBUG_INFO) { DEBUG_printState(); }
		
		if (state.isEmpty()) {
			return false; // No step to be undo
		}
		state.pop();
		varStateMap.removeStackLayer();
		return true;
	}

	/**
	 * Adds new level into state stack. 
	 * threadNumsToClone specifies which {@link ThreadAutomatons} has to be cloned (deep copy), 
	 * 		for other thread states will be copied only reference.
	 *
	 * @param threadNumsToClone Number of threads where state will be later modify so we want deep copy.
	 */
	private void createNewStateLevel(Set<Integer> threadNumsToClone) {
		if (DEBUG) {System.out.println("createNewStateLevel(" + threadNumsToClone + ")");}
		Map<Integer, ThreadAutomatons> newLevel = new HashMap<Integer, ThreadAutomatons>();
		
		if (state.isEmpty()) {
			// Nothing to copy
			state.add(newLevel);
			return;
		}
		
		Map<Integer, ThreadAutomatons> topLevel = state.peek();
		for(Entry<Integer, ThreadAutomatons> entry : topLevel.entrySet()) {
			final Integer processedThread = entry.getKey();
			ThreadAutomatons automatons = entry.getValue();

			if (threadNumsToClone.contains(processedThread) && automatons != null) {
				// Automatons == null --> 
				//   it's existing thread that ends all calls (has empty stack and waits for new events)
				//   nothing to clone

				// Making deep copy
				automatons = automatons.clone(processEventsCallCount);
			}
			newLevel.put(processedThread, automatons);
		}
		state.add(newLevel);
	}
	
	
	/**
	 * Prints out into standard output stack.
	 */
	public void DEBUG_printState() {
		System.out.println("DEBUG - Printing " + this.getClass().getSimpleName() + " state");
		System.out.println("  Stack state - depth:" + state.size());
		for(Map<Integer, ThreadAutomatons> map : state) {
			System.out.println("    " + map);
		}
	}

	public static void DEBUG_printEvents(List<EventItem> events, int indent) {
		if (events==null) {
			return;
		}
		
		String indentation = ThreadAutomatons.DEBUG_indetStringBuffer(null, indent).toString();
		
		for(EventItem event : events) {
			System.out.print(indentation);
			System.out.println(event);
		}
		
	}

}
