package org.ow2.dsrg.fm.tbpjava.checker;

/**
 * Represents important events (@see ProtocolListener) that occur in tested environment.
 * <br>
 * {@link EventItem} is typically created in {@link ProtocolListener} 
 *   than pass throw {@link EventParser} into proper {@link ThreadAutomatons}. 
 *   {@link ThreadAutomatons} process incoming {@link EventItem} and moves with automatons accordingly.
 *
 * <br>
 * Can "compare" events {@link #isAssociatedEvent(EventItem)}, support stack printing.  
 * <br>
 * Immutable class.
 * 
 */
class EventItem {
	/// Specify types of represented events 
	public enum EventTypes { 
		EVENT_CALL, /// Event is call of method declared in any (provided/required) interface.
		EVENT_RETURN /// Event is return from method declared in any (provided/required) interface.
	};

	final public EventItem.EventTypes type; /// Specify represented event type.
	final public int thread; /// Number of thread that caused event
	final public String methodName; /// Called method name (returning method) 
	final public String interfaceName; /// (Specification scope) interface name with represented method
	final public String fullName; /// Combination of interface and method name. 
	final public int line; /// Line in source code where event takes place
	
	public EventItem(int thread, EventItem.EventTypes type, String methodName, String interfaceName, int line) {
		this.type = type;
		this.thread = thread;
		this.methodName = methodName;
		this.interfaceName = interfaceName;
		this.line = line;
		
		this.fullName = interfaceName + "." + methodName;
	}
	
	/**
	 * Check if given event is associated with this instance. 
	 *  It means that one of event is CALL and another RETURN of same method in same thread.
	 * 
	 * @param event Event to test. Cann't be null.
	 * @return True if given event is from same thread on same interface and method, only types are different, false otherwise.
	 */
	public boolean isAssociatedEvent(EventItem event) {
		assert event != null;
		return (type != event.type) && (thread == event.thread) && (methodName == event.methodName) && (interfaceName == event.interfaceName);
	}
	
	public String toString() {
		return "EventItem( type=" + type + ", thread=" + thread + ", methodName=" + methodName + ", interfaceName=" + interfaceName + ", line=" + line + ")";
	}

	public final static int PRINT_EVENT_TRACE_START_OFFET_EVENT_TYPE = 0;  /// Index where in event stack trace starts column with event type
	public final static int PRINT_EVENT_TRACE_START_OFFET_THREAD = 15; /// Index where in event stack trace starts column with thread number
	public final static int PRINT_EVENT_TRACE_START_OFFET_INTERFACE_TYPE = 23; /// Index where in event stack trace starts column with type of interface (provided, required)
	public final static int PRINT_EVENT_TRACE_START_OFFET_INTERFACE_NAME = 34; /// Index where in event stack trace starts column with interface name
	public final static int PRINT_EVENT_TRACE_START_OFFET_METHOD_NAME = 70; /// Index where in event stack trace starts column with method name
	public final static int PRINT_EVENT_TRACE_START_OFFET_END = 81; /// Index where in event stack trace output ends

	/**
	 * Converts event into human readable form. This form is used in event stack trace.
	 * 
	 * @param pl Protocol listener. It's used to determine type of interface.
	 * @return Similar to {@link #toString()} but more user friendly.
	 */
	public String toStringEventTrace(ProtocolListener pl) {
		StringBuffer result = new StringBuffer();
		
		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_EVENT_TYPE);
		if (type == EventTypes.EVENT_CALL) {
			result.append("call");
		} else if (type == EventTypes.EVENT_RETURN) {
			result.append("return");
		}

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_THREAD);
		result.append("| ");
		result.append(thread);
		
		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_INTERFACE_TYPE);
		result.append("| ");
		if (pl.getProvItfs2implClass().containsKey(interfaceName)) {
			result.append("provided");
		} else if (pl.getReqItfs2implClass().containsKey(interfaceName)) {
			result.append("required");
		} else {
			result.append("????");
		}

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_INTERFACE_NAME);
		result.append("| ");
		result.append(interfaceName);

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_METHOD_NAME);
		result.append("| ");
		result.append(methodName);

		return result.toString();
	}
	
	/**
	 * Gets header with column info for event stack trace from {@link #toStringEventTrace(ProtocolListener)}
	 * 
	 * @return Gets header with column info for event stack trace from {@link #toStringEventTrace(ProtocolListener)}
	 */
	public static String eventTraceHeader() {
		StringBuffer result = new StringBuffer();
		
		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_EVENT_TYPE);
		result.append("Event type");

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_THREAD);
		result.append("| ");
		result.append("Thread");

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_INTERFACE_TYPE);
		result.append("| ");
		result.append("IF type");
		
		
		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_INTERFACE_NAME);
		result.append("| ");
		result.append("Interface name");

		fillSpacesToLen(result, PRINT_EVENT_TRACE_START_OFFET_METHOD_NAME);
		result.append("| ");
		result.append("Method name");

		return result.toString();

	}
	
	
	/**
	 * Gets line of "----" of with same length as trace header.
	 * @return Gets line of "----" of with same length as trace header.
	 */
	public static String eventTraceSeparator() {
		StringBuffer result = new StringBuffer(PRINT_EVENT_TRACE_START_OFFET_END);
		for(int i = 0; i < PRINT_EVENT_TRACE_START_OFFET_END; i++) {
			result.append('-');
		}
		return result.toString();
		
	}
	
	/**
	 * For correct output formating. Add spaces into the end of SB to have length len.
	 * @param sb StringBuffer to modify
	 * @param len Final length of SB after adding spaces.
	 */
	private static void fillSpacesToLen(StringBuffer sb, int len) {
		for(int i = sb.length(); i < len; i++) {
			sb.append(' ');
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((fullName == null) ? 0 : fullName.hashCode());
		result = prime * result
				+ ((interfaceName == null) ? 0 : interfaceName.hashCode());
		result = prime * result
				+ ((methodName == null) ? 0 : methodName.hashCode());
		result = prime * result + thread;
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof EventItem))
			return false;
		EventItem other = (EventItem) obj;
		if (fullName == null) {
			if (other.fullName != null)
				return false;
		} else if (!fullName.equals(other.fullName))
			return false;
		if (interfaceName == null) {
			if (other.interfaceName != null)
				return false;
		} else if (!interfaceName.equals(other.interfaceName))
			return false;
		if (methodName == null) {
			if (other.methodName != null)
				return false;
		} else if (!methodName.equals(other.methodName))
			return false;
		if (thread != other.thread)
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}
	
	

}