package org.ow2.dsrg.fm.tbpjava.checker;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbpjava.checker.ThreadStateStack.ReturnValues;
import org.ow2.dsrg.fm.tbplib.ltsa.Edge;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.ltsa.State;
import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.resolved.CVRef;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.LastCallRef;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedAssignment;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNull;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedReturn;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedUndefinedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedValue;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition.ConditionType;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Class holds logic of automaton moves.
 * <br>
 * Internally it traverse automaton from given start state. 
 * It uses deep first strategy. Stack that hold state 
 * <br>
 * Note: most important method {@link #moveAutomatons(Set)}
 * 
 * @author Alfifi
 *
 */
final class AutomatonsMoves {
	private static final boolean DEBUG = false; // Set to true if you want to see debug prints from this object
	//private static final boolean DEBUG_INFO = DEBUG; // More talky debug informations
	
	private static final boolean FINAL_AUTOMATON_STATE_GO_THROUGHT = false;
	private StatesMapper varStatesMap; /// Used for mapping between automatons states and sets of variables arrays (variablesState)
	private AutomatonWalker walker; /// Used for generating new edges to process.
	
	private LTSAComponentSpecification specification; /// Parsed specification of processed component
	
	private Map<Edge, String> reverseEdgeMap; /// Maps edge to event names (automatons associated with events contains mapped edge). See {@link AutomatonsMoves#createReverseEdgeMap}
	
	/**
	 * @param specification Parsed TBP specification of processed component where automaton take place.
	 * @param varStatesMap Mapper that (is used for mapping) maps {@link AutomatonState}s and {@link VariablesEvaluation}s together. Cann't be null.
	 * 
	 */
	public AutomatonsMoves(LTSAComponentSpecification specification, StatesMapper varStatesMap) {
		assert specification != null;
		assert varStatesMap != null;
		this.specification = specification;
		this.varStatesMap = varStatesMap;
		this.reverseEdgeMap = AutomatonsMoves.createReverseEdgeMap(specification);
		this.walker = new AutomatonWalker();
	}
	
	/** Getter for reverse edge map. For each edge in any automaton gets name of event that is connected with LTSAAutomaton in which edge take place*/
	final public Map<Edge, String> getReverseEdgeMap() {
		return reverseEdgeMap;
	}
	
	/**
	 * Define way of {@link AutomatonsMoves#processEdge(Set, org.ow2.dsrg.fm.tbpjava.checker.AutomatonWalker.EdgeToProcess, PROCESS_MODE)} behavior.
	 *
	 * <br>
	 * There are two types of behavior. 
	 *  One for active forward moving when event occur (moves until call, return or mutex enter or final state).
	 *  Another for moving from mutex enters before call event (moves until call or return or final state).
	 *
	 */
	public enum PROCESS_MODE {
		/**
		 *  First behavior mode. The {@link AutomatonsMoves#processEdge(Set, org.ow2.dsrg.fm.tbpjava.checker.AutomatonWalker.EdgeToProcess, PROCESS_MODE)} method
		 *  stops processing and save state as unprocessed whenever find call (emit) edge, 
		 *  return edge (processed) or mutex enter edge (CVRef edge) (unprocessed) or final state (unprocessed).
		 *  
		 *  <br>Note: This mode is used from {@link AutomatonsMoves#moveAutomatons(Set)}.
		 */
		PROCESS_MODE_MUTEXES_STOP,
		/**
		 *  Second behavior mode. The {@link AutomatonsMoves#processEdge(Set, org.ow2.dsrg.fm.tbpjava.checker.AutomatonWalker.EdgeToProcess, PROCESS_MODE)} method
		 *  stops and save save state as unprocessed whenever find call (emit) edge,
 		 *  return edge (processed) or final state.
 		 *  <br>
 		 *  In this mode mutexes are processed normally it means, if possible mutexes are locked, entered and processing continues.
		 *  
		 *  <br>Note: This mode is used from {@link AutomatonsMoves#moveAutomatons1(Set)}.
		 */
		PROCESS_MODE_EXECUTE_MUTEXES
	}

	/**
	 * Save as final state. State will be processed (by {@link ThreadAutomatons}) during next event processing.
	 * When walking throw automaton we found {@link State} where we could move and where event invocation or return take place)  
	 * 
	 * @param newStoredAutomatonStates Set where save given automaton state. Given set should contain only {@link AutomatonState}s where you can be in {@link LTSAAutomaton} after current event.
	 * @param sourceAS Source event from which new event has to be created (by change of state property)
	 * @param variables Variables values that have to be assigned with saves state. 
	 * @param edge Edge of automaton that will be saved. (Newly created automaton state will be identical with sourceAS except edge and state property)
	 * @param edgeActionProcessed Marks that action on edge was processed (action take place true -> [mutex lock, assigned take place, call returned], false -> [mutex not entered, call in process not returned])
	 */
	final private void saveNewAutomatonState(Set<AutomatonState> newStoredAutomatonStates, AutomatonState sourceAS, VariablesEvaluation variables, Edge edge, boolean edgeActionProcessed) {
		AutomatonState newAS = new AutomatonState(sourceAS, edge, edgeActionProcessed);
		newStoredAutomatonStates.add(newAS); // Save new state
	
		// Create mappings for new state
		varStatesMap.setMappings(newAS, variables);
	}
	
	/**
	 * Save as state for further processing. State will be saved is final if the target automaton state is the final state in the automaton, otherwise it will be planned for further processing 
	 * in the {@link #processEdge(Set, org.ow2.dsrg.fm.tbpjava.checker.AutomatonWalker.EdgeToProcess, PROCESS_MODE)} during processing of current event.
	 * 
	 * @param newStoredAutomatonStates Set where save given automaton state. Given set should contain only {@link AutomatonState}s where you can be in {@link LTSAAutomaton} after current event.
	 * @param as Source event from which new event has to be created (by change of state property)
	 * @param variables Variables values that have to be assigned with saves state. 
	 * @param edge Edge of automaton that will be saved. (Newly created automaton state will be identical with sourceAS except edge and state property)
	 * @param edgeActionProcessed Marks that action on edge was processed (action take place true -> [mutex lock, assigned take place, call returned], false -> [mutex not entered, call in process not returned])
	 */
	final private boolean planFurtherProcessing(Set<AutomatonState> newStoredAutomatonStates, AutomatonState as, VariablesEvaluation variables, Edge edge, boolean edgeActionProcessed) {

		final State targetState = edge.getTarget();
		final boolean isTargetStateFinal = as.getAutomaton().getFin().equals(targetState);

		if (isTargetStateFinal) {
			saveNewAutomatonState(newStoredAutomatonStates, as, variables, edge, true);
			return false;
		}
		
		assert ((isTargetStateFinal && FINAL_AUTOMATON_STATE_GO_THROUGHT) || !isTargetStateFinal);
		
		// Not final state or go through final states
		walker.setStateToExplore(as, variables, targetState);
		return true;
		
	}

	/**
	 * Process edge in automaton. 
	 * Change variables evaluation if assignment takes place in this edge, stops on emits(calls) and returns.
	 * For more detailed behavior description for each edge type see documentation.
	 * 
	 * <br>Note: Don't check if emit is calling expected method. Only saves state so further filtering (removing) is necessary during result postprocessing.
	 * <br>Note: Don't check if the return is expected event. Only saves state so further filtering (removing) is necessary during result postprocessing. 
	 * 
	 * @param newStoredAutomatonStates Set where save automaton states where you can be in {@link LTSAAutomaton} after given event.
	 * @param edge2process Represents edge that should by processed by method.
	 * @param processingMode Specify how has method behave. See {@link PROCESS_MODE} definition. (Specify how process mutexes whether stop or continue processing)
	 * @return true if it's possible go through given edge2process(next state planned for processing), false otherwise (processing of path ends )  
	 */
	private boolean processEdge(Set<AutomatonState> newStoredAutomatonStates, AutomatonWalker.EdgeToProcess edge2process, PROCESS_MODE processingMode) {
		
		// Don't have state
		boolean result = false; // represent return value 
		
		// Process specific edge
		final Edge e = edge2process.getEdge();
		
		final State targetState = e.getTarget();
		final Object edgeType = e.getData();
		
		final AutomatonState as = edge2process.getAs();
		
		final boolean isTargetStateFinal = as.getAutomaton().getFin().equals(targetState);
		
		VariablesEvaluation variables = edge2process.getVariables();
		// Known edge types
		//	X TBPResolvedAssignments ...
		//	X TBPResolvedEmit ... call on required interface, automaton where process next events exists
		//	X TBPResolvedReturn
		//	X TBPResolvedImperativeNull
		//	X TBPResolvedUndefinedEmit ... call on required interface, automaton of called event not exists
		//  X TBPResolvedCondition ... conditions (switch, ...)
		//	X CVRef .. reference to variable (should be part of assignment or mutex)

		if (edgeType instanceof TBPResolvedImperativeNull) {
			// Go through and don't change any variable evaluation
			result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
		} else 

		if (edgeType == null) {
			result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
		} else 

		if (edgeType instanceof TBPResolvedAssignment) {
			final TBPResolvedAssignment edgeAssigment = (TBPResolvedAssignment)edgeType;
			
			// There are 2 types Assignments -> 
			//   a) from constant(variable, ...) 
			//   b) from Emit(result of another automaton) 

			TBPResolvedValue tbpValue = edgeAssigment.getValue();
			if (tbpValue.isReference()) {
				// a) reference can be constant
				// b) reference can be LastCallRef ... return value of last called automaton
				
				final Reference ref = tbpValue.getReference();
				if (ref instanceof LastCallRef) {
					// variant a) last call ref
					ReturnValues retValType = varStatesMap.getLastCallReturnValueType(as.getThreadNum(), variables);
					if (retValType == ReturnValues.RET_VAL_NOT_SET) {
						// Possible we could ignore the assignment and don't assign any value
						throw new RuntimeException("Internal error - No return values specified");
					} else if (retValType == ReturnValues.RET_VAL_SET) {
						String lastCallReturnValue = varStatesMap.getLastCallReturnValue(as.getThreadNum(), variables);
						variables = variables.setVariableValue(edgeAssigment.getIdf().getName(), lastCallReturnValue);
						result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
					} else if (retValType == ReturnValues.RET_VAL_UNDEFINED_EMIT) {
						Typedef assignedType = edgeAssigment.getIdf().getType();
						for(String assignedValue : assignedType.getEnums()) {
							// I need to create more copies to process 
							variables = variables.setVariableValue(edgeAssigment.getIdf().getName(), assignedValue);
							result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
						}
						
					} else {
						throw new RuntimeException("Unknown enum value " + retValType + " in the " + ReturnValues.class.getName());
					}
				} else if (ref instanceof ConstantRef) {
					// variant c) ... a constant
					variables = variables.setVariableValue(edgeAssigment.getIdf().getName(), tbpValue.getReference().getName());
					result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
				} else {
					// Potentially here can be FPRef or CVRef
					throw new RuntimeException("Unknown assignment reference value type " + ref.getClass().toString());
				}
			} else {
				// c) from Emit(result of another automaton)

				// Don't check if correct call. Filtering is done later. 
				saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, false);
				result = false; // Final moving state, we don't go through call
			} // End Method call 
		} else // TBPResolvedAssignment
		
		if (edgeType instanceof TBPResolvedEmit) {

			// Don't check if correct call. Filtering is done later. 
			saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, false);
			result = false; // Final moving state, we don't go through call
		} else // TBPResolvedEmit
		
		if (edgeType instanceof TBPResolvedUndefinedEmit) {

			// Don't check if correct call. Filtering is done later. 
			saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, false);
			varStatesMap.setLastCallReturnValueTypeUndefinedEmit(as.getThreadNum(), variables);
			// Update last call return value
			result = false; // Final moving state, we don't go through call
		} else // TBPResolvedUndefinedEmit

		
		if (edgeType instanceof TBPResolvedReturn) {
			TBPResolvedReturn edgeReturn = (TBPResolvedReturn)edgeType;
			
			ConstantRef retValRef = edgeReturn.getReturnValue();
			String retVal = null;

			if (retValRef != null) {
				// We are returning any value (not procedure return)
				retVal = retValRef.getName();
				varStatesMap.setReturnValue(as.getThreadNum(), variables, retVal);
			}
			// Return means final state. Stop processing and return.
			//walker.setStateToExplore(as, variables, targetState); 

			saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, true);
			result = false; // Final moving state, we don't go through call
		} else // TBPResolvedReturn
	
		if (edgeType instanceof TBPResolvedCondition) {
			TBPResolvedCondition edgeCond = (TBPResolvedCondition)edgeType;
			
			ConditionType cond_type = edgeCond.getConditionType();
			if (cond_type == ConditionType.CONDITION_ANY) {
				// nothing to change, try to go through
				result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
			} else if (cond_type == ConditionType.CONDITION_VARIABLE_CONSTANT) {
				// Check value if variable with constant
				String variableName = edgeCond.getLeft().getName();
				String variableValue = variables.getVariableValue(variableName);
				
				String constantValue = edgeCond.getRight().getName();
				
				if ( ( (edgeCond.isNegated() == false) && (variableValue.equals(constantValue) == true ) ) ||
					 ( (edgeCond.isNegated() == true)  && (variableValue.equals(constantValue) == false) ) ) {
					// Condition passed .. go through

					result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
				} else { 
					// Disabled edge .. condition is not true
					result = false;
				}
			} else if (cond_type == ConditionType.CONDITION_VARIABLE_IN_CONSTANTS) {
				String variableName = edgeCond.getLeft().getName();
				String variableValue = variables.getVariableValue(variableName);
				
				for(ConstantRef constReference : edgeCond.getRightList()) {
					String constantValue = constReference.getName();
					if ( ( (edgeCond.isNegated() == false) && (variableValue.equals(constantValue) == true ) ) ||
					     ( (edgeCond.isNegated() == true)  && (variableValue.equals(constantValue) == false) ) ) {
						// Condition holds
						result = planFurtherProcessing(newStoredAutomatonStates, as, variables, e, true);
					} else {
						// Part of condition doesn't holds -> Whole condition don't holds -> don't go through edge 
						result = false;
						break;
					}
				}
			} else if (cond_type == ConditionType.CONDITION_MULTIPLE_VARIABLES_TO_CONSTANT_ONE_BY_ONE) {
				throw new RuntimeException("This condition exists only in inlined automatons that we are not using.");
			} else {
				// Newly added unknown condition type.
				throw new RuntimeException("Unknown condition type - " + cond_type.toString());
			}
			
		} else // TBPResolvedCondition
		
		if (edgeType instanceof CVRef) {
			// Should be mutex
			CVRef edgeCVRef = (CVRef)edgeType;
			
			// Check for mutex type
			if ( !Typedef.MUTEX_TYPE.equals(edgeCVRef.getType()) ) {
				// Unknown meaning CVRef not pointing to mutex ... unknown semantics
				throw new RuntimeException("Unknown CVRef (not representing mutex) referenced type - " + edgeCVRef.getType());
			}
			
			// Test processing mode whether stop and save state of lock and continue processing current trace.
			if (processingMode == PROCESS_MODE.PROCESS_MODE_MUTEXES_STOP) {
				// Save state as unprocessed (fill be processed layer before call) and stop processing
				if (isTargetStateFinal) {
					throw new RuntimeException("Internal error - unexpected automaton - final state directly after mutex enter");
				}
				saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, false);
				result = false; // We don't go through call
			} else if (processingMode == PROCESS_MODE.PROCESS_MODE_EXECUTE_MUTEXES) {
				// Now work with mutex
				String mutexVariableName = edgeCVRef.getName();
				assert mutexVariableName != null; // Unknown variable name
				String mutexState = variables.getVariableValue(mutexVariableName);
				assert mutexState != null; // Unknown mutex state

				// Check state of mutex
				if (Typedef.UNLOCKED.equals(mutexState)) {
					// Mutex unlocked ... lock them and continue processing
					
					variables = variables.setVariableValue(mutexVariableName, Typedef.LOCKED);

					// Save into stack and continue processing
					if (isTargetStateFinal) {
						throw new RuntimeException("Internal error - unexpected automaton - final state directly after mutex enter");
						//saveNewAutomatonState(newStoredAutomatonStates, as, variables, e, true);
						//result = false;
					} 
					if ((isTargetStateFinal && FINAL_AUTOMATON_STATE_GO_THROUGHT) || !isTargetStateFinal) {
						// Not final state or go through final states
						// Save into stack and continue processing
						walker.setStateToExplore(as, variables, targetState);
						result = true;
					}
				} else if (Typedef.LOCKED.equals(mutexState)) {
					// Mutex locked ... current automaton trace is disabled ... stop processing and backtrack
					result = false;
				} else {
					throw new RuntimeException("Unknown mutex (" + mutexVariableName + ") state (neither " + Typedef.LOCKED + " nor " + Typedef.UNLOCKED + " - actual value is " + mutexState + ")");
				}
				
			} else {
				//Unknown processing mode
				throw new RuntimeException("New unknown enum " + PROCESS_MODE.class.getSimpleName() + " entry called " + processingMode + ". Update checker code please.");
			}
			
		} else // CVRef - Mutexes

		
		{
			// The way how to found new unknown types of edges
			// Another unknown edge types
			throw new RuntimeException(AutomatonsMoves.class.getSimpleName() + ":automatonProcessEdge - unknown edge type - e" + edgeType.getClass().toString());
		}
		
		return result;
	}
	
	/** 
	 * Moves with states in given set.
	 * <br>
	 * Takes processed {@link AutomatonState} from given set one by one. (Unprocessed stated are removed)
	 *  For each {@link VariablesEvaluation} associated to processed {@link AutomatonState}
	 *   moves state according automaton edges.
	 * For each such pair in result set will be all reachable returns and emits (calls) and mutes enters states stored. 
	 * (Only such states that are reachable from processed pair. Only first such states ... doesn't move through returns and emits)
	 * {@link VariablesEvaluation} will be updates a mapped to new state in {@link StatesMapper}.
	 * 
	 * <br>Note: It mean find all reachable calls and returns mutex enters.
	 * <br>Note: It's supposed that this method will be called after return events or on new automaton start state.
	 * <br>Note: Behavior is this method is similar to {@link #moveAutomatonsEnterMutexes(Set)}, but this methods stops on mutex enters nd filters different events.
	 * 
	 * @param currentAutomatonStates List of states to process. (Start points of DFS searching)
	 * @param threadNum Specify thread that caused moving with automatons 
	 * @return Set where are stored automaton states after movement in automatons.
	 */
	public Set<AutomatonState> moveAutomatons(Set<AutomatonState> currentAutomatonStates/*, AUTOMATON_MOVING_TYPE moveType*/) {
		assert currentAutomatonStates != null;
		if (DEBUG) { System.out.println("  " + AutomatonsMoves.class.getSimpleName() + ".moveAutomatons(currentASs=" + currentAutomatonStates + ")"); }


		AutomatonWalker.EdgeToProcess edge2Process = new AutomatonWalker.EdgeToProcess(); // temporary object where next pair edge/variables are stored before processing

		// Holds states in automaton after event execution (result)
		Set<AutomatonState> newStoredAutomatonStates = new HashSet<AutomatonState>(currentAutomatonStates.size());

		Set<AutomatonState> statesToProcess = filterProcessedStates(currentAutomatonStates);
		
		// Store all mappings to process
		// Note: We cannot obtain mapping during processing AutomatonState. 
		//  Because during processing of previous AS we can modify (add) new VariablesEvalutaion to another AS that will be processed later again
		//   -> same VariablesEvaluation can represent more calls  
		Map<AutomatonState, Set<VariablesEvaluation>> mappingsToProcess = getStatesMappings(statesToProcess);

		// Main processing cycle
		for(Entry<AutomatonState, Set<VariablesEvaluation>> entry : mappingsToProcess.entrySet()) {
			AutomatonState as = entry.getKey();
			Set<VariablesEvaluation> variableStates = entry.getValue();
			if (DEBUG) { System.out.println("    " + AutomatonsMoves.class.getSimpleName() + "automatonsMoveAutomatons - processing state as=" + as); }

			assert variableStates != null; // From previous for loop
			
			// Check if there is final state in automaton after event
			if (as.getAutomaton().getFin().equals(as.getState())) {
				// After event there is final state (no move necessary)
				for(VariablesEvaluation variables : variableStates) {
					saveNewAutomatonState(newStoredAutomatonStates, as, variables, as.getEdge(), true);
				}
				if (!FINAL_AUTOMATON_STATE_GO_THROUGHT) {
					continue; // No moving though final state
				}
				
			}
			
			for(VariablesEvaluation variables : variableStates) {
				if (DEBUG) { System.out.println("      " + AutomatonsMoves.class.getSimpleName() + "automatonsMoveAutomatons - processing state as=" + as + ", withVars= " + variables); }
				
				// Start automaton processing
				walker.clean();
				walker.setStateToExplore(as, variables, as.getState());
				
				// Until any undiscovered pair edge/variables to process
				while( (edge2Process = walker.getNextStateToExplore(edge2Process)) != null) {
					processEdge(newStoredAutomatonStates, edge2Process, PROCESS_MODE.PROCESS_MODE_MUTEXES_STOP);
				}
				
			}
		}

		if (DEBUG) { System.out.println("  " + AutomatonsMoves.class.getSimpleName() + ".moveAutomatons( currentASs=" + currentAutomatonStates + ")\n    return=" + newStoredAutomatonStates); }
		return newStoredAutomatonStates;
	}
	
	/** 
	 * Moves with states in given set.
	 * <br>
	 * Takes unprocessed mutexes {@link AutomatonState} from given set one by one. (Other states are remains in result set unchanged)
	 *  For each {@link VariablesEvaluation} associated to processed {@link AutomatonState}
	 *   moves state according automaton edges.
	 * For each such pair in result set will be all reachable returns and emits (calls) states stored. 
	 * (Such states that are reachable from processed pair. Only first such states only ... doesn't move through returns emits)
	 * {@link VariablesEvaluation} will be updates a mapped to new state in {@link StatesMapper}.
	 * 
	 * <br>Note: It mean find all reachable calls and returns mutex enters.
	 * <br>Note: After call, it's necessary filter (remove) stored states in result set, according currently processed event.
	 * 
	 * @param currentAutomatonStates List of states to process. (Start points of DFS searching)
	 * @param threadNum Specify thread that caused moving with automatons 
	 * @return Set where are stored automaton states after movement in automatons.
	 */
	public Set<AutomatonState> moveAutomatonsEnterMutexes(Set<AutomatonState> currentAutomatonStates) {
		assert currentAutomatonStates != null;
		if (DEBUG) { System.out.println("  " + AutomatonsMoves.class.getSimpleName() + ".moveAutomatons1(currentASs=" + currentAutomatonStates + ")"); }

		AutomatonWalker.EdgeToProcess edge2Process = new AutomatonWalker.EdgeToProcess(); // temporary object where next pair edge/variables are stored before processing

		// Holds states in automaton after event execution (result)
		Set<AutomatonState> newStoredAutomatonStates = new HashSet<AutomatonState>(currentAutomatonStates.size());
		
		// We focus only on mutexes enters, that are unprocessed, 
		//  we process them and continue in walking through all mutex enters until call
		//  or return (final state) take place
		Set<AutomatonState>statesToProcess = filterUnprocessedMutexes(currentAutomatonStates);
		
		// All states in input that won't be processed are copied into result. 
		newStoredAutomatonStates.addAll(currentAutomatonStates);
		newStoredAutomatonStates.removeAll(statesToProcess);

		// Store all mappings to process
		// Note: We cannot obtain mapping during processing AutomatonState. 
		//  Because during processing of previous AS we can modify (add) new VariablesEvalutaion to another AS that will be processed later again
		//   -> same VariablesEvaluation can represent more calls  
		Map<AutomatonState, Set<VariablesEvaluation>> mappingsToProcess = getStatesMappings(statesToProcess);

		// Main processing cycle
		for(Entry<AutomatonState, Set<VariablesEvaluation>> entry : mappingsToProcess.entrySet()) {
			AutomatonState as = entry.getKey();
			Set<VariablesEvaluation> variableStates = entry.getValue();
			if (DEBUG) { System.out.println("    " + AutomatonsMoves.class.getSimpleName() + "automatonsMoveAutomatons1 - processing state as=" + as); }

			assert variableStates != null; // From previous for loop
			
			for(VariablesEvaluation variables : variableStates) {
				if (DEBUG) { System.out.println("      " + AutomatonsMoves.class.getSimpleName() + "automatonsMoveAutomatons1 - processing state as=" + as + ", withVars= " + variables); }
				
				// Start automaton processing
				walker.clean();
				walker.setEdgeToExplore(as, variables, as.getEdge());
				
				// Until any undiscovered pair edge/variables to process 
				while( (edge2Process = walker.getNextStateToExplore(edge2Process)) != null) {
					processEdge(newStoredAutomatonStates, edge2Process, PROCESS_MODE.PROCESS_MODE_EXECUTE_MUTEXES);
				}
				
			}
		}

		if (DEBUG) { System.out.println("  " + AutomatonsMoves.class.getSimpleName() + ".moveAutomatons( currentASs=" + currentAutomatonStates + ")\n    return=" + newStoredAutomatonStates); }
		return newStoredAutomatonStates;
		
	}

	/**
	 * Filter given set of {@link AutomatonState}s. In result set there are only states 
	 *   that are marked as processed (true == {@link AutomatonState#getEdgeActionProcessed()})
	 *   
	 * @param toFilter Set with {@link AutomatonState} to be filtered. Not modified.
	 * @return Subset of given set where {@link AutomatonState#getEdgeActionProcessed()} holds.
	 */
	private Set<AutomatonState> filterProcessedStates(Set<AutomatonState> toFilter) {
		Set<AutomatonState> result = new HashSet<AutomatonState>();
		for(AutomatonState as : toFilter) {
			if (as.getEdgeActionProcessed()) {
				result.add(as);
			}
			
		}
		return result;
	}

	/**
	 * Filter given set of {@link AutomatonState}s. In result set there are only states 
	 *   that represents mutexes which are unprocessed (false == {@link AutomatonState#getEdgeActionProcessed()})
	 *   
	 * @param toFilter Set with {@link AutomatonState} to be filtered. Not modified.
	 * @return Subset of given set where only specified condition holds.
	 */
	private Set<AutomatonState> filterUnprocessedMutexes(Set<AutomatonState> toFilter) {
		Set<AutomatonState> result = new HashSet<AutomatonState>();
		for(AutomatonState as : toFilter) {
			if (as.getEdgeActionProcessed() == false) {
				// check if represents mutex
				
				// Edge that is used as "enter" to represented state
				Edge e = as.getEdge();
				
				if (e == null) {
					continue; // process next AutomatonState
				}
				
				Object eData = e.getData();
				// Mutex enter is represented as CVRef(MutexTypeVariable)
				if (eData instanceof CVRef) {
					CVRef cv = (CVRef)eData;
					
					if (Typedef.MUTEX_TYPE.equals(cv.getType())) {
						// Is mutex.
						result.add(as);
					}
				}
			}
			
		}
		return result;
	}
	
	/**
	 * Gets {@link VariablesEvaluation} associated to {@link AutomatonState}s in given set. 
	 * Moves all mappings into PROCESSING state.
	 * 
	 * @param states Set of {@link AutomatonState} to use.
	 * @return For each {@link AutomatonState} from states creates entry with associated {@link VariablesEvaluation} in result map.
	 */
	private Map<AutomatonState, Set<VariablesEvaluation>> getStatesMappings(Set<AutomatonState> states) {
		Map<AutomatonState, Set<VariablesEvaluation>> result = new HashMap<AutomatonState, Set<VariablesEvaluation>>(states.size());
		
		for(AutomatonState as : states) {
			Set<VariablesEvaluation> values = varStatesMap.getMappings(as);
			if (values != null) {
				result.put(as, values);
			}
		}
		
		return result;
	}
	
	
	/**
	 * Takes care about call. Passes on ({@link VariablesEvaluation}) mappings associated with calling state to start state in called automaton.
	 *  
	 * @param callingTA {@link ThreadAutomatons} where calling state take place. Used for error reporting.
	 * @param callingStates Set of {@link AutomatonState} where search for calls.
	 * @param calledStatesSet Empty set where save start states of called automatons.
	 * @return True if any call was found and state added into calledStatesSet, false otherwise
	 */
	public boolean mapCallingStatesWithCalledStartState(ThreadAutomatons callingTA, Set<AutomatonState> callingStates, Set<AutomatonState> calledStatesSet) {
		boolean wasStateAdded = false;

		for(AutomatonState as : callingStates) {
			Edge e = as.getEdge();
			Object edgeType = e.getData();

			// Separate cases based on call types
			//	TBPResolvedAssignment ... call with return values 
			//	TBPResolvedEmit ... standard call without return value
			//	TBPResolvedUndefinedEmit .. call of unknown automaton cannot add mappings
			if (edgeType instanceof TBPResolvedAssignment) {
				TBPResolvedAssignment edgeAssigment = (TBPResolvedAssignment)edgeType;
				
				assert edgeAssigment.getValue().isMethodCall() == true; // if not true ... than state shouldn't be saved

				Binding calledMethod = edgeAssigment.getValue().getMethodCall().getBinding();
				mapCallingStatesWithCalledStartStateProcessCallMappings(callingTA, as, calledMethod, calledStatesSet);
				wasStateAdded = true;

			} else if (edgeType instanceof TBPResolvedEmit) {
				TBPResolvedEmit edgeEmit = (TBPResolvedEmit)edgeType;

				Binding calledMethod = edgeEmit.getBinding();
				mapCallingStatesWithCalledStartStateProcessCallMappings(callingTA, as, calledMethod, calledStatesSet);
				wasStateAdded = true;
			}
		}
		
		return wasStateAdded;
	}
	
	/**
	 * Process mapping for one call. Used from {@link AutomatonsMoves#mapCallingStatesWithCalledStartState(ThreadAutomatons, Set, Set)}
	 *
	 * @param callingTA {@link ThreadAutomatons} where calling state take place. Used for error reporting. 
	 * @param callingState State in automaton that "call" (target of calling edge)
	 * @param calledMethod Called method. Cann't be null.
	 * @param calledStatesSet Empty set where save start states of called automatons.
	 */
	private void mapCallingStatesWithCalledStartStateProcessCallMappings(ThreadAutomatons callingTA, AutomatonState callingState, Binding calledMethod, Set<AutomatonState> calledStateSet) {
		assert calledMethod != null;

		// We process one by one all mappings associated with callingState
		//   It's necessary drive processing by VariableEvaluations because variables values 
		//   determines which binding and LTSAAutomaton to use.
		Set<VariablesEvaluation> variablesSet = varStatesMap.getMappings(callingState);
		for(VariablesEvaluation variables : variablesSet) {
			// Check if all parameters are constant (all are bounded)
			Binding calledMethodBinding = calledMethod;

			// Some parameter of called function are not constants (try to repair)
			if (calledMethodBinding.getParameterNamesNotBoundToConstant().size() > 0) {
				// Some parameter is of called method is variable name 
				//   -> we need bound the with constant equivalent to it's 
				//      value in current environment. (pass by value
				calledMethodBinding = calledMethodBinding.extractConstantBinding(); // Now contain only parameter that was bound to constant
				
				// Now we will bound one be one "variable" parameters according VariablesEvaluation
				for(String parameterName : calledMethod.getParameterNamesNotBoundToConstant()) {
					// Obtain variable name used as parameter
					Reference paramRef = calledMethod.getValue(parameterName);
					
					// We expecting component variables (represented by CVRef)
					if (paramRef instanceof CVRef) {
						CVRef paramCVRef = (CVRef)paramRef;
						String variableName  = paramCVRef.getName();
						ConstantRef paramValue = variables.getVariableValueReference(variableName);
						assert paramValue != null; // If null then unknown variable name
						
						calledMethodBinding.bindParameter(parameterName, paramValue);
					} else {
						// paramRef null or unexpected type (ConstantRef, FPRef, LastCalRef)
						String error = "INTERNAL ERROR - Unexpected parametery type (or null) " + paramRef.getClass().getName() + " Method name " + calledMethod.getFullname() + ". Parameter name " + parameterName; 
						callingTA.setError(error);
						throw new RuntimeException(error);
					}
				}
			} // end if (!binding.isBound())
			
			assert calledMethodBinding.isBound() == true;
			// Now we have in proper Binding -> we can found LTSAAutomaton

			LTSAAutomaton calledAutomaton = specification.getReactions().get(calledMethodBinding);

			if (calledAutomaton == null) {
				String error = "INTERNAL ERROR - Calling undefined automaton " + calledMethod; 
				callingTA.setError(error);
				throw new RuntimeException(error);
			}
			
			// Create AutomatonState that represents start state in called automaton
			//   and map variables evaluation
			AutomatonState newAS = new AutomatonState(calledAutomaton, calledAutomaton.getStart(), callingState.getThreadNum());
			varStatesMap.setMappingsAsCall(newAS, variables);
			calledStateSet.add(newAS);
		} // end for each variablesEvaluation
	}

	
	/**
	 * Process variable mapping during return.
	 * <br>
	 * Moves mapping from current (where return take place) {@link AutomatonState}) back to previous
	 *  {@link AutomatonState} where call take place. And update {@link VariablesEvaluation} according return value
	 *  if needed. 
	 * 
	 * @param returningTA {@link ThreadAutomatons} where return take place. Used for error reporting.
	 * @param returningStates Set of {@link AutomatonState} which should return.
	 * @param threadNum Number of thread (from EventItem) where return take place. (Needed to obtain return value)
	 */
	public void processReturnValues(ThreadAutomatons returningTA, Set<AutomatonState> returningStates, int threadNum) {

		// For each stored automaton state (final state) find all variables mappings and 
		// each pair one by one process -> variables associate back to caller state and updates return value
		for(AutomatonState as : returningStates) {
			Set<VariablesEvaluation> variablesSet = varStatesMap.getMappings(as);

			for(VariablesEvaluation variables : variablesSet) {
				// Obtain return value
				String retVal = varStatesMap.getReturnValue(threadNum, variables);
				
				// AutomatonState where call take place
				AutomatonState callerAS = varStatesMap.setMappingsAsReturn(threadNum, variables);

				if (callerAS == null) {
					String error = "Implementation violates specification - required return without any found"; 
					returningTA.setError(error);
					throw new RuntimeException(error);
					//continue;
				}

				Edge e = callerAS.getEdge();
				assert e != null; // If e == null -> state is badly save !! internal error in call routines
				
				Object edgeType = e.getData();

				// Separate cases based on call types
				//	TBPResolvedAssignment ... call with return values 
				//	TBPResolvedEmit ... standard call without return value
				//	TBPResolvedUndefinedEmit .. call of unknown automaton no return value processing
				if (edgeType instanceof TBPResolvedAssignment) {
					TBPResolvedAssignment edgeAssigment = (TBPResolvedAssignment)edgeType;

					assert edgeAssigment.getValue().isMethodCall() == true; // if not true ... than cannot set return value

					// l-value variable -> edgeAssigment.getIdf().getName()
					// r-value          -> retVal
					
					// Change mapping of variables according stored return value
					varStatesMap.moveMappingToProcessing(variables);
					variables = variables.setVariableValue(edgeAssigment.getIdf().getName(), retVal);
					varStatesMap.setMappings(callerAS, variables);
				} else if (edgeType instanceof TBPResolvedUndefinedEmit) {
					throw new RuntimeException("Unexpected behavior ... should not process return values of the non existing automaton");
				}
			}

		}
	}
	
	/**
	 * Create subset of given currentAutomatonStates, than hold only AutomatonStates associated with event.
	 * 
	 * <br>Note: State is associated with event if 
	 *   a) "Call" event ({@link EventItem.EventTypes#EVENT_CALL}) then action on incoming edge in {@link AutomatonState} has to be call of same method
	 *   b) "Return" event ({@link EventItem.EventTypes#EVENT_RETURN}) then action on incoming edge in {@link AutomatonState} has to be return on edge from automaton that represents returning automaton.
	 * 
	 * <br>Note: For {@link AutomatonState} that don't represents neither call(emit) nor return throws {@link RuntimeException}
	 * 
	 * @param currentAutomatonStates Set of states to be filtered. Set not modified. Cann't be null. 
	 * @param event Event used for filtering. Cann't be null.
	 * @return Gets set of {@link AutomatonState}s which was given in currentAutomatonsStates and are associated with event.
	 */
	public Set<AutomatonState> filterStates(Set<AutomatonState> currentAutomatonStates, EventItem event) {
		assert event != null;
		assert currentAutomatonStates != null;
		if (DEBUG) { System.out.println("DEBUG " + AutomatonsMoves.class.getSimpleName() + ".filterStates(currentASs=" + currentAutomatonStates + ", event=" + event + ")"); }
		
		// Holds states in automaton after event execution (result)
		Set<AutomatonState> resultSet = new HashSet<AutomatonState>(currentAutomatonStates.size());

		for(AutomatonState as : currentAutomatonStates) {
			// Check edge we go into current AutomatonState as if corresponds to given event
			Edge incommingEdge = as.getEdge();
			
			boolean expectedEdge = checkEventEdgeCorrespondent(incommingEdge, event);
			if (expectedEdge) {
				resultSet.add(as);
			} else {
				// Unexpected call. Destroy all mappings for invalid state (that will be forgotten). (Not to be processed layer)
				varStatesMap.getMappings(as); /// Move them into processing state. This states will never be given back into mapped.
			}
		
		}
		if (DEBUG) { System.out.println("DEBUG " + AutomatonsMoves.class.getSimpleName() + ".filterStates .. result=" + resultSet); }
		
		return resultSet;
		
	}
	
	/**
	 * Create subset of given currentAutomatonStates, than hold only AutomatonStates associated with associated given return value.
	 * 
	 * For all state from currentAutomatonStates removes all mappings with not set return value or with different return value. 
	 * If for given automaton state doesn't remain any variables mapping then automaton state is filtered out.
	 * 
	 * @param currentAutomatonStates Set of states to be filtered. Set not modified. Cann't be null. 
	 * @param event Event we process. Specify thread number of acting thread.
	 * @param returnValue expectedReturnValue. Null means no filtering
	 * @return Gets set of {@link AutomatonState}s which was given in currentAutomatonsStates and are associated with event.
	 */
	public Set<AutomatonState> filterStatesOnReturnValue(Set<AutomatonState> currentAutomatonStates, EventItem event, String returnValue) {
		assert currentAutomatonStates != null;
		if (DEBUG) { System.out.println("DEBUG " + AutomatonsMoves.class.getSimpleName() + ".filterStatesOnReturnValue(currentASs=" + currentAutomatonStates + ", returnValue=" +returnValue + ")"); }

		if (returnValue == null) {
			// No filtering needed
			return currentAutomatonStates;
		}
		// Holds states in automaton after event execution (result)
		Set<AutomatonState> resultSet = new HashSet<AutomatonState>(currentAutomatonStates.size());

		Map<AutomatonState, Set<VariablesEvaluation>> mappingsToProcess = getStatesMappings(currentAutomatonStates);

		// Main processing cycle
		for(Entry<AutomatonState, Set<VariablesEvaluation>> entry : mappingsToProcess.entrySet()) {
			AutomatonState as = entry.getKey();
			Set<VariablesEvaluation> variableStates = entry.getValue();
			if (DEBUG) { System.out.println("    " + AutomatonsMoves.class.getSimpleName() + "filterStatesOnReturnValue - processing state as=" + as); }

			assert variableStates != null; // From previous for loop
			int variablesEvaluationCnt = 0;
			for(VariablesEvaluation variables : variableStates) {
				if (DEBUG) { System.out.println("      " + AutomatonsMoves.class.getSimpleName() + "filterStatesOnReturnValue - processing state as=" + as + ", withVars= " + variables); }

				String retVal = varStatesMap.getReturnValue(event.thread, variables);
				if (returnValue.equals(retVal)) {
					varStatesMap.setMappings(as, variables);
					variablesEvaluationCnt++;
				}
			} // For each evaluation of the state
			if (variablesEvaluationCnt > 0) {
				resultSet.add(as);
			}
		}

		return resultSet;
	}
	/**
	 * Check if edge represent (call or return) associated with event. 
	 * It means if given event is call then edge has to represent call of same method,
	 * if event is return then edge has to represent return from same automaton.
	 * 
	 * <br>Note: It suppose that edges are result of {@link #moveAutomatons(Set)} 
	 *   so throw {@link RuntimeException} unexpected edge is given.
	 * 
	 * @param edge Edge to check.
	 * @param event Event to check.
	 * @return True if edge represent given event, false otherwise.
	 */
	private boolean checkEventEdgeCorrespondent(Edge edge, EventItem event) {
		// Process specific edge
		final Object edgeType = edge.getData();
		
		// Known edge types - Expected edgeType
		//	X TBPResolvedAssignments
		//	X TBPResolvedEmit
		//	X TBPResolvedReturn
		//	X TBPResolvedUndefinedEmit
		// Unexpected -> error reported
		//	X TBPResolvedImperativeNull
		//  X TBPResolvedCondition
		//	X CVRef
		// Another edge
		
		if (edgeType instanceof TBPResolvedImperativeNull) {
			// No such state should remain in AutomatonState set after moveAutomatons call
			throw new RuntimeException("Unexpected method usage " + edgeType);
			// return false;
		} else 

			if (edgeType == null) {
			// No such state should remain in AutomatonState set after moveAutomatons call
			throw new RuntimeException("Unexpected method usage edgeType=null");
			// return false;
		} else 
		
		if (edgeType instanceof CVRef) {
			// No such state should remain in AutomatonState set after moveAutomatons call
			throw new RuntimeException("Unexpected method usage " + edgeType);
			// return false;
		} else // CVRef - Mutexes

		if (edgeType instanceof TBPResolvedCondition) {
			// No such state should remain in AutomatonState set after moveAutomatons call
			throw new RuntimeException("Unexpected method usage " + edgeType);
			// return false;
		} else // TBPResolvedCondition

		if (edgeType instanceof TBPResolvedAssignment) {
			final TBPResolvedAssignment edgeAssigment = (TBPResolvedAssignment)edgeType;
			
			// There are 2 types Assignments -> 
			//   a) from constant(variable, ...) 
			//   b) from Emit(result of another automaton) 

			TBPResolvedValue tbpValue = edgeAssigment.getValue();
			if (tbpValue.isReference()) {
				// a) from constant(variable, ...) ... not correct state
				// b) from LastCallRef ... not correct state
				
				// No such state should remain in AutomatonState set after moveAutomatons call
				throw new RuntimeException("Unexpected method usage " + edgeType);
				// return false;
			} else {
				// b) from Emit(result of another automaton) 
				// Method call
				TBPResolvedEmit calledMethod = tbpValue.getMethodCall();
				Binding calledMethodBinding = calledMethod.getBinding();
				
				if (event.type == EventItem.EventTypes.EVENT_CALL) {
					 if (calledMethodBinding.getFullname().equals(event.fullName)) {
							// Found expected calling node
							return true;
						} else {
							// Unwanted call ... calling different method
							return false;
						}
				} else if (event.type == EventItem.EventTypes.EVENT_RETURN) {
					// Return event when expecting call 
					return false;
				}
			} // End Method call 
		} else // TBPResolvedAssignment
		
		if (edgeType instanceof TBPResolvedEmit) {
			TBPResolvedEmit edgeEmit = (TBPResolvedEmit)edgeType;
			Binding calledMethodBinding = edgeEmit.getBinding();
			
			if (event.type == EventItem.EventTypes.EVENT_CALL) {
				if (calledMethodBinding.getFullname().equals(event.fullName)) {
					// Found expected calling node
					return true;
				} else {
					// Unwanted call ... calling different method
					//   stop processing
					return false;
				}
			} else if (event.type == EventItem.EventTypes.EVENT_RETURN) {
				// Return event ... find call edge
				// Unwanted call ... expecting return event
				//   stop processing
				return false;
			}
		} else // TBPResolvedEmit
		
		if (edgeType instanceof TBPResolvedUndefinedEmit) {
			TBPResolvedUndefinedEmit edgeUndefEmit = (TBPResolvedUndefinedEmit)edgeType;			
			MethodCall calledMethod = edgeUndefEmit.getMethodCall();

			if (event.type == EventItem.EventTypes.EVENT_CALL) {
				if (calledMethod.getFullname().equals(event.fullName)) {
					// Found expected calling node
					return true;
				} else {
					// Unwanted call ... calling different method
					//   stop processing
					return false;
				}
			} else if (event.type == EventItem.EventTypes.EVENT_RETURN) {
				// Return event ... find call edge
				// Unwanted call ... expecting return event
				//   stop processing
				return false;
			}
		} else // TBPResolvedUndefinedEmit

		
		if (edgeType instanceof TBPResolvedReturn) {
			if (event.type == EventItem.EventTypes.EVENT_CALL) {
				// Unwanted call ... expecting return
				//   stop processing
				return false;
			} else if (event.type == EventItem.EventTypes.EVENT_RETURN) {
				// Check if return from correct method
				final String fullMethodName = reverseEdgeMap.get(edge);
				assert fullMethodName != null; // Unknown edge
				
				if (fullMethodName.equals(event.fullName)) {
					// Return from expected method
					return true;
				} else {
					// Return from unexpected automaton
					return false;
				}
			}
		} else // TBPResolvedReturn
		
		{
			// The way how to found new unknown types of edges
			throw new RuntimeException(AutomatonsMoves.class.getSimpleName() + ":automatonProcessEdge - unknown edge type - e" + edgeType.getClass().toString());
		}

		return false;
	}
	
	/**
	 * Automatons that are in given automaton set are processed. 
	 *   From each {@link AutomatonState} that represents unprocessed action state 
	 *   ( {@link AutomatonState#getEdgeActionProcessed()} equals false)
	 *   makes new copy that represents processed action state and add them into result set.
	 *  <br> 
	 *  {@link VariableEvaluation} mappings are corrected. 
	 *  It means that points to newly created automaton state. 
	 *  Mappings to old {@link AutomatonState} are removed. 
	 *  <br>
	 *   {@link AutomatonState} from input that represents processed action state won't 
	 *   be in result.
	 *  
	 *  <br>Note: Used during return processing. 
	 *    This method change meaning {@link AutomatonState} that represent call,
	 *    from call not processed (not returned), into call processed (method called) 
	 *    (after return we can move forward from {@link AutomatonState}) 
	 *    
	 * @param currentAutomatonStates Set of automaton state to process.
	 * @return Gets set of "Unprocessed action states" from input converted into "Processed action"  
	 */
	public Set<AutomatonState> edgeActicionProcess(Set<AutomatonState> currentAutomatonStates) {
		Set<AutomatonState> result = new HashSet<AutomatonState>();
		
		for(AutomatonState as : currentAutomatonStates) {
			// Takes care only for calls that was not processed until now
			if (as.getEdgeActionProcessed() == false) {
				AutomatonState asProcessed = as.edgeActionProcessed();
				result.add( asProcessed);
				
				// Change variable mappings from old to new processed state
				Set<VariablesEvaluation> varEvals = varStatesMap.getMappings(as);
				for(VariablesEvaluation values : varEvals) {
					varStatesMap.setMappings(asProcessed, values);
				}
			}
		}
		return result;
	}
	

	static private Map<LTSAComponentSpecification, Map<Edge, String>> reverseEdgeMapCache = new HashMap<LTSAComponentSpecification, Map<Edge,String>>(1); /// Holds cache for processed reverse maps
	
	/**
	 * Process automatons and create reverse map. For each edge, create mapping it's source automaton. (Event associated with automaton).
	 * <br>
	 * More precisely: Returns map where is mapping for each Edge in referenced {@link LTSAAutomaton}s to name of event in form  
	 *   "InterfaceName.MethodName" for reactions, "." for thread section automaton.
	 * 
	 * <br>Note: This version only checks if not stored in {@link #reverseEdgeMapCache}, otherwise delegate work to {@link #createNewReverseEdgeMap(LTSAComponentSpecification)} to create new map that is stred into cache. 
	 * 
	 * @param spec Component specification where all {@link LTSAAutomaton}s to process are defined. Cann't be null.
	 * @return Map where is mapping for each Edge in referenced {@link LTSAAutomaton}s to name of event.  
	 */
	static public Map<Edge, String> createReverseEdgeMap(LTSAComponentSpecification spec) {
		assert spec != null;

		synchronized (reverseEdgeMapCache) {
			if (!reverseEdgeMapCache.containsKey(spec)) {
				// Mapping not created ... create new
				reverseEdgeMapCache.put(spec, createNewReverseEdgeMap(spec));
			}
			
			return reverseEdgeMapCache.get(spec);

		}
	}

	/**
	 * Process automatons and create reverse map. For each edge, create mapping it's source automaton. (Event associated with automaton).
	 * <br>
	 * More precisely: Returns map where is mapping for each Edge in referenced {@link LTSAAutomaton}s to name of event in form  
	 *   "InterfaceName.MethodName" for reactions, "." for thread section automaton.
	 * 
	 * <br>Note: Result reverse set is used in {@link AutomatonsMoves#filterStates(Set, EventItem)}
	 * 
	 * @param spec Component specification where all {@link LTSAAutomaton}s to process are defined. Cann't be null.
	 * @return Map where is mapping for each Edge in referenced {@link LTSAAutomaton}s to name of event.  
	 */
	static public Map<Edge, String> createNewReverseEdgeMap(LTSAComponentSpecification spec) {
		assert spec != null;
		
		Map<Edge, String> result = new HashMap<Edge, String>();

		// Process provisions
		for( Entry<Binding, LTSAAutomaton> reactionPair : spec.getReactions().entrySet()) {
			final LTSAAutomaton automaton = reactionPair.getValue();
			final String eventName = reactionPair.getKey().getFullname();
			
			createNewReverseEdgeMapProcessAutomaton(automaton, eventName, result);
		}

		// Process threads
		for( Entry<String, LTSAAutomaton> threadPair : spec.getThreads().entrySet()) {
			final LTSAAutomaton automaton = threadPair.getValue();
			final String eventName = ".";

			createNewReverseEdgeMapProcessAutomaton(automaton, eventName, result);
		}
		
		return result;
	}
	
	/**
	 * Visit all automatons edges and add them into result map with eventName value.
	 * <br>
	 * Note: Assuming that each edge is used only in one automaton.
	 * 
	 * @param automaton Automaton to explore. Cann't be null.
	 * @param eventName Name that will be associated to visited edges
	 * @param result Map where add mappings for visited edges. Cann't be null.
	 */
	private static void createNewReverseEdgeMapProcessAutomaton(
			LTSAAutomaton automaton, String eventName, Map<Edge, String> result) {
		assert automaton != null;
		assert result != null;
		Stack<State> statesToProcess = new Stack<State>();
		Set<State> processedStates = new HashSet<State>();
		
		statesToProcess.push(automaton.getStart());
		
		while(!statesToProcess.isEmpty()) {
			State state = statesToProcess.pop();
			// Check if not processed
			if (!processedStates.contains(state)) {
				// Not processed state
				processedStates.add(state);
				
				// Process each outgoing edges
				for(Edge e : state.getEdges()) {
					result.put(e, eventName);
					final State targetState = e.getTarget();
					if (!processedStates.contains(targetState)) {
						// Not processed -> plan to process 
						statesToProcess.push(targetState);
					}
				} // end for each outgoing edge
			} // end if Unprocessed edge
		} // End while state planned to process
	}
	
	/**
	 * Checks whether in given set of states exists processed final state.
	 * 
	 * @param states Set of automaton states to check. Cannot be null.
	 * @return True if processed final automaton state in given set exists, false otherwise.
	 */
	public static boolean isEndStateInSet(Set<AutomatonState> states) {
		assert states != null;
		for(AutomatonState as : states) {
			if (as.getAutomaton().getFin().equals(as.getState())) {
				return true;
			}
		}
		return false;
	}
}
