package org.ow2.dsrg.fm.tbpjava.checker;

import java.util.List;
import java.util.Set;
import java.util.Stack;

import org.ow2.dsrg.fm.tbplib.ltsa.Edge;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.State;

/**
 * Takes care about walking through automatons. Generates edges to process. 
 * Ensures that infinite cycle will not occur.
 * 
 *  <br>Note: Main methods are {@link #getNextStateToExplore(AutomatonWalker.EdgeToProcess)} and {@link #setStateToExplore(AutomatonState, VariablesEvaluation, State)}
 *  <br>Note: Currently implemented as DFS.
 *  <br>Note: Uses 2 support classes {@link AutomatonWalker.EdgeToProcess} that represents output, and {@link AutomatonWalker.AutomatonProcessingState} that is used for internal state represent
 */
public final class AutomatonWalker {
	// Implementation notes
	//   Stack statesToProcess doesn't contain APS entries (for states) with no outgoing edges. (Leaves)
	//   AutomatonProcessingState.processedStates properties always hold index of edge to process.
	
	/**
	 * Represent {@link Edge} in automatons that should be processed.
	 *  Output of {@link AutomatonWalker}.
	 * 
	 * Note: Used as structure. Without internal logics.
	 * Note: Setters are intended to use only from {@link AutomatonWalker}
	 */
	static public final class EdgeToProcess {
		private AutomatonState as; /// Automaton state, where current event processing started (stated saved before event processing stated) 
		private Edge edge; /// Edge to process
		private VariablesEvaluation variables; /// Variables values representing
		
		public EdgeToProcess() {
			this.as = null;
			this.edge = null;
			this.variables = null;
		}

		public EdgeToProcess(AutomatonState as, Edge edge, VariablesEvaluation variables) {
			this.as = as;
			this.edge = edge;
			this.variables = variables;
		}


		public final AutomatonState getAs() {
			return as;
		}


		public final Edge getEdge() {
			return edge;
		}


		public final VariablesEvaluation getVariables() {
			return variables;
		}


		public final void setAs(AutomatonState as) {
			this.as = as;
		}


		public final void setEdge(Edge edge) {
			this.edge = edge;
		}


		public final void setVariables(VariablesEvaluation variables) {
			this.variables = variables;
		}
		
	}
	
	/** 
	 * Internal structure of {@link AutomatonWalker} that holds (part of) state during automaton traversal.
	 * It represent one {@link State} that is intended for further processing. 
	 * It holds number of processed outgoing {@link Edge} from {@link State}
	 * 
	 * <br>Note: Not immutable, processedEdges can increase by calling  
	 */
	private static final class AutomatonProcessingState {
		private AutomatonState as; /// {@link AutomatonState} where processing of current path started. To this automaton state was given Variables mapped
		private State state; // State in {@link LTSAAutomaton} which instance represents. Note:Overrides state in "as" entry. We don't store this property into new {@link AutomatonState} to remove some allocations.
		private int processedEdges; /// Counter of processed (returned) outgoing edges from state. 
		private VariablesEvaluation variables; /// Variables evaluation associated to state.
		
		
		public AutomatonProcessingState(AutomatonState as, VariablesEvaluation variables, State state) {
			assert as != null;
			assert state != null;
			
			this.as = as;
			this.state = state;
			this.variables = variables;
			this.processedEdges = 0;
		}

		public final AutomatonState getAs() {
			return as;
		}

		public final State getState() {
			return state;
		}

		public final int getProcessedEdges() {
			return processedEdges;
		}

		/**
		 * Only for internal purposes of {@link AutomatonWalker}. Not intended to be call by user.
		 *
		 * Marks that another edge with origin in state {@link #state} is processed.
		 * It means adds one to {@link #processedEdges}
		 * 
		 * Note: Throws RuntimeException if called more times than there are edges from {@link #state}. 
		 */
		public final void nextEdgeProcessed() {
			assert processedEdges < state.getEdges().size();
			
			if (processedEdges >= state.getEdges().size()) {
				throw new RuntimeException("Error in processing state automaton states - some edges processed multiple times");
			}
			processedEdges++;
		}

		public final VariablesEvaluation getVariables() {
			return variables;
		}
		
		
		//TODO StatesMapper -> check whether mapping are same. Equality of variables evaluation not enough.
		/**
		 * Check whether given object represents same state during processing 
		 *  this instance.
		 * <br>
		 * Note: {@AutomatonProcessingState} during processing move in automatons
		 *  are same if for path during processing start in same {@link AutomatonState},
		 *  now are in same {@link State} of {@link LTSAAutomaton} (paths can be different),
		 *  and {@link VariablesEvaluation} has same values. 
		 *  (not needed to be equal, equality has different meaning)
		 * <br>Note: The most clean solution for {@link VariablesEvaluation} is to check
		 *  {@link VariablesEvaluation} has same values and same associated mapped 
		 *  {@link AutomatonState} in {@link StatesMapper}. 
		 * 
		 * @param other Object to test whether is not similar.
		 * @return true if sates can be assumed same from {@link AutomatonWalker} point of view, false otherwise
		 */
		public boolean isSame(AutomatonWalker.AutomatonProcessingState other) {
			assert other != null;
			
			if (other == this) {
				return true;
			}
			
			return (as == other.as) && (state == other.state) && (variables.equalsValues(other.variables));
		}
		
	}
	
	private Stack<AutomatonWalker.AutomatonProcessingState> statesToProcess; /// List of states for further processing
	
	public AutomatonWalker() {
		statesToProcess = new Stack<AutomatonWalker.AutomatonProcessingState>();
	}
	
	/**
	 * Adds given pair state/variables into list of states to explore (process outgoing edges).
	 *
	 * Later will be outgoing edges (of given state) returned by {@link #getNextStateToExplore(EdgeToProcess)}
	 *  (with same Variables and same AutomatonState). 
	 *  
	 * @param sourceAS {@link AutomatonState} where processing of current path started. 
	 * @param variables Variables evaluation associated to state.
	 * @param state State which edges are planned for further processing
	 */
	public void setStateToExplore(AutomatonState sourceAS, VariablesEvaluation variables, State state) {
		// Check whether state has any descendants (child) 
		//  if not, there is no reason why to save state for further processing
		if (state.getEdges().isEmpty() == true) {
			return;
		}
		
		// Don't plan processing of final states. At final state should processing stop. 
		//  (Notice, typically final edge hasn't any successor, so first check remove them too)
		if (sourceAS.getAutomaton().getFin().equals(state)) {
			return;
		}
		
		AutomatonWalker.AutomatonProcessingState newState = new AutomatonProcessingState(sourceAS, variables, state);

		if (cycleDetection(newState)) {
			// No cycle, can be added
			statesToProcess.push(newState);
		}
	}
	
	/** 
	 * Adds given pair edge/variables into to be explore list.
	 *
	 * Later will be given edge returned by {@link #getNextStateToExplore(EdgeToProcess)}
	 *  (with same Variables and same AutomatonState).
	 * 
	 * <br>Note: Internally creates internal {@link State} that contains only the one given edge,
	 *    and add them into states to be processed. This internal {@link State} never leak outside
	 *    {@link AutomatonWalker}.
	 * <br>Note: Method is used for processing of mutexes. (In unprocessed state).
	 *   @see AutomatonsMoves#moveAutomatonsEnterMutexes(Set)
	 * 
	 * @param sourceAS {@link AutomatonState} where processing of current path started.
	 * @param variables Variables evaluation associated to edge.
	 * @param edge Edge to be planed for further processing.
	 */ 
	public void setEdgeToExplore(AutomatonState sourceAS, VariablesEvaluation variables, Edge edge) {
		// Create new virtual {@link State} with only one edge (given edge). This state is then set to be processed. 
		//  New virtual state cannot leave  AutomatonWalker !!!
		State newState = new State();
		newState.addEdge(edge); // TBPLib cannot check that edge.source equal to state we edge add.
		
		setStateToExplore(sourceAS, variables, newState);
	}
	
	/**
	 * Get edges that should be processed.
	 * 
	 * @param result Instance where store result. If null new object is allocated. 
	 * @return object that represents edge that should be processed (in {@link AutomatonsMoves#automatonProcessEdge(Set, AutomatonWalker.EdgeToProcess, EventItem)})
	 *   null if there is no edge for processing.  
	 */
	public AutomatonWalker.EdgeToProcess getNextStateToExplore(AutomatonWalker.EdgeToProcess result) {
		if (statesToProcess.isEmpty()) {
			// Nothing where start exploring, work done
			return null;
		}
		
		AutomatonWalker.AutomatonProcessingState startAPS = statesToProcess.peek();
		List<Edge> edges = startAPS.getState().getEdges(); // Edges of state selected for processing ... startAPS->state

		// prepare return value
		if (result == null) {
			// Not given item, where save state ... create new
			result = new EdgeToProcess();
		}
		result.setAs(startAPS.getAs());
		result.setEdge(edges.get( startAPS.getProcessedEdges()));
		result.setVariables(startAPS.getVariables());

		// There will remain another unprocessed edge, update only processed edge counter.
		startAPS.nextEdgeProcessed();
		
		// Check how to actualize internal informations about processed edges of state
		if (edges.size() == startAPS.getProcessedEdges()) {
			// Last unprocessed edge
			// Remark: startAPS->state has same outgoing edge
			// Method returns last edge of startAPS->state ... not another state to process ... remove from states ready to process 
			statesToProcess.pop();
		}
		
		return result;
	}
	
	/**
	 * Removes all states that are planed to explore. So no edge is here to process.
	 * After call AutomatonWalker behave as if newly created.  
	 */
	public void clean() {
		statesToProcess.clear();
	}
	
	/**
	 * Check if {@link AutomatonProcessingState} can be added into 
	 *  set of states planned for further processing.
	 * 
	 * @param as State you tries to add.
	 * @return True if state can be added into stack and no cycle occur, false if state currently saved in stack (cycle occur)
	 */
	private boolean cycleDetection(AutomatonWalker.AutomatonProcessingState as) {
		for(AutomatonWalker.AutomatonProcessingState savedAS : statesToProcess) {
			if (savedAS.isSame(as)) return false;
		}
		return true;
	}
	
}