package org.ow2.dsrg.fm.tbpjava.checker;


import org.ow2.dsrg.fm.tbplib.ltsa.Edge;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.State;
//Note: State corresponds to states in nondeterministic automatons after 
//execution same (some) sequence of transitions. 

/**
* Structure that holds one automaton state. 
* <br>
* Note: Immutable class.
*/
public final class AutomatonState {
	// To be automaton state unique in whole checking system have to contains thread number too.
	final private int threadNum;
	
	final private LTSAAutomaton automaton; /// Automaton that state belongs
	final private State state; /// State in automaton that represents
	final private Edge edge; /// Edge we get into state (it's needed for call/return processing)

	/**
	 *  Holds false if {@link AutomatonState#getEdge()} action is not currently processed (emit==call that don't return), return event not processed, enter into mutex but not change mappings, variables not modified.
	 *  Holds true if action in {@link AutomatonState#getEdge()} is done (call processed -> returned from call, ....)
	 */
	final private boolean edgeActionProcessed; /// Mark if action stored in edge was processed (true) or wait to process (false) (call not returned, enter to mutex). If edge null, holds true.
	
	final private int hashCode;
	final private int HASH_CONSTANT = 37; 

	
	/**
	 * Should be used only for creating new start entry while processing provided/required call.
	 * Edge is set to null.
	 *
	 * Note: No incoming edge -> no action to process -> {@link #edgeActionProcessed} is set to true
	 *  
	 * @param automaton Automaton that holds stored stated.
	 * @param state State in automaton that is represented. Cann't be null. Should by start state of {@link #automaton}. 
	 * @param threadNum Number of thread which caused event.
	 */
	public AutomatonState(LTSAAutomaton automaton, State state, int threadNum) {
		assert state != null;
		
		this.automaton = automaton;
		this.state = state;
		this.threadNum = threadNum;
		
		this.edge = null;
		this.edgeActionProcessed = true;

		hashCode = calculateHashCode();
	}

	/**
	 * Create automaton state from another given. Property {@link #state) is set as edge {@link Edge#getTarget()}.
	 * Is intended to storing state during automaton traversal. Especially is needed to set edge before Emit(Call).
	 * 
	 * @param source Original automaton
	 * @param edge Edge to set. Modify state to edge target. Edge cann't be null.
	 * @param edgeActionProcessed Flag that marks whether was action associated with given edge processed or not.
	 */
	public AutomatonState(AutomatonState source, Edge edge, boolean edgeActionProcessed) {
		assert edge != null;
		
		this.automaton = source.automaton;
		this.state = (edge==null ? null : edge.getTarget());
		this.threadNum = source.threadNum;
		this.edge = edge;
		this.edgeActionProcessed = edgeActionProcessed;
		
		hashCode = calculateHashCode();
	}
	
	/**
	 * Internal constructor that can set all entries. 
	 * 
	 * <br>Note: Handle with case. Can set properties inconsistently.  
	 */
	private AutomatonState(int threadNum, LTSAAutomaton automaton, State state, Edge edge, boolean edgeActionProcessed) {
		assert (edge == null || state.equals( edge.getTarget())); /// State is target of edge
		
		this.threadNum = threadNum;
		this.automaton = automaton; /// Automaton state belongs to
		this.state = state; /// State in automaton that represents
		this.edge = edge; /// Edge we get into state (it's needed for call/return processing)
		this.edgeActionProcessed = edgeActionProcessed; /// Mark if action stored in edge was processed (true) or wait to process (false) (call not returned, enter to mutex). If edge null, holds true.
		hashCode = calculateHashCode();
	}
	
	/**
	 * Gets clone of instance but action on edge is marked as processed.
	 * 
	 * <br>Note: Assumes that {@link #edgeActionProcessed} is set to false.
	 * 
	 * @return Instance copy but action associated with edge is processed.  
	 */
	public AutomatonState edgeActionProcessed() {
		assert edgeActionProcessed == false;
		
		return new AutomatonState(threadNum, automaton, state, edge, true);
	}

	public int hashCode() {
		return hashCode;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		if (hashCode != obj.hashCode()) {
			return false;
		}

		AutomatonState other = (AutomatonState) obj;
		if (state == null) {
			if (other.state != null)
				return false;
		} else if (!state.equals(other.state))
			return false;

		if (threadNum != other.threadNum) {
			return false;
		}

		return true;
	}

	/**
	 * @return the threadNum
	 */
	public final int getThreadNum() {
		return threadNum;
	}

	/**
	 * @return the automaton
	 */
	public final LTSAAutomaton getAutomaton() {
		return automaton;
	}

	/**
	 * @return the state
	 */
	public final State getState() {
		return state;
	}

	/**
	 * @return the edge
	 */
	public final Edge getEdge() {
		return edge;
	}
	
	public final boolean getEdgeActionProcessed() {
		return edgeActionProcessed;
	}
	
	/**
	 * Convert internal state into string.
	 * @return String representing automaton state in human readable form.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append('[');
		sb.append("state=");
		  sb.append(state);
		sb.append(", automaton=");
		  sb.append(automaton);
//		sb.append(", threadNum=");
//		  sb.append(threadNum);
		sb.append(", edge=");
		  sb.append(edge);
		  sb.append("[" + (edge==null?"null":edge.getData()) + "]");
//		sb.append(", bindings=");
//		  sb.append(bindings);
//		sb.append(", edgeActionProcessed=");
//		  sb.append(edgeActionProcessed);
//		sb.append(", hashCode=");
//		  sb.append(hashCode);
		sb.append(']');
		return sb.toString();
	}
	
	/**
	 * Calculate hash code for object.
	 * 
	 * Note: All properties (except {@link #hashCode}) has to be set before this call.
	 * 
	 * @return hashCode value.
	 */
	final private int calculateHashCode() {
		int result = 0;
		result = (result * HASH_CONSTANT) + (edgeActionProcessed  ? 1 : 0);
		result = (result * HASH_CONSTANT) + (edge !=null ? edge.hashCode() : 0);
		result = (result * HASH_CONSTANT) + state.hashCode();
		result = (result * HASH_CONSTANT) + threadNum;
		return result;
	}

}
