package org.ow2.dsrg.fm.tbpjava;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.antlr.runtime.RecognitionException;
import org.ow2.dsrg.fm.tbpjava.envgen.CodeGenerator;
import org.ow2.dsrg.fm.tbpjava.envgen.EnvValueSets;
import org.ow2.dsrg.fm.tbpjava.envgen.Indenter;
import org.ow2.dsrg.fm.tbpjava.envgen.ProvisionToString;
import org.ow2.dsrg.fm.tbpjava.envgen.StubGenerator;
import org.ow2.dsrg.fm.tbpjava.envgen.VisitorTypeCheck;
import org.ow2.dsrg.fm.tbpjava.envgen.ProvisionToString.Style;
import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbpjava.utils.LineCouningOutputStream;
import org.ow2.dsrg.fm.tbpjava.utils.Type2String;
import org.ow2.dsrg.fm.tbplib.EventTable;
import org.ow2.dsrg.fm.tbplib.EventTableImpl;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.parsed.ParsedComponentSpecification;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAccept;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedProvisionContainerNode;
import org.ow2.dsrg.fm.tbplib.parser.ParserFacade;
import org.ow2.dsrg.fm.tbplib.resolved.ResolvedComponentSpecification;


/**
 * Public main class intended for generating environment for Threaded Behavior Protocols compliance checking. 
 * 
 * @author Alfifi
 *
 */
//TODO Refactor .... hide logics, decompose the separate methods SOFA, FRACTAL .. SEPARATE CLASSES ( see configuration) into separate classes
public class EnvGenerator {
	private static final boolean DEBUG = false; // was true

	/// String used to separate name of component and provision in class name of generated environment. 
	public static final String CLASS_NAME_SEPARAROT_COMPONENT_PROVISION = "_";

	/// Property name in generated class where instance of checked component is stored. 
	public static final String PROPERTY_COMPONENT_INSTANCE = "comp";

	public static final String PROPERTY_INTERFACES_PREXIF = "iface_";
	
	public static final String METHOD_GENERATED_PROTOCOL_NAME = "runEnvironment";

	public static final String SOFA_SERVER_FULL_NAME = "org.objectweb.dsrg.sofa.SOFAServer";
	public static final String SOFA_CLIENT_FULL_NAME = "org.objectweb.dsrg.sofa.SOFAClient";
	public static final String FRACTAL_BINDING_CONTROLER_FULL_NAME = "org.objectweb.fractal.api.control.BindingController";

	public static final String METHOD_MAIN_INSTANCE_NAME = "inst";
	
	/// Default name for provisions without given name by user. Used for generating result class and file name
	public static final String PROVISIONS_ANONYMOUS_SUFFIX = "anon_";

	public static final boolean EXCEPTIONS_STOP_ON_EXCEPTION = true; /// If true, all uncatch exception are thrown out of program as RuntimeException,  if false all uncatch expeptions are printed to standard output. (and execution continues -> ends)  

	private int cntAnonynousProvisions = 0; /// Counts processed anonymous provisions. 
	
	private PrintStream infoStream = null;
	private Configuration config = null;
	
	//private Map<String, String> provItfsPropertiesNames = new HashMap<String, String>();
	
	// Holds tree with component specification
	private ParsedComponentSpecification spec = null;
	
	/**
	 * Holds information about generated environment needed for checking
	 *
	 */
	public static class EnvironmentDescription {
		public String mainClassName; /// Class name where executable environment takes place (where main method takes place)

		public Map<String, String> provItfs2envFieldName; /// Maps provided interface names from the TBP specification to field in generated environment where instance of the interface is stored.
		public Map<String, String> compImpl2envFieldName; /// Maps component implementation classes to fields into the environment where theirs instances are stored
		
		/**
		 * Maps required interfaces to fields to we have to set required interface. 
		 *   (by methods specified in {@link Configuration.RequiredInterfaceSettingMethod}
		 */
		public Map<String, String> reqItfs2envFielName;
		public Map<Integer, TBPParsedAccept> line2ProvidedCall;
		
		public EnvironmentDescription() {
			provItfs2envFieldName = new HashMap<String, String>();
			compImpl2envFieldName = new HashMap<String, String>();
			reqItfs2envFielName = new HashMap<String, String>();
			line2ProvidedCall = new HashMap<Integer, TBPParsedAccept>();
		}
	}

	public EnvGenerator(Configuration config, PrintStream outStream) {
		this.config = config;
		this.infoStream = outStream;
	}
	
	/**
	 * Getter for spec property.
	 * @return Returns currently processed TBP specification. Value is valid after call {@link #generateEnvironment(TBPParsedProvisionContainerNode, Map)}
	 */
	public ParsedComponentSpecification getComponentSpecification() {
		return spec;
	}

	/**
	 * Gets processed component specification in form of LTSA automatons.
	 * @return Gets processed component specification.
	 */
	public LTSAComponentSpecification getComponentLTSA() {
		EventTable et = new EventTableImpl();
		ResolvedComponentSpecification resolve = spec.resolve(et);			
		LTSAComponentSpecification result = resolve.makeLTSA(et);
		return result;
	}

	
	/**
	 * 
	 * @return Map contains for each generated environment mapping from generated class name to (provided interface implementation class map)
	 */
	public Map<String, EnvironmentDescription> generateEnvironments() {
		spec = null;
		try {
			spec = ParserFacade.parseFile( config.getEnvProtocol());
		} catch (IOException e) {
			infoStream.println("Invalid input file name");
			if (DEBUG) { infoStream.println(e); e.printStackTrace(infoStream); }
			return null;
		} catch (RecognitionException e) {
			infoStream.println("Syntax error in specification file");
			if (DEBUG) { infoStream.println(e); e.printStackTrace(infoStream); }
			return null;
		}
		
		// Check protocol for errors
		VisitorTypeCheck specCheck = new VisitorTypeCheck(config, infoStream);
		if (specCheck.completeCheck(spec)) {
			// Error in specification found
			return null;
		}
		
		Map< String, EnvironmentDescription> result = new HashMap< String, EnvironmentDescription>();

		// Checks and prepare operations that should be done only once (to limit number of reported errors)
		spec.limitReentrancy( config.getComponentReentrancyLimit());

		List<TBPParsedProvisionContainerNode> provisions = spec.getProvisions();
		for( TBPParsedProvisionContainerNode node : provisions) {
			infoStream.println("Processing " + node.getName() + " provision.");
			EnvironmentDescription ed = generateEnvironment(node); 
			if (ed != null) {
				result.put(ed.mainClassName, ed);
			}
		}
		
		return result;
	}
	
	/**
	 * Tests if given class has public constructor without parameters.
	 * @param className Name of class to test
	 * @param notFoundWarning Prints warning (true) or error (false) if class cannot be found. 
	 */
	@SuppressWarnings("unused")
	private void checkExistenceNonparametricConstructor(String className, boolean notFoundWarning) {
		// Obtaining target class
		Class<?> checkedClass = null;
		try {
			checkedClass = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(className));
		} catch (ClassNotFoundException e) {
			if (notFoundWarning) {
				infoStream.println("Info - Class " + Type2String.removeGenerics(className) + " cannot be found. Cann't check existence of nonparametric constructor.");
			} else {
				infoStream.println("Error - Class " + Type2String.removeGenerics(className) + " cannot be found.");
			}
			return;
		}
		
		// Check for constructor
		try {
			checkedClass.getConstructor();
		} catch (SecurityException e) {
			if (notFoundWarning) {
				infoStream.println("Info - Class " + className + " cannot accessed (Security manager blocks access). Cann't check existence of nonparametric constructor.");
			} else {
				infoStream.println("Error -  Class " + className + " cannot accessed (Security manager blocks access).");
			}
			if (DEBUG) { e.printStackTrace(infoStream); }
		} catch (NoSuchMethodException e) {
			infoStream.println("Error - Class " + className + " has to have a non-parametric constructor.");
		}
	}

	
	/**
  	 * @param mappings For generated environment class  adds mapping from generated environment class full name to map of implementation classes for provided interfaces.  
	 */
	private EnvironmentDescription generateEnvironment(TBPParsedProvisionContainerNode node) {
		if (node == null) {
			return null;
		}

		String provisionName = node.getName();
		if (provisionName == null) { // No user given provision name .... generate default name
			cntAnonynousProvisions++;
			provisionName = PROVISIONS_ANONYMOUS_SUFFIX + Integer.toString(cntAnonynousProvisions);
		}

		// Name of generated class that tests given provision
		String envClassName = config.getComponentName() + "_" + provisionName; /// Simple name of class (without package part)
		String envClassFullName = envClassName; /// Full name of generated class with package part
		
		
		String pkgComponent = config.getComponentImplementationClasses().get(0); // We choice first implementation class as a base package.
		pkgComponent = StubGenerator.getPackage(Type2String.removeGenerics(pkgComponent));
		if (pkgComponent != null) {
			envClassFullName = pkgComponent + "." + envClassName;
		}
		
		// Name of file with generated class
		String envFileName = config.getEnvTargetDir() + CodeGenerator.patchName(envClassFullName.replace('.', '/')) + ".java";

		// Maps provided interfaces with class that implements that interface.
		EnvironmentDescription result = new EnvironmentDescription();
		result.mainClassName = envClassFullName;
		
		// Generate names for properties
		prepareEnvironmentDescription(true, result);

		// Name of generated class that implements provision
		if (config.getEnvGenerate()) {
			LineCouningOutputStream outFile = null;
			try {
				StubGenerator.generateDirectoryStructureForFile(envFileName);
				outFile = new LineCouningOutputStream(envFileName);
			} catch (FileNotFoundException e) {
				infoStream.println("Error - File (" + envFileName + ") creation error.");
				if (DEBUG) { e.printStackTrace(infoStream); }
				return null;
			}

			infoStream.println( new ProvisionToString(Style.STYLE_COMPACT, false).visitParsedProvisionContainerNode(node));
			
			Indenter indenter = new Indenter(outFile.getPrintStream());
			CodeGenerator provisionParser = new CodeGenerator(infoStream, outFile, indenter, result, config);

			genInitialClassCode(outFile, indenter, envClassName, envClassFullName, pkgComponent);
			genProperties(outFile, indenter, result);
			genConstructor(outFile, indenter, envClassName, result);
			genProtocolCode(outFile, indenter, provisionParser, node, result);
			genMain(outFile, indenter, envClassName);
			genEndingClassCode(outFile, indenter);
			outFile.close();
		} else {
			// Don't generate environment files, only fill mappings
			genConstructor(null, null, envClassName, result);
		}
		
		return result;
	}

	private boolean prepareEnvironmentDescription(boolean notFoundWarning, EnvironmentDescription envDesc) {
		boolean result = prepareEnvironmentDescriptionProvidedInterfaces(notFoundWarning, envDesc);
		if (result == false) {
			result = generateEnvironmentDescriptionRequiredInterfaces(notFoundWarning, envDesc);
		}
		return result;
	}
	
	/**
	 *  Fills {@link EnvironmentDescription#reqItfs2envFielName} field.
	 * @param notFoundWarning
	 * @param envDesc Environment to update
	 * @return TReu if error is found.
	 */
	private boolean generateEnvironmentDescriptionRequiredInterfaces(
			boolean notFoundWarning, EnvironmentDescription envDesc) {
		
		Configuration.RequiredInterfaceSettingMethod method = config.getEnvRequierdItfcSettingMethod();

		switch(method) {
			case SETTERS: {
				Map<String, Class<?>> interfaceNameToClass = new HashMap<String, Class<?>>();
				for(Entry<String, String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
					Class<?> itfsImplClass = getClass(reqItfsEntry.getValue(), "Component required interface " + reqItfsEntry.getValue() + "not found. Cannot verify setters of the component", "", false);
					if (itfsImplClass == null) {
						continue;
					}
					if (!itfsImplClass.isInterface()) {
						infoStream.println("Error - " + reqItfsEntry.getValue() + " is not interface. Wrong definition of required interfaces. Check configuration.");
						continue;
					}
					interfaceNameToClass.put(reqItfsEntry.getKey(), itfsImplClass);
				}
				
				// Check if error not shown
				if (interfaceNameToClass.size() != config.getComponentRequiredInterfaces().size()) {
					return true;
				}
				
				// Check all component classes whether it is possible to set required interface - fill fields
				for(String compClassName : config.getComponentImplementationClasses()) {
					String compFieldName = envDesc.compImpl2envFieldName.get(compClassName);
					
					Class<?> compImplClass = getClass(compClassName, "Component class " + compClassName + " not found. Cannot create component to required interface mapping", "", notFoundWarning);
					if (compImplClass == null) {
						continue; // Assume that no interfaces are required by this class error is reported later.
					}
					
					for(Entry<String,String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
						Class<?> reqItfsClass = interfaceNameToClass.get(reqItfsEntry.getKey());
						try {
							Method setter = compImplClass.getMethod("set"+reqItfsEntry.getKey(), reqItfsClass);
							
							if (setter == null) {
								continue;
							}
							
							int modifiers = setter.getModifiers();
							if (!Modifier.isAbstract(modifiers) && Modifier.isPublic(modifiers)) {
								if (envDesc.reqItfs2envFielName.containsKey(reqItfsEntry.getKey())) {
									// Multiple provided interfaces of single type - prohibited in this mode
									infoStream.println("Error - Setter for " + reqItfsEntry.getKey() + " is implemented by more component classes. Only a single instance for each provided interface type is permitted.");
									return true;
								} else {
									// This class implements current interface
									envDesc.reqItfs2envFielName.put(reqItfsEntry.getKey(), compFieldName);
								}
							}
						} catch (NoSuchMethodException e) {
							// OK, may be some other component class
						}
					}
				}
				
				if (envDesc.reqItfs2envFielName.size() != config.getComponentRequiredInterfaces().size()) {
					// Not all provided interfaces are implemented
					for(Entry<String, String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
						if (!envDesc.reqItfs2envFielName.containsKey(reqItfsEntry.getKey())) {
							infoStream.println("Error - " + reqItfsEntry.getKey() + " interface cannot be set for any component. No public set" + reqItfsEntry.getKey() + "(" + reqItfsEntry.getValue() + ") method found.");
						}
					}
					return true;
				}
				break;
			}
			case SOFA2_CLIENT: {
				// Check if the primary component class implements SOFAClientInterface
				String primaryComponentClassName = config.getComponentImplementationClasses().get(0);
				String primaryCompFieldName = envDesc.compImpl2envFieldName.get(primaryComponentClassName);
				assert primaryCompFieldName != null; // Should be set in prepareEnvironmentDescriptionProvidedInterfaces
				
				for(String reqItfs : config.getComponentRequiredInterfaces().keySet()) {
					envDesc.reqItfs2envFielName.put(reqItfs, primaryCompFieldName);
				}
				
				// Check if given component implements all provided interfaces or SOFA2Server for finding provided interfaces.
				Class<?> checkedComponentClass = getClass(primaryComponentClassName, "Component class " + primaryComponentClassName + " not found", "Cannot check whether component implements SOFAClient interfaces.", notFoundWarning);
				if (checkedComponentClass == null) {
					return !notFoundWarning;
				}

				Class<?> ifaceSofaClientClass = getClass(SOFA_CLIENT_FULL_NAME, "Interface " + SOFA_CLIENT_FULL_NAME + " not found.", "Cannot check that interface is implemented by component.", notFoundWarning);
				if (ifaceSofaClientClass == null) {
					return !notFoundWarning;
				}

				// Check whether imports given interface 
				if (!ifaceSofaClientClass.isAssignableFrom(checkedComponentClass)) {
					infoStream.println("Error - Interface " + SOFA_CLIENT_FULL_NAME + " is not implemented by component.");
					return true;
				}
				break;
			}
			case FRACTAL: {
				// Check if the primary component class implements SOFAClientInterface
				String primaryComponentClassName = config.getComponentImplementationClasses().get(0);
				String primaryCompFieldName = envDesc.compImpl2envFieldName.get(primaryComponentClassName);
				assert primaryCompFieldName != null; // Should be set in prepareEnvironmentDescriptionProvidedInterfaces
				
				for(String reqItfs : config.getComponentRequiredInterfaces().keySet()) {
					envDesc.reqItfs2envFielName.put(reqItfs, primaryCompFieldName);
				}
				
				// Check if given component implements all provided interfaces or SOFA2Server for finding provided interfaces.
				Class<?> checkedComponentClass = getClass(primaryComponentClassName, "Component class " + primaryComponentClassName + " not found", "Cannot check whether component implements Fractor BindingControler interfaces.", notFoundWarning);
				if (checkedComponentClass == null) {
					return !notFoundWarning;
				}

				Class<?> ifaceFractalBindingControler = getClass(FRACTAL_BINDING_CONTROLER_FULL_NAME, "Interface " + FRACTAL_BINDING_CONTROLER_FULL_NAME + " not found.", "Cannot check that interface is implemented by component.", notFoundWarning);
				if (ifaceFractalBindingControler == null) {
					return !notFoundWarning;
				}

				// Check whether imports given interface 
				if (!ifaceFractalBindingControler.isAssignableFrom(checkedComponentClass)) {
					infoStream.println("Error - Interface " + FRACTAL_BINDING_CONTROLER_FULL_NAME + " is not implemented by component.");
					return true;
				}
				break;
			}
			case FIELDS: {
				Map<String, Class<?>> interfaceNameToClass = new HashMap<String, Class<?>>();
				for(Entry<String, String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
					Class<?> itfsImplClass = getClass(reqItfsEntry.getValue(), "Component required interface " + reqItfsEntry.getValue() + "not found. Cannot fields of the component", "", false);
					if (itfsImplClass == null) {
						continue;
					}
					if (!itfsImplClass.isInterface()) {
						infoStream.println("Error - " + reqItfsEntry.getValue() + " is not interface. Wrong definition of required interfaces. Check configuration.");
						continue;
					}
					interfaceNameToClass.put(reqItfsEntry.getKey(), itfsImplClass);
				}
				
				// Check if error not shown
				if (interfaceNameToClass.size() != config.getComponentRequiredInterfaces().size()) {
					return true;
				}

				// Check all component classes whether it is possible to set required interface - fill fields
				for(String compClassName : config.getComponentImplementationClasses()) {
					String compFieldName = envDesc.compImpl2envFieldName.get(compClassName);
					
					Class<?> compImplClass = getClass(compClassName, "Component class " + compClassName + " not found. Cannot create component to required interface mapping", "", notFoundWarning);
					if (compImplClass == null) {
						continue; // Assume that no interfaces are required by this class error is reported later.
					}
					
					for(Entry<String,String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
						Class<?> reqItfsClass = interfaceNameToClass.get(reqItfsEntry.getKey());
						try {
							Field field = compImplClass.getField(reqItfsEntry.getKey());
							
							if (field == null) {
								continue;
							}
							
							int modifiers = field.getModifiers();
							if (!Modifier.isPublic(modifiers)) {
								continue;
							}
							
							if (!field.getType().isAssignableFrom(reqItfsClass)) {
								continue;
							}
							if (envDesc.reqItfs2envFielName.containsKey(reqItfsEntry.getKey())) {
								// Multiple provided interfaces of single type - prohibited in this mode
								infoStream.println("Error - Field " + reqItfsEntry.getKey() + " can be found in more compoent classes. Only one public field with this name is permited within all component classes.");
								return true;
							} else {
								// This class implements current interface
								envDesc.reqItfs2envFielName.put(reqItfsEntry.getKey(), compFieldName);
								continue;
							}
						} catch (NoSuchFieldException e) {
							// OK, may be some other component class
						}
						
						// Check if environment class is in a same package as component class to be able to use default visibility 
						String compPkg = StubGenerator.getPackage(compClassName);
						String envPkg = StubGenerator.getPackage( envDesc.mainClassName); // Package of the generated class
						if (! ((compPkg == null && envPkg == null) /*Both in default package*/ || ( compPkg != null && compPkg.equals(envPkg)))) {
							continue;
						}
						// Try default if package are same (not checks predecessors)
						try {
							Field field = compImplClass.getDeclaredField(reqItfsEntry.getKey());
							
							int modifiers = field.getModifiers();
							if (Modifier.isPrivate(modifiers) || Modifier.isProtected(modifiers)) {
								continue;
							}
							if (!field.getType().isAssignableFrom(reqItfsClass)) {
								continue;
							}
							if (envDesc.reqItfs2envFielName.containsKey(reqItfsEntry.getKey())) {
								// Multiple provided interfaces of single type - prohibited in this mode
								infoStream.println("Error - Field " + reqItfsEntry.getKey() + " can be found in more compoent classes. Only one public field with this name is permited within all component classes.");
								return true;
							} else {
								// This class implements current interface
								envDesc.reqItfs2envFielName.put(reqItfsEntry.getKey(), compFieldName);
								continue;
							}
						} catch (NoSuchFieldException e) {
							// OK, may be some other component class
						}
						
					}
				}
				
				if (envDesc.reqItfs2envFielName.size() != config.getComponentRequiredInterfaces().size()) {
					// Not all provided interfaces are implemented
					for(Entry<String, String> reqItfsEntry : config.getComponentRequiredInterfaces().entrySet()) {
						if (!envDesc.reqItfs2envFielName.containsKey(reqItfsEntry.getKey())) {
							infoStream.println("Error - " + reqItfsEntry.getKey() + " interface cannot be set for any component. No component has public field with " + reqItfsEntry.getKey() + " name and " + reqItfsEntry.getValue() + " type.");
						}
					}
					return true;
				}
				break;
			}
			default:
				throw new RuntimeException("Internal error - Unknown value of enum " + Configuration.RequiredInterfaceSettingMethod.class.getName() + " value is" + method); 
		}
		return false;
	}

	/**
	  * @param notFoundWarning If class with interface cann't be found. Print warning (true) or error (false)
	  * @return True if error is found.
	  */
	private boolean prepareEnvironmentDescriptionProvidedInterfaces(boolean notFoundWarning, EnvironmentDescription result) {

		if (config.getEnvProvidedItfsAccesMethod() == Configuration.ProvidedInterfaceAccessMethod.COMPONENT_CLASSES) {
			// GenerateMapping
			//   for each component impl class create separate field
			for(String componentClassName : config.getComponentImplementationClasses()) {
				result.compImpl2envFieldName.put( componentClassName, PROPERTY_INTERFACES_PREXIF + CodeGenerator.patchName(componentClassName));
			}
			
			Map<String, Class<?>> interfaceNameToClass = new HashMap<String, Class<?>>();
			for(Entry<String, String> provItfsPair : config.getComponentProvidedInterfaces().entrySet()) {
				Class<?> itfsImplClass = getClass(provItfsPair.getValue(), "Component provided interface " + provItfsPair.getValue() + "not found. Cannot create component classes to provided interface mapping", "", false);
				if (itfsImplClass == null) {
					continue;
				}
				if (!itfsImplClass.isInterface()) {
					infoStream.println("Error - " + provItfsPair.getValue() + " is not interface. Wrong definition of provided interfaces. Check configuration.");
					continue;
				}
				interfaceNameToClass.put(provItfsPair.getKey(), itfsImplClass);
			}
			// Check if error not shown
			if (interfaceNameToClass.size() != config.getComponentProvidedInterfaces().size()) {
				return true;
			}

			// Check whether all component provide all interfaces - fill fields
			for(String compClassName : config.getComponentImplementationClasses()) {
				String compFieldName = result.compImpl2envFieldName.get(compClassName);
				
				Class<?> compImplClass = getClass(compClassName, "Component class " + compClassName + " not found. Cannot create component to provided interface mapping", "", notFoundWarning);
				if (compImplClass == null) {
					continue; // Assume that no interfaces are implemented by this class error is reported later.
				}
				
				for(Entry<String, Class<?>> itfsPair : interfaceNameToClass.entrySet()) {
					// Check whether imports given interface 
					if (itfsPair.getValue().isAssignableFrom(compImplClass)) {
						if (result.provItfs2envFieldName.containsKey(itfsPair.getKey())) {
							// Multiple provided interfaces of single type - prohibited in this mode
							infoStream.println("Error - " + itfsPair.getKey() + " is implemented by more component classes. Only single instance for each provided interface type is permitted.");
							return true;
						} else {
							// This class implements current interface
							result.provItfs2envFieldName.put(itfsPair.getKey(), compFieldName);
						}
					}
				}
			}
			
			// Check whether all provided interface are implemented
			if (result.provItfs2envFieldName.size() != config.getComponentProvidedInterfaces().size()) {
				// Not all provided interfaces are implemented
				for(Entry<String, String> provItfsPair : config.getComponentProvidedInterfaces().entrySet()) {
					if (!result.provItfs2envFieldName.containsKey(provItfsPair.getKey())) {
						infoStream.println("Error - " + provItfsPair.getKey() + " interface is not implemented by any component class.");
					}
				}
				return true;
			}
			
		} else if (config.getEnvProvidedItfsAccesMethod() == Configuration.ProvidedInterfaceAccessMethod.SEPARATE_CLASSES) {
			//TODO Enable all possible combinations not only with SOFA - but FRACTAL, SETTERS and FIELDS 
			// Each interface separate field.
			for(String ifaceName : config.getComponentProvidedInterfaces().keySet()) {
				String fieldName = PROPERTY_INTERFACES_PREXIF+ CodeGenerator.patchName(ifaceName);
				result.provItfs2envFieldName.put(ifaceName, PROPERTY_INTERFACES_PREXIF+ CodeGenerator.patchName(fieldName));
			}

			//TODO Fractal check
			// Check if the primary component class implements SOFAServer
			String primaryComponentClassName = config.getComponentImplementationClasses().get(0);
			result.compImpl2envFieldName.put(primaryComponentClassName, PROPERTY_COMPONENT_INSTANCE);
			
			// Check if given component implements all provided interfaces or SOFA2Server for finding provided interfaces.
			Class<?> checkedComponentClass = getClass(primaryComponentClassName, "Component class " + primaryComponentClassName + " not found", "Cannot check provided interfaces validity.", notFoundWarning);
			if (checkedComponentClass == null) {
				return !notFoundWarning;
			}

			Class<?> ifaceSofaServerClass = getClass(SOFA_SERVER_FULL_NAME, "Interface " + SOFA_SERVER_FULL_NAME + " not found.", "Cannot check that interface is implemented by component.", notFoundWarning);
			if (ifaceSofaServerClass == null) {
				return !notFoundWarning;
			}

			// Check whether imports given interface 
			if (!ifaceSofaServerClass.isAssignableFrom(checkedComponentClass)) {
				infoStream.println("Error - Interface " + SOFA_SERVER_FULL_NAME + " is not implemented by component.");
				return true;
			}
		} else {
			throw new RuntimeException("Internal error - Unknown value of enum " + Configuration.ProvidedInterfaceAccessMethod.class.getName() + " value is" + config.getEnvProvidedItfsAccesMethod()); 
		}

		// No error
		return false;
	}
	
	private Class<?> getClass(String className, String errorMessage, String infoMessageSuffix, boolean warningIfClassNotFound) {
		Class<?> result = null;
		try {
			result = Thread.currentThread().getContextClassLoader().loadClass(Type2String.removeGenerics(className)); 
			// checkedComponentClass = Class.forName( Type2String.removeGenerics(config.getComponentImplementationClass()));
		} catch (ClassNotFoundException e) {
			if (warningIfClassNotFound) {
				infoStream.println("Info - " + errorMessage + " "  + infoMessageSuffix);
			} else {
				infoStream.println("Error - " + errorMessage);
			}
			if (DEBUG) { e.printStackTrace(); }
			return null;
		}
		return result;
	}

	/**
	 * Generates starting part of java file - package, imports and class header line.
	 */
	private void genInitialClassCode(LineCouningOutputStream outFile, Indenter indenter, String envClassName, String fullClassName, String pkgComponent) {
		if (pkgComponent != null) {
			pkgComponent = "package " + pkgComponent + ";";
		} else {
			// Default package
			pkgComponent = "// default package;";
		}

		if (config.getEnvGenerate()) {
			indenter.indent(); outFile.println("// EnvGenerator");
			// Print package part
			indenter.indent(); outFile.println(pkgComponent);
			indenter.indent(); outFile.println();

			// Print imports
			indenter.indent(); outFile.println("import gov.nasa.jpf.jvm.Verify;");
			indenter.indent(); outFile.println("import org.ow2.dsrg.fm.tbpjava.envgen.EnvValueSets;");
			
			indenter.indent(); outFile.println();
			indenter.indent(); outFile.println("public class " + envClassName + " {");
			indenter.addLevel();
		}
	}

	private void genProperties(LineCouningOutputStream outFile, Indenter indenter, EnvironmentDescription result) {
		// Generate EnvValueSets properties
		indenter.indent(); outFile.println();
		indenter.indent(); outFile.println("private " + StubGenerator.PROPERTY_OBTAIN_VALUES_TYPE + " " + StubGenerator.PROPERTY_OBTAIN_VALUES + ";");

		indenter.indent(); outFile.println();

		// Generate properties for tested component classes
		for(Entry<String, String> fieldEntry : result.compImpl2envFieldName.entrySet()) {
			indenter.indent(); outFile.println("private " + fieldEntry.getKey() + " " + fieldEntry.getValue() + ";");
		}
		indenter.indent(); outFile.println();
		
		if (config.getEnvProvidedItfsAccesMethod() == Configuration.ProvidedInterfaceAccessMethod.SEPARATE_CLASSES) {
			// Generate properties for each provided interface
			indenter.indent(); outFile.println();
			for(String provIfaceName : config.getComponentProvidedInterfaces().keySet()) {
				String provIfaceType = config.getComponentProvidedInterfaces().get(provIfaceName);
				String provIfacePropertyName = result.provItfs2envFieldName.get(provIfaceName);
				
				indenter.indent(); outFile.println("private " + provIfaceType + " " + provIfacePropertyName + ";");
				// Initialization code is generated in constructor 
			}
		}
	}

	/**
	 * @return Mapping. For provided interface assigns class that implements that interface.
	 * 
	 * Note: outFile and indenter parameters can be null if and only if (config.getEnvGenerate() is false)
	 */ 
	private void genConstructor(LineCouningOutputStream outFile, Indenter indenter, String envClassName, EnvironmentDescription envDesc) {
		if (config.getEnvGenerate()) {
			indenter.indent(); outFile.println();
			indenter.indent(); outFile.println("public " + envClassName + "() {");
			indenter.addLevel();
			
			indenter.indent(); outFile.println("try {");
			indenter.addLevel();
			
			indenter.indent(); outFile.println();
			indenter.indent(); outFile.println(StubGenerator.PROPERTY_OBTAIN_VALUES + "= new " + config.getEnvValueSets() + "();");

			// Create component classes instances
			indenter.indent(); outFile.println();
			for(Entry<String, String> implClassEntry : envDesc.compImpl2envFieldName.entrySet()) {
				indenter.indent(); outFile.println(implClassEntry.getValue() + " = new " + generateComponentConstructorCall(implClassEntry.getKey()) + ";");
			}

			// Set required interfaces
			// Setting required interfaces
			if (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.SETTERS) {
				for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
					//String setterName = StubGenerator.getClassName(config.getComponentRequiredInterfaces().get(reqItfsName));
					String setterName = reqItfsName;
					String envFieldName = envDesc.reqItfs2envFielName.get(reqItfsName);
					indenter.indent(); outFile.println(envFieldName + ".set" + setterName + "( new " + 
						StubGenerator.getGeneratedClassFullName(reqItfsName, config) + Type2String.getGenerics(config.getComponentRequiredInterfaces().get(reqItfsName)) + "(" + StubGenerator.PROPERTY_OBTAIN_VALUES + "));");
				}
			} else if (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.SOFA2_CLIENT) {
				for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
//					String interfaceName = StubGenerator.getClassName(config.getComponentRequiredInterfaces().get(reqItfsName));
					String interfaceName = reqItfsName;
					String envFieldName = envDesc.reqItfs2envFielName.get(reqItfsName);
					indenter.indent(); outFile.println(envFieldName + ".setRequired(" + interfaceName + "( new " + 
							StubGenerator.getGeneratedClassFullName(reqItfsName, config) + Type2String.getGenerics(config.getComponentRequiredInterfaces().get(reqItfsName)) + "(" + StubGenerator.PROPERTY_OBTAIN_VALUES + "));");
				}
			} else if (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.FRACTAL) {
				for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
//					String interfaceName = StubGenerator.getClassName(config.getComponentRequiredInterfaces().get(reqItfsName));
					String interfaceName = reqItfsName;
					String envFieldName = envDesc.reqItfs2envFielName.get(reqItfsName);
					indenter.indent(); outFile.println(envFieldName + ".bindFc(\"" + interfaceName + "\", new " + 
							StubGenerator.getGeneratedClassFullName(reqItfsName, config) + Type2String.getGenerics(config.getComponentRequiredInterfaces().get(reqItfsName)) + "(" + StubGenerator.PROPERTY_OBTAIN_VALUES + "));");
				}
			} else if (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.FIELDS) {
				for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
//					String interfaceName = StubGenerator.getClassName(config.getComponentRequiredInterfaces().get(reqItfsName));
					String interfaceName = reqItfsName;
					String envFieldName = envDesc.reqItfs2envFielName.get(reqItfsName);
					indenter.indent(); outFile.println(envFieldName + "." + interfaceName + " = new " + 
							StubGenerator.getGeneratedClassFullName(reqItfsName, config) + Type2String.getGenerics(config.getComponentRequiredInterfaces().get(reqItfsName)) + "(" + StubGenerator.PROPERTY_OBTAIN_VALUES + ");");
				}
			} else {
				throw new RuntimeException("Internal error - Unknown value of enum " + Configuration.RequiredInterfaceSettingMethod.class.getName() + " value is" + config.getEnvRequierdItfcSettingMethod()); 
			}

			indenter.indent(); outFile.println();
		}
		
		if (config.getEnvProvidedItfsAccesMethod() == Configuration.ProvidedInterfaceAccessMethod.SEPARATE_CLASSES) {
			// Generate properties for each provided interface
			for(Entry<String, String> provIfaceEntry : envDesc.provItfs2envFieldName.entrySet()) {
				if (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.SOFA2_CLIENT) {
					String primaryComponentClassName = config.getComponentImplementationClasses().get(0);
					String primaryComponentFieldName = envDesc.compImpl2envFieldName.get(primaryComponentClassName);
					assert primaryComponentFieldName != null;
					
					indenter.indent(); outFile.println(provIfaceEntry.getValue() + " = " + primaryComponentFieldName + ".getProvided(\"" + provIfaceEntry.getKey() + "\";");
					
				} else if ( (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.SETTERS)
				         || (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.FRACTAL)
				         || (config.getEnvRequierdItfcSettingMethod() == Configuration.RequiredInterfaceSettingMethod.FIELDS)) {
					throw new RuntimeException("Internal error - Unsupported configuration combination. " + Configuration.ProvidedInterfaceAccessMethod.SEPARATE_CLASSES.toString() + " can be used onyl with " + Configuration.RequiredInterfaceSettingMethod.SOFA2_CLIENT); // 
				} else {
					throw new RuntimeException("Internal error - Unknown value of enum " + Configuration.RequiredInterfaceSettingMethod.class.getName() + " value is" + config.getEnvRequierdItfcSettingMethod()); 
				}
			}
		} else if (config.getEnvProvidedItfsAccesMethod() == Configuration.ProvidedInterfaceAccessMethod.COMPONENT_CLASSES) {
			// Nothing to do
		} else {
			throw new RuntimeException("Internal error - Unknown value of enum " + Configuration.ProvidedInterfaceAccessMethod.class.getName() + " value is" + config.getEnvProvidedItfsAccesMethod());
		}

		// Generate end of constructor
		indenter.removeLevel();
		indenter.indent(); outFile.println("} catch (Throwable e) {");
		indenter.addLevel();
		indenter.indent(); outFile.println("throw new RuntimeException(e);");
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
		// End of constructor body
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
	}

	/**
	 * Gets string with call of the class constructor.
	 *   At first it tries to use non-parametric constructor,
	 *   then if use first public constructor it found.
	 * Parameters for the constructor are generated from {@link EnvValueSets}
	 * 
	 * @param className Name of the class to instantiate.
	 * @return String that can be passed into 
	 */
	private String generateComponentConstructorCall(String className) {
		Class<?> compImplClass = getClass(className, "Component class " + className + " not found. Cannot generate constructor call", "", true);
		if (compImplClass == null) {
			// Assumes non-parametric constructor
				return className+ "(); // Error - EnvGenerator cannot found " + className + " class. Assuming non-parametric constructor exists.";

			//Prints error when executed
				//return "null; throw new RuntimeException(\"Error - EnvGenerator cannot found " + className + " class. Cannot generate proper constructor call.\")";
		}
		
		// Check for non-parametric constructor at first
		try {
			Constructor<?> constructor = compImplClass.getConstructor();
			
			int constructorModifier = constructor.getModifiers();
			if (Modifier.isPublic(constructorModifier)) {
				return className + "()";
			}
			
		} catch (SecurityException e) {
			infoStream.println("Error -  Class " + className + " cannot be accessed (Security manager blocks access).");
			// Assumes non-parametric constructor
				return className+ "(); // Error - EnvGenerator cannot get constructors of " + className + " class. Assuming non-parametric constructor exists.";
			//Prints error when executed
				//return "null; throw new RuntimeException(\"Error - EnvGenerator cannot get constructors of " + className + " class. Cannot generate proper constructor call.\")";
		} catch (NoSuchMethodException e) {
			// No non-parametric constructor ... try others
		}
		
		// Try to found any public constructor
		try {
			Constructor<?>[] constructors = compImplClass.getConstructors();
			
			// Sort the constructors according to number of parameters and then lexicographically
			Arrays.sort(constructors, 0, constructors.length, new Comparator<Constructor<?>>() {

				@Override
				public int compare(Constructor<?> left, Constructor<?> right) {
					int leftParams = left.getParameterTypes().length;
					int rightParams = right.getParameterTypes().length;
					
					int result = rightParams - leftParams;
					if (result == 0) {
						// Same number of parameter
						String leftStr = left.toGenericString();
						String rightStr = right.toGenericString();
						
						result = leftStr.compareTo(rightStr);
					}
					return result;
				}
			});
			
			for(Constructor<?> constructor : constructors) {
				int constructorModifier = constructor.getModifiers();
				if (Modifier.isPublic(constructorModifier)) {
					// Generate method call code parameters
					StringBuffer result = new StringBuffer();
					result.append(className);
					result.append('(');
					Type[] params = constructor.getGenericParameterTypes();
					for(int i = 0; i < params.length; i++) {
						// Reusing code from stub generator for obtaining values from envSets
						result.append( StubGenerator.genCodeObtainingValue(params[i], config.getComponentName(), className, EnvValueSets.COMPONENT_CONSTRUCTOR_PARAMETER));
						if ( i <  params.length-1 ) {
							// Not last parameter
							result.append(", ");
						};
					}
					result.append(')');
					return result.toString();
				}
			}
		} catch (SecurityException e) {
			infoStream.println("Error -  Class " + className + " cannot be accessed (Security manager blocks access).");
			// Non parametric constructor is not suitable 
				// return className+ "(); // Error - EnvGenerator cannot get constructors of " + className + " class. Assuming non-parametric constructor exists.";
			// Prints error when executed
				return "null; throw new RuntimeException(\"Error - EnvGenerator cannot get constructors of " + className + " class. Cannot generate proper constructor call.\")";
		}
		
		return "null; throw new RuntimeException(\"Error - no suitable (public) constructor in " + className + " class found. Cannot generate proper constructor call.\")";
	}

	private void genProtocolCode(LineCouningOutputStream outFile, Indenter indenter, CodeGenerator provisionParser, TBPParsedProvisionContainerNode node, EnvironmentDescription envDesc) {
		Indenter.IndentLevel originalLevel = indenter.getIndentLevel();

		indenter.indent(); outFile.println();
		indenter.indent(); outFile.println("public void " + METHOD_GENERATED_PROTOCOL_NAME + "() {");
		indenter.addLevel();
		indenter.indent(); outFile.println("try {");
			provisionParser.generateCode(node);
		indenter.restoreIndentLevel(originalLevel);
		indenter.addLevel();
		indenter.indent(); outFile.println("} catch (Throwable e) {");
		indenter.addLevel();
		// Check way how to handle exceptions
		if(EXCEPTIONS_STOP_ON_EXCEPTION) {
			indenter.indent(); outFile.println("throw new RuntimeException(e);");
		} else {
			indenter.indent(); outFile.println("System.out.println(\"Error - Unhandled Exception occured\");");
			indenter.indent(); outFile.println("System.out.println(e);");
		}
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
		indenter.restoreIndentLevel(originalLevel);
		indenter.indent(); outFile.println("}");
	}

	private void genMain(LineCouningOutputStream outFile, Indenter indenter, String envClassName) {
		indenter.indent(); outFile.println();
		indenter.indent(); outFile.println("public static void main(String[] args) {");
		indenter.addLevel();
		indenter.indent(); outFile.println(envClassName + " " + METHOD_MAIN_INSTANCE_NAME + " = new " + envClassName + "();");
		indenter.indent(); outFile.println(METHOD_MAIN_INSTANCE_NAME + "." + METHOD_GENERATED_PROTOCOL_NAME + "();");
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
			
	}
	private void genEndingClassCode(LineCouningOutputStream outFile, Indenter indenter) {
		indenter.indent(); outFile.println();
		indenter.removeLevel();
		indenter.indent(); outFile.println("}");
		indenter.indent(); outFile.println();
	}

	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		
		if (args.length < 1) {
			System.out.println("Invalid argments number");
			System.out.println("envGenerator usage");
			System.out.println("  envGenerator configurationFile Optional-ParametersOverwriteingConfigurationFileDefaules");
			return;
		}
		
		String configuratioFile = args[0];
		String[] overwritingsParameters = new String[args.length-1];
		// Removing first parameter ... Coping args[1 .. end] into overwritingsParameters[0 ++]
		for(int i = 1; i < args.length; i++) {
			overwritingsParameters[i-1] = args[i];
		}
		
		Configuration config = new Configuration(overwritingsParameters, configuratioFile, System.out);
		if (config.isConfigurationError()) {
			System.out.println("Configuration contais errors.");
			return;
		}
		
		
		// Generation phase

		/// Environment classes for each provisions
		EnvGenerator generator = new EnvGenerator(config, System.out);
		generator.generateEnvironments();
		
		// Stub classes for required interface
		for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
			StubGenerator sb = new StubGenerator(config, reqItfsName, System.out);
			sb.generateStub();
		}
	}

}
